<?php
/**
 * Model base class
 * Use '_' as prefix to avoid duplicate name on inheritance class.
 * @package base
 * @since 1.0
 * @author JJB <splitedragon@hotmail.com>
 * @copyright Copyright (c) 2019, EVENING
 */
class Model
{
    protected $_db;
    protected $_sql;
    protected $_data;
    protected $_handler;
    protected $_trans_flag;

    public function __construct()
    {
        $this->_db = DB::init();
        $this->_handler = NULL;
        $this->_sql = '';
        $this->_data = NULL;
        $this->_trans_flag = false;
    }

    protected function _setSql($sql)
    {
        $this->_sql = $sql;
    }

    protected function _setData($data)
    {
        $this->_data = $data;
    }

    protected function _getData()
    {
        return $this->_data;
    }

    public function _getHandler()
    {
        return $this->_handler;
    }

    public function _transSwitch($flag=false)
    {
        $this->_trans_flag = $flag;
        return $this;
    }

    /**
     * @param $sql
     * @param null|array $data
     * @return $this
     */
    public function _set($sql, $data=NULL)
    {
        $this->_setSql($sql);
        $this->_setData($data);
        return $this;
    }

    /**
     * Get prepared SQL.
     *
     * @return string
     */
    public function _dumpQuery()
    {
        $query = $this->_sql;
        $param = $this->_getData();
        return Db::dumpQuery($query, $param?$param:[]);
    }

    /**
     * Use to get last inserted id. If unset, default is 'id' as primary key
     * !Important
     * On transaction, must call lastInsertId() before commit().
     * @param $name null|field name
     * @return int|mixed
     */
    public function _lastInserted($name=NULL)
    {
        return $this->_db->lastInsertId($name);
    }

    /**
     * Use to get of row count for current query.
     * !Important: returns the number of rows affected by the last DELETE, INSERT, or UPDATE statement executed
     * @return int
     */
    public function _rowCount()
    {
        $handler = $this->_getHandler();
        if($handler) {
            return $handler->rowCount();
        }
        return 0;
    }

    /**
     * Prepare and execute sql.
     * @param null $data
     * @return bool|null
     */
    public function _run()
    {
        $this->_handler = NULL;
        $_sql = $this->_sql;
        $_data = $this->_getData();
        $_trans_flag = $this->_trans_flag;

        try
        {
            if($_trans_flag === true) {
                $this->_transBegin();
            }
            $this->_handler = $this->_db->prepare($_sql);

            if(is_array($_data) && count($_data) > 0) {
                $result = $this->_handler->execute($_data);
            } else {
                $result = $this->_handler->execute();
            }

            if($_trans_flag === true) {
                $this->_transCommit();
            }

            return $result;
        }
        catch (Exception $e) {
            //throw new Exception($e->getMessage());
            if($_trans_flag === true) {
                $this->_transBack();
            }

            if(gIsCLI()) {
                $message = $e->getMessage().PHP_EOL.':::SQL:::'.PHP_EOL.$this->_dumpQuery();
            } else{
                $message = $e->getMessage().'<br>:::SQL:::<br>'.$this->_dumpQuery();
            }

            throw new Exception($message);
        }
        catch (Throwable $t) {
            if($_trans_flag === true) {
                $this->_transBack();
            }
            if(gIsCLI()) {
                $message = $t->getMessage().PHP_EOL.':::SQL:::'.PHP_EOL.$this->_dumpQuery();
            } else{
                $message = $t->getMessage().'<br>:::SQL:::<br>'.$this->_dumpQuery();
            }

            throw new Exception($message);
        }
    }

    /**
     * @param $data Array
     * @return array Returns an array containing all of the result set rows.
     */
    public function _getAll()
    {
        $result = $this->_run();
        $handler = $this->_getHandler();
        if($handler && $result===true) {
            return $handler->fetchAll();
        }
        return NULL;
    }

    /**
     * @param $data Array
     * @return array Fetches the next row from a result set.
     */
    public function _getRow()
    {
        $result = $this->_run();
        $handler = $this->_getHandler();
        if($handler && $result===true) {
            return $handler->fetch();
        }
        return NULL;
    }

    // Notice:
    // In fact, insert(), update(), remove() is equals with _run().
    // i.e. insert($data) === _run($data)

    /**
     * @param $data Array data for insert.
     * @return bool|null false: query fails, null: sql error, true: success insert.
     */
    public function _insert()
    {
        $result = $this->_run();
        $handler = $this->_getHandler();
        if($handler && $result===true) {
            return true;
        }
        return false;
    }

    /**
     * @param $data Array data for update.
     * @return bool|null false: query fails, null: sql error, true: success update.
     */
    public function _update()
    {
        $result = $this->_run();
        $handler = $this->_getHandler();
        if($handler && $result===true) {
            return true;
        }
        return false;
    }

    /**
     * @param $data Array data for delete.
     * @return bool|null false: query fails, null: sql error, true: success remove.
     */
    public function _remove()
    {
        $result = $this->_run();
        $handler = $this->_getHandler();
        if($handler && $result===true) {
            return true;
        }
        return false;
    }

    public function _transBegin()
    {
        if (!Db::$transIndex++) {
            return $this->_db->beginTransaction();
        }
        $this->_db->exec('SAVEPOINT trans'.Db::$transIndex);
        return Db::$transIndex >= 0;
    }

    public function _transCommit()
    {
        if (!--Db::$transIndex) {
            return $this->_db->commit();
        }
        return Db::$transIndex >= 0;
    }

    public function _transBack()
    {
        if (--Db::$transIndex) {
            $this->_db->exec('ROLLBACK TO trans'.(Db::$transIndex + 1));
            return true;
        }
        return $this->_db->rollBack();
    }
}