<?php
/**
 * About routing
 * The goals of routing is how to find the controller and action.
 * Assume the request url as following:
 * http://localhost/a/b
 *
 * 1. Generally,
 * controller path: /controllers/
 * controller name: 'AController' at AController.php
 * action name: 'b()'
 *
 * 2. If defined 'a' in gModules().
 * controller path: /controllers/a/
 * controller name: 'BController' at BController.php
 * action name: '' but auto-match to 'index()' // index() is default action.
 */

/**
 * Router base class
 * @package base
 * @since 1.0
 * @author JJB <splitedragon@hotmail.com>
 * @copyright Copyright (c) 2019, EVENING
 */
class Router
{
    private static $_url = array();
    private static $config = null;

    public static function get($key=NULL)
    {
        return ($key == NULL)? self::$_url : self::$_url[$key];
    }

    public static function set($key, $val)
    {
        self::$_url[$key] = $val;
    }

    public static function initConfig()
    {
        if(!is_array(self::$config)) {
            self::$config = Config::routes();
        }
    }

    public static function getRequestURL()
    {
        // Added by jjb 2020.04.24. For CLI version.
        if(gIsCLI() && !isset($_SERVER["REQUEST_URI"])) {
            $args = $_SERVER['argv'];//array_slice($_SERVER['argv'], 1);
            $_SERVER["REQUEST_URI"] = $args ? implode('/', $args) : '';
        }
        // End adding

        $request = strtok($_SERVER["REQUEST_URI"],'?');
        $request = self::clearURL($request); // Always starting from '/'
        $request = substr($request, strlen(BASE_URL));
        $request = preg_replace('/^\\/index.php\\//', '', $request.'/', 1); // Modified by jjb
        $request = self::clearURL($request); // Always start from '/' to end with no '/'
        $request = Filter::decode_url($request);

        return $request;
    }
    /**
     * Analyze the requested URL.
     *
     * @throws Exception
     * @return array Analyzed request url array.
     */
    public static function prepare()
    {
        self::initConfig();
        $path = '';
        $definedModule = null;
        $definedRoute = null;
        $controller = null;
        $action = null;
        $model = null;
        $argument = null;
        $callback = null;
        $default_router = self::$config['default'];

        $request = self::getRequestURL(); // Always starting from '/'

        $definedRoute = self::inRoutes($request);

        if(!empty($definedRoute) && is_array($definedRoute)) {
            $key = $definedRoute['key'];
            $rule = $definedRoute['rule'];

            if(!is_array($rule) && is_callable($rule)) {
                $callback = $rule;
                $m = $request==$key? '' : substr($request, strlen($key));
                $arr = explode('/', $m);
                $argument = self::removeEmptyItem($arr);
                goto next;
            }

            if($key == '*') {
                $controller = isset($rule[0])? $rule[0] : null;
                $action = isset($rule[1])? $rule[1] : null;
                $argument = null;
            } else {
                $m = $request==$key? '' : substr($request, strlen($key));
                $arr = explode('/', $m);
                $arr = self::removeEmptyItem($arr);

                $controller =   isset($rule[0])? $rule[0] : null;
                if($controller != null) {
                    if(isset($rule[1])) {
                        $action = $rule[1];
                        $argument = empty($arr)? null : $arr;
                    } else {
                        $action = isset($arr[0])? $arr[0] : null;
                        @array_shift($arr);
                        $argument = empty($arr)? null : $arr;
                    }
                }
            }
            $path = isset($rule[2])? $rule[2] : '';
        }
        else {
            $b = explode('/', $request);
            // Check modules!
            $definedModule = self::inModules($request);

            if ($definedModule != null) {
                $m = substr($request, strlen($definedModule));
                $b = explode('/', self::clearURL($m));
            }
            $b = self::removeEmptyItem($b);
            $size = count($b);

            if($size>0) { // Request is now including at least a controller.
                // Detect arguments.
                if($size > 2) { // At least one or more arguments were included in the request.
                    $argument = array_slice($b, 2);
                }

                $controller = isset($b[0]) ? $b[0] : null;
                $action = isset($b[1]) ? $b[1] : null;
            }

            if($definedModule != null) {
                $path = $definedModule.$path;
            }
        }

        $controller = $controller==null? $default_router['controller'] : $controller;
        $action = $action==null? $default_router['action'] : $action;

        if (!preg_match("/^[a-z0-9_]+$/i", $controller)) {
            throw new Exception('Invalid controller: '.$controller);
        }
        if (!preg_match("/^[a-z0-9_]+$/i", $action)) {
            throw new Exception('Invalid action: '.$action);
        }

        $model = ucwords($controller);
        $controller = ucwords($controller);

        next:

        $url = array();
        $url['request'] = $request;
        $url['defined_module'] = $definedModule;
        $url['defined_route'] = $definedRoute;
        $url['path'] = $path;
        $url['controller'] = $controller;
        //$url['model'] = $model;
        $url['action'] = $action;
        $url['argument'] = $argument;
        $url['callback'] = $callback;

        // Modified by jjb 2020.04.24. For CLI version.
        if(gIsCLI()) {
            $url['http'] = $_SERVER['REQUEST_URI'];
        } else {
            $url['http'] = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://" . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
        }
        // End modifying


        self::set('http', $url['http']);
        self::set('request', $url['request']);
        self::set('defined_module', $url['defined_module']);
        self::set('defined_route', $url['defined_route']);
        self::set('path', $url['path']);
        self::set('controller', $url['controller']);
        //self::set('model', $url['model']);
        self::set('action', $url['action']);
        self::set('argument', $url['argument']);
        self::set('callback', $url['callback']);
        //self::set('GET', $_GET);
        //self::set('POST', $_POST);
        //var_dump(self::get());
        return self::get();
    }

    public static function redirect($url)
    {
        @header("Location: ".Filter::toStr($url));
        exit;
    }

    public static function reload()
    {
        //header("Location: ".$_SERVER['PHP_SELF']);
        @header("Refresh: 0");
        exit;
    }

    public static function backward()
    {
        echo "<script>window.history.back();</script>";
    }

    public static function forward()
    {
        echo "<script>window.history.forward();</script>";
    }

    public static function requestMethod()
    {
        $method = isset($_SERVER['REQUEST_METHOD'])? strtoupper($_SERVER['REQUEST_METHOD']) : 'CLI'; // GET, POST, PUT, DELETE
        return $method;
    }

    public static function controller($val)
    {
        $path = trim($val, " \t\n\r\0\x0B\\/");
        $path = str_replace(array('\\', '/'), DS, $path);
        $a = explode(DS, $path);
        $size = count($a);
        if(!isset($a[$size-1]) || empty($a[$size-1])) {
            return null;
        }

        $controllerName = ucwords($a[$size-1]).'Controller';
        array_splice($a, $size-1);
        $a = self::removeEmptyItem($a);
        $path = implode(DS, $a);

        //$fileName = strtolower($controllerName).'.php';
        $fileName = $controllerName.'.php'; // For case-sensitive of UNIX

        $file = gURL('controllers').DS.$path;
        $file .= $path!=''? DS.$fileName : $fileName;
        return array('filePath'=>$file, 'className'=>$controllerName);
    }

    public static function model($val)
    {
        $path = trim($val, " \t\n\r\0\x0B\\/");
        $path = str_replace(array('\\', '/'), DS, $path);
        $a = explode(DS, $path);
        $size = count($a);
        if(!isset($a[$size-1]) || empty($a[$size-1])) {
            return null;
        }

        $modelName = ucwords($a[$size-1]).'Model';
        array_splice($a, $size-1);
        $a = self::removeEmptyItem($a);
        $path = implode(DS, $a);

        //$fileName = strtolower($modelName).'.php';
        $fileName = $modelName.'.php'; // For case-sensitive of UNIX

        $file = gURL('models').DS.$path;
        $file .= $path!=''? DS.$fileName : $fileName;
        return array('filePath'=>$file, 'className'=>$modelName);
    }

    public static function view($val)
    {
        $path = trim($val, " \t\n\r\0\x0B\\/");
        $path = str_replace(array('\\', '/'), DS, $path);
        $a = explode(DS, $path);
        $size = count($a);
        if(!isset($a[$size-1]) || empty($a[$size-1])) {
            return null;
        }

        //$viewName = ucwords($a[$size-1]);
        $viewName = $a[$size-1];
        array_splice($a, $size-1);
        $a = self::removeEmptyItem($a);
        $path = implode(DS, $a);

        //$fileName = strtolower($viewName).'.php';
        $fileName = $viewName.'.php'; // For case-sensitive of UNIX

        $file = gURL('views').DS.$path;
        $file .= $path!=''? DS.$fileName : $fileName;

        return array('filePath'=>$file, 'className'=>$viewName);
    }

    protected static function inRoutes($val)
    {
        $method = isset($_SERVER['REQUEST_METHOD'])? strtoupper($_SERVER['REQUEST_METHOD']) : 'CLI'; // GET, POST, PUT, DELETE

        $default = self::$config['default'];

        $result = null;

        /*if(!is_array(self::$_routes)) {
            self::$_routes = gRoutes();
        }*/
        $_routes = self::$config['routes'];

        //$routes = isset($_routes[$method])? $_routes[$method] : array(); // 만일 GET, POST 등을 구분하여 route.config.php작성하였다면 이코드 적용
        $routes = $_routes;  // 만일 GET, POST 등을 구분없이 route.config.php작성하였다면 이코드 적용

        $key = NULL;
        foreach ($routes as $k=>$item) {
            $b = self::clearURL($k);
            $pos = strpos($val.'/', $b.'/');

            if($pos === 0) {
                $result = $item;
                $key = $b;
                break;
            }
        }
        if($key==NULL || $result==NULL ) {
            if (array_key_exists('*', $routes)) {
                $result = $routes['*'];
                $key = '*';
            } else {
                return NULL;
            }
        }

        return array('key'=>$key, 'rule'=>$result);
    }

    protected static function inModules($val)
    {
        $module = NULL;
        /*if(!is_array(self::$_modules)) {
            self::$_modules = gModules();
        }*/

        $modules = self::$config['modules'];

        $a = self::clearURL($val);

        for($i=0; $i<count($modules); $i++){
            $b = self::clearURL($modules[$i]);
            if(strpos($a.'/', $b.'/') === 0) {
                $module = $b;
                break;
            }
        }

        return $module;
    }

    protected static function removeEmptyItem($arr)
    {
        return array_values(array_filter($arr, function($item) {
            return $item!='';
        }));
    }

    /**
     * Fix wrong slash in path
     * /a/////b//////c/ changed to /a/b/c
     *
     * @param $val URL string
     * @return string
     */
    protected static function clearURL($val)
    {
        /*$a = explode('/', $val);
        $a = self::removeEmptyItem($a);
        $result = implode('/', $a);
        return $result;*/

        $result = trim($val, " \t\n\r\0\x0B/");
        $result = preg_replace('/\\/\\/+/', '/', $result);
        return '/'.$result;
    }
}