/**
 * Copyright (c) 2012 Jacob Lee <letsgolee@naver.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA or check at http://www.gnu.org/licenses/gpl.html
 */

/**
 * Advanced Encryption Standard (AES)
 *
 * http://en.wikipedia.org/wiki/Advanced_Encryption_Standard
 *
 * Example 1)
 *
 * var aes = new AES();
 * var plaintext = 'This is the text';
 * var ciphertext = aes.encrypt(plaintext, 'password', {bits: 256, // default value
 *													  utf8: true // default value
 * });
 * if (ciphertext === false) {
 *	 alert(aes.messages[aes.messages.length-1]);
 * }
 * else {
 *	  var text = aes.decrypt(ciphertext, 'password');
 *	  if (text === false) alert(cbc.messages[aes.messages.length-1]);
 *	  else {
 *			if (text == plaintext) alert('OK');
 *			else alert('Something Wrong in the code!);
 *	   }
 * }
 *
 * Example 2)
 * 
 * var aes = new AES();
 * var plaintext = 'This is the text!';
 * aes.setKeySize(256);
 * var ciphertext = aes.encrypt(plaintext, 'password');
 *
 * var text = aes.decrypt(ciphertext, 'password');
 */
function AES() {
	/**
	 * AES Mode: Cipher-block chaining (CBC)
	 */
	this.Mode = 'CBC';

	/**
	 * Messages that are appended while encrypting & decrypting
	 */
	this.Messages = [];

	/* Default to 256 Bit Encryption */
	/* refer setKeySize() */
	this.Nr = 14; // number of round
	this.Nk = 8;  // key size
};

/*
AES.generateXTIME = function() {
	var ret = new Array(256);

	for(var i = 0; i < 128; i++) {
		ret[i] = i << 1;
		ret[128 + i] = (i << 1) ^ 0x1b;
	}

	return ret;
};
*/
AES.xtime = function(x) {
	return x < 0x80 ? x << 1 : ((x - 0x80) << 1) ^ 0x1b;
};

AES.invertArray = function(array) {
	var ret = [];
	for (var i = 0; i < array.length; i++)
		ret[array[i]] = i;
	return ret;
};

AES.randomBytesArray = function(len, start, end) {
	if(!len) len = 8;
	if(start == null) start = 0;
	if(end == null) end = 255;
	var bytes = new Array(len);
	var field = [];
	for(var i = start; i <= end; i++) field[i] = i;
	for(i = 0; i < len; i++)
		bytes[i] = field[Math.floor(Math.random()*field.length)];
	return bytes;
};

AES.generateSalt = function() {
	var bytes = AES.randomBytesArray(8);
	return AES.ByteArray.toString(bytes);
	//return AES.ByteArray.toString(new PRNG().getBytes(new Array(8)));
};


AES.Hex = {
	/**
	 * Converts a string into a hexadecimal string
	 * returns the characters of a string to their hexadecimal charcode equivalent
	 * Works only on byte chars with charcode < 256. All others chars are converted
	 * into "xx"
	 *
	 * @return hex string e.g. "hello world" => "68656c6c6f20776f726c64"
	 */
	fromString: function(str) {
		if(!str) str = "";
		var hs ='';
		var hv ='';
		for (var i=0, cnt=str.length; i < cnt; i++) {
			hv = str.charCodeAt(i).toString(16);
			hs += (hv.length == 1) ? '0'+hv : hv;
		}
		return hs;
	},

	/**
	 * Converts a hex string into a string
	 * returns the characters of a hex string to their char of charcode
	 *
	 * @return hex string e.g. "68656c6c6f20776f726c64" => "hello world"
	 */
	toString: function(str) {
		if(!str) str = "";
		var s = "";
		for(var i=0, cnt=str.length; i < cnt; i+=2){
			s += String.fromCharCode(parseInt(str.substring(i,i+2),16));
		}
		return s;
	}
};


AES.ByteArray = {
	/**
	 * Converts a string into an array of char code bytes
	 * returns the characters of a hex string to their char of charcode
	 *
	 * @return hex string e.g. "68656c6c6f20776f726c64" => "hello world"
	 */
	fromString: function(s) {
		if(!s) s = "";
		var b = [];
		for(var i=0, cnt=s.length; i < cnt; i++){
			 b[i] = s.charCodeAt(i);
		}
		return b;
	},

	toString: function(b) {
		var s = '';
		for(var i=0, cnt=b.length; i < cnt; i++){
			 s += String.fromCharCode(b[i]);
		}
		return s;
	}
};

/*
AES.SBOX = [
	99,  124, 119, 123, 242, 107, 111, 197, 48,  1,   103, 43,  254, 215, 171,
	118, 202, 130, 201, 125, 250, 89,  71,  240, 173, 212, 162, 175, 156, 164,
	114, 192, 183, 253, 147, 38,  54,  63,  247, 204, 52,  165, 229, 241, 113,
	216, 49,  21,  4,   199, 35,  195, 24,  150, 5,   154, 7,   18,  128, 226,
	235, 39,  178, 117, 9,   131, 44,  26,  27,  110, 90,  160, 82,  59,  214,
	179, 41,  227, 47,  132, 83,  209, 0,   237, 32,  252, 177, 91,  106, 203,
	190, 57,  74,  76,  88,  207, 208, 239, 170, 251, 67,  77,  51,  133, 69,
	249, 2,   127, 80,  60,  159, 168, 81,  163, 64,  143, 146, 157, 56,  245,
	188, 182, 218, 33,  16,  255, 243, 210, 205, 12,  19,  236, 95,  151, 68,
	23,  196, 167, 126, 61,  100, 93,  25,  115, 96,  129, 79,  220, 34,  42,
	144, 136, 70,  238, 184, 20,  222, 94,  11,  219, 224, 50,  58,  10,  73,
	6,   36,  92,  194, 211, 172, 98,  145, 149, 228, 121, 231, 200, 55,  109,
	141, 213, 78,  169, 108, 86,  244, 234, 101, 122, 174, 8,   186, 120, 37,
	46,  28,  166, 180, 198, 232, 221, 116, 31,  75,  189, 139, 138, 112, 62,
	181, 102, 72,  3,   246, 14,  97,  53,  87,  185, 134, 193, 29,  158, 225,
	248, 152, 17,  105, 217, 142, 148, 155, 30,  135, 233, 206, 85,  40,  223,
	140, 161, 137, 13,  191, 230, 66,  104, 65,  153, 45,  15,  176, 84,  187,
	22
];
*/
AES.SBOX = [
	0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5,
	0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76,
	0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0,
	0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0,
	0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc,
	0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15,
	0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a,
	0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75,
	0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0,
	0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84,
	0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b,
	0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf,
	0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85,
	0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8,
	0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5,
	0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2,
	0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17,
	0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73,
	0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88,
	0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb,
	0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c,
	0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79,
	0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9,
	0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08,
	0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6,
	0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a,
	0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e,
	0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e,
	0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94,
	0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf,
	0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68,
	0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16
];

AES.SBOX_INVERSE = AES.invertArray(AES.SBOX);

//AES.SHIFTBY = [0,5,10,15,4,9,14,3,8,13,2,7,12,1,6,11];
AES.SHIFTBY = [0x0,0x5,0xa,0xf,0x4,0x9,0xe,0x3,0x8,0xd,0x2,0x7,0xc,0x1,0x6,0xb];

AES.SHIFTBY_INVERSE = AES.invertArray(AES.SHIFTBY);

//AES.XTIME = AES.generateXTIME();

AES.BLOCK_SIZE = 16;

AES.prototype.randomAESKey = function(len)
{
	if (!len) len = 8;
	// range: 32-126 refer to ASCII code table
	var bytes = AES.randomBytesArray(len, 32, 126);
	return AES.ByteArray.toString(bytes);
};


AES.prototype.setKeySize = function(size) {
	if (typeof size == 'undefined' || size == null) size = 256;

	switch (size) {
		case 128:
			this.Nr = 10;
			this.Nk = 4;
			break;
		case 192:
			this.Nr = 12;
			this.Nk = 6;
			break;
		case 256:
			this.Nr = 14;
			this.Nk = 8;
			break;
		default:
			this.appendMsg('AES.setKeySize encounted an error! Invalid Key Size Specified:' + size);
			return false;
	}
	return true;
};

AES.prototype.appendMsg = function(msg) {
	this.Messages.push(msg);
};

AES.prototype.displayMsg = function(index) {
	if (typeof index === 'undefined') index = this.Messages.length-1;
	return (this.Messages.length > 0 && index < this.Messages.length) ? this.Messages[index] : '';
};


AES.prototype.encrypt = function(plaintext, password, options) {
	if (typeof options === 'undefined') options = {};
	var bits = (typeof options.bits !== 'undefined') ? options.bits : 256;
	var utf8 = (typeof options.utf8 !== 'undefined') ? options.utf8 : true;

	if (plaintext == null || typeof plaintext != 'string' || !plaintext.length) {
		this.appendMsg('The plaintext is blank!');
		return '';
	}

	var salt = AES.generateSalt();

	if (!this.setKeySize(bits)) return false;

	// Creates Password Based Encryption KEY and IV(initial vector)
	var pbe = this._createPBEKey(password, salt);
	if (!pbe) return false;

	if (utf8)
		plaintext = UTF8.encode(plaintext);
	
	var ciphertext = this.rawEncrypt(plaintext, pbe.key, pbe.iv);

	return (ciphertext === false) ? false : BASE64.encode('Salted__' + salt + ciphertext);
};


AES.prototype.rawEncrypt = function(plaintext, key, iv) {
	var byteArray = this._PKCS5pad(AES.ByteArray.fromString(plaintext));

	var keySchedule = this._createKeySchedule(key);

	var nBlocks = Math.ceil(byteArray.length / AES.BLOCK_SIZE);

	var cipherText = '';
	var plainBlock = [];
	var state = AES.ByteArray.fromString(iv);

	for (var b = 0; b < nBlocks; b++) {
		// XOR last block and next data block, then encrypt that
		plainBlock = byteArray.slice(b * AES.BLOCK_SIZE, b * AES.BLOCK_SIZE + AES.BLOCK_SIZE);
		state = this._xorBlock(state, plainBlock);
		state = this._encryptBlock(state.slice(), keySchedule); // encrypt block
		cipherText += AES.ByteArray.toString(state);
	}

	return cipherText;
};



AES.prototype.decrypt = function(ciphertext, password, options) {
	if (typeof options === 'undefined') options = {};
	var bits = (typeof options.bits !== 'undefined') ? options.bits : 256;
	var utf8 = (typeof options.utf8 !== 'undefined') ? options.utf8 : true;

	if (ciphertext == null || typeof ciphertext != 'string' || !ciphertext.length) {
		this.appendMsg('The ciphertext is blank!');
		return '';
	}

	ciphertext = BASE64.decode(ciphertext);

	if (ciphertext.substr(0,8) != 'Salted__') {
		this.appendMsg('The given text is not a AES ciphertext!');
		return false;
	}

	var salt = ciphertext.substr(8,8); // extract salt from crypted text

	if (!this.setKeySize(bits)) return false;

	var pbe = this._createPBEKey(password, salt);
	if (!pbe) return false;

	var plaintext = this.rawDecrypt(ciphertext.substr(16), // remove salt: Salted__(8 bytes) + salt(8 bytes)
									pbe.key,
									pbe.iv);

	if (plaintext === false) return false;

	if (utf8)
		plaintext = UTF8.decode(plaintext);

	return plaintext;
};

AES.prototype.rawDecrypt = function(ciphertext, key, iv) {
	var byteArray = AES.ByteArray.fromString(ciphertext);

	if (key.length != this.Nk * 4 || iv.length < AES.BLOCK_SIZE) {
		this.appendMsg('AES.decrypt encountered an error! The length of key or iv may not be correct!');
		return false;
	}		
	if (byteArray.length % AES.BLOCK_SIZE != 0) {
		this.appendMsg('AES.decrypt encountered an error! The length of ciphertext is not correct!');
		return false;
	}

	var keySchedule = this._createKeySchedule(key);

	// separate byteArray into block
	var nBlocks = Math.ceil(byteArray.length / AES.BLOCK_SIZE);

	var plainText = '';
	var state = AES.ByteArray.fromString(iv);
	var cipherBlock = [];
	var dec_state = [];

	for (var b = 0; b < nBlocks; b++){
		cipherBlock = byteArray.slice(b * AES.BLOCK_SIZE, b * AES.BLOCK_SIZE + AES.BLOCK_SIZE);
		dec_state = this._decryptBlock(cipherBlock, keySchedule); // decrypt ciphertext block
		plainText += AES.ByteArray.toString((b == nBlocks-1) ? this._PKCS5unpad(this._xorBlock(state, dec_state)) : this._xorBlock(state, dec_state));
		state = cipherBlock.slice(); // save old ciphertext for next round
	}

	return plainText;
};


AES.prototype._PKCS5pad = function(byteArray) {
	// PKCS#5 padding
	var padding = AES.BLOCK_SIZE - (byteArray.length % AES.BLOCK_SIZE);

	for(var c = 0; c < padding; c++)
		byteArray[byteArray.length] = padding;

	return byteArray;
};


AES.prototype._PKCS5unpad = function(block) {
	// remove PKCS#5 Padding
	var padding = block[block.length-1];

	if (padding == block[block.length-padding] && padding <= 16)
		block = block.slice(0, block.length-padding);
	
	return block;
};


AES.prototype._createKeySchedule = function(key) {
	var keyBytes = [];

	for(var i = 0; i < this.Nk * 4; i++) {
		keyBytes[i] = isNaN(key.charCodeAt(i)) ? 0 : key.charCodeAt(i);
	}

	// generate key schedule
	return this._expandKey(keyBytes);
};


AES.prototype._expandKey  = function(input) {
	var key = input.slice();
	var kl = key.length; // key.lenth == this.Nk * 4
	var Rcon = 1; 
/*
	for (var i = kl; i < 16 * (this.Nr + 1); i += 4) {
		var word = key.slice(i-4, i); // 4-byte word
		if (i % kl == 0) {
//
//			word = new Array(AES.SBOX[word[1]] ^ Rcon, AES.SBOX[word[2]],
//											 AES.SBOX[word[3]], AES.SBOX[word[0]]);
//
			word = this._subWord(this._rotWord(word));
			word[0] ^= Rcon;

			if ((Rcon <<= 1) >= 256)
				Rcon ^= 0x11b;
		}
		else if ((kl > 24) && (i % kl == 16)) {
//
//			word = new Array(AES.SBOX[word[0]], AES.SBOX[word[1]], AES.SBOX[word[2]], AES.SBOX[word[3]]);
//
			word = this._subWord(word);
		}
		for(var j = 0; j < 4; j++)
			key[i + j] = key[i + j - kl] ^ word[j];
	}
*/

	for (var i = kl; i < 16 * (this.Nr + 1); i += 4) {
		var word = key.slice(i-4, i); // 4-byte word
		if (i % kl == 0) {
			word = this._subWord(this._rotWord(word));
			word[0] ^= Rcon;

			if ((Rcon <<= 1) >= 256)
				Rcon ^= 0x11b; // [0x1,0x2,0x4,0x8,0x10,0x20,0x40,0x80]
		}
		else if ((kl > 24) && (i % kl == 16)) {
			word = this._subWord(word);
		}
		for(var j = 0; j < 4; j++)
			key[i + j] = key[i + j - kl] ^ word[j];
	}

	return key;
};


AES.prototype._subWord = function(w) {
	// apply SBOX to 4-byte word w
	for (var i = 0; i < 4; i++) {
		w[i] = AES.SBOX[w[i]];
	}
	return w;
};

AES.prototype._rotWord = function(w) {
	// rotate 4-byte word w left by one byte
	var tmp = w[0];
	for (var i = 0; i < 4; i++) {
		w[i] = (i==3) ? tmp : w[i+1];
	}
	return w;
};


AES.prototype._xorBlock = function(a1, a2) {
	// xor the elements of two arrays together
	 var res = [];

	 for(var i = 0; i < a1.length; i++)
			res[i] = a1[i] ^ a2[i];

	 return res;
};


AES.prototype._subBytes = function(state, inverse) {
	var box = (typeof(inverse) == 'undefined') ? AES.SBOX.slice() : AES.SBOX_INVERSE.slice();

	for(var i = 0; i < 16; i++)
		state[i] = box[state[i]];

	return state;
};

AES.prototype._addRoundKey = function(state, rkey) {
	for(var i = 0; i < 16; i++)
		state[i] ^= rkey[i];

	return state;
};

AES.prototype._shiftRows = function(state, inverse) {
	var shifttab = (typeof(inverse) == 'undefined') ? AES.SHIFTBY.slice() : AES.SHIFTBY_INVERSE.slice();
	//var h = new Array().concat(state); // or state.slice()
	var h = state.slice();

	for(var i = 0; i < 16; i++)
		state[i] = h[shifttab[i]];

	return state;
};

AES.prototype._mixColumns = function(state, inverse) {
	for(var i = 0; i < 16; i += 4) {
		var s0 = state[i + 0], s1 = state[i + 1];
		var s2 = state[i + 2], s3 = state[i + 3];
		var h = s0 ^ s1 ^ s2 ^ s3;
/*
		if (typeof(inverse) == 'undefined') {
			state[i + 0] ^= h ^ AES.XTIME[s0 ^ s1];
			state[i + 1] ^= h ^ AES.XTIME[s1 ^ s2];
			state[i + 2] ^= h ^ AES.XTIME[s2 ^ s3];
			state[i + 3] ^= h ^ AES.XTIME[s3 ^ s0];
		} else {
			var xh = AES.XTIME[h];
			var h1 = AES.XTIME[AES.XTIME[xh ^ s0 ^ s2]] ^ h;
			var h2 = AES.XTIME[AES.XTIME[xh ^ s1 ^ s3]] ^ h;
			state[i + 0] ^= h1 ^ AES.XTIME[s0 ^ s1];
			state[i + 1] ^= h2 ^ AES.XTIME[s1 ^ s2];
			state[i + 2] ^= h1 ^ AES.XTIME[s2 ^ s3];
			state[i + 3] ^= h2 ^ AES.XTIME[s3 ^ s0];
		}			
*/
		if (typeof(inverse) == 'undefined') {
			state[i + 0] ^= h ^ AES.xtime(s0 ^ s1);
			state[i + 1] ^= h ^ AES.xtime(s1 ^ s2);
			state[i + 2] ^= h ^ AES.xtime(s2 ^ s3);
			state[i + 3] ^= h ^ AES.xtime(s3 ^ s0);
		} else {
			var xh = AES.xtime(h);
			var h1 = AES.xtime(AES.xtime(xh ^ s0 ^ s2)) ^ h;
			var h2 = AES.xtime(AES.xtime(xh ^ s1 ^ s3)) ^ h;
			state[i + 0] ^= h1 ^ AES.xtime(s0 ^ s1);
			state[i + 1] ^= h2 ^ AES.xtime(s1 ^ s2);
			state[i + 2] ^= h1 ^ AES.xtime(s2 ^ s3);
			state[i + 3] ^= h2 ^ AES.xtime(s3 ^ s0);
		}			

	}
	return state;
};


AES.prototype._encryptBlock = function(input, key) {
	var len = key.length;
	var block = input.slice();

	block = this._addRoundKey(block, key.slice(0, 16));

	for(var round = 1; round < this.Nr; round++) {
		block = this._subBytes(block);
		block = this._shiftRows(block);
		block = this._mixColumns(block);
		block = this._addRoundKey(block, key.slice(round * 16, (round * 16) + 16));
	}
	block = this._subBytes(block);
	block = this._shiftRows(block);
	block = this._addRoundKey(block, key.slice(this.Nr * 16, len));
/*
	for(var round = 1; round <= this.Nr; round++) {
		block = this._subBytes(block);
		block = this._shiftRows(block);
		if (round < this.Nr) { // last round? don't mixColumns
			block = this._mixColumns(block);
		}
		block = this._addRoundKey(block, key.slice(round * 16, (round * 16) + 16));
	}
*/
	return block;
};


AES.prototype._decryptBlock = function(input, key) {
	var len = key.length;
	var block = input.slice();

	block = this._addRoundKey(block, key.slice(len - 16, len));
	block = this._shiftRows(block, 1);//1=inverse operation
	block = this._subBytes(block, 1);//1=inverse operation
	for(var round = this.Nr-1; round >= 1; round--) {
		block = this._addRoundKey(block, key.slice(round * 16, (round * 16) + 16));
		block = this._mixColumns(block, 1);//1=inverse operation
		block = this._shiftRows(block, 1);//1=inverse operation
		block = this._subBytes(block, 1);//1=inverse operation
	}
	block = this._addRoundKey(block, key.slice(0, 16));
/*
	for(var round = this.Nr; round >= 1; round--) {
		block = this._addRoundKey(block, key.slice(round * 16, (round * 16) + 16));
		if (round < this.Nr) {// last block? don't mixColums
			block = this._mixColumns(block, 1); // inverse operation
		}
		block = this._shiftRows(block, 1); // inverse operation
		block = this._subBytes(block, 1); // inverse operation
	}
	block = this._addRoundKey(block, key.slice(0, 16));
*/
	return block;
};


AES.prototype._createPBEKey = function(password, salt) {
	// creates Password Based Encryption key and initial vector
	var round = 3;
	var data00 = password + salt;
	var result = '';
/*
	var md5_hash = [];

	for(var i = 0; i < round; i++){
		result = AES.MD5(AES.Hex.toString(result) + data00);
		md5_hash[i] = result;
	}

	switch(this.Nk){
		case 4:// 128 bit
			key = md5_hash[0];
			iv = md5_hash[1];
			break;
		case 6:// 192 bit
			key = md5_hash[0] + md5_hash[1].substr(0,16);
			iv = md5_hash[2];
			break;
		case 8:// 256 bit
			key = md5_hash[0] + md5_hash[1];
			iv = md5_hash[2];
			break;
		default:
			this.appendMsg('The allowed bits are 128, 192 and 256');
			return false;
	}
	return {
		key: AES.Hex.toString(key),
		iv: AES.Hex.toString(iv)
	};
*/
	var md5_hash = '';

	for(var i = 0; i < round; i++){
		result = AES.MD5(AES.Hex.toString(result) + data00);
		md5_hash += result;
	}

	// md5 returns 128-bit hash value, which means 32 bytes.
	// 128 bit => 4 * 4(Nk) = 16 bytes key size.
	// 192 bit => 4 * 6(Nk) = 24 bytes key size.
	// 256 bit => 4 * 8(Nk) = 32 bytes key size.
	// iv is always 16 bytes. */
	//
	// AES.Hex.toString function converts two-byte hex to make one-byte string
	// therefore the key length is (4 * Nk) * 2
	// the iv length is 16 * 2
	return {
		key: AES.Hex.toString(md5_hash.substr(0, 2 * 4 * this.Nk)),
		iv: AES.Hex.toString(md5_hash.substr(md5_hash.length - (16 * 2)))
	};
};



AES.MD5 = function(string, utf8_encoding) {
	function RotateLeft(lValue, iShiftBits) {
		return (lValue<<iShiftBits) | (lValue>>>(32-iShiftBits));
	}

	function AddUnsigned(lX,lY) {
		var lX4,lY4,lX8,lY8,lResult;
		lX8 = (lX & 0x80000000);
		lY8 = (lY & 0x80000000);
		lX4 = (lX & 0x40000000);
		lY4 = (lY & 0x40000000);
		lResult = (lX & 0x3FFFFFFF)+(lY & 0x3FFFFFFF);
		if (lX4 & lY4) {
			return (lResult ^ 0x80000000 ^ lX8 ^ lY8);
		}
		if (lX4 | lY4) {
			if (lResult & 0x40000000) {
				return (lResult ^ 0xC0000000 ^ lX8 ^ lY8);
			} else {
				return (lResult ^ 0x40000000 ^ lX8 ^ lY8);
			}
		} else {
			return (lResult ^ lX8 ^ lY8);
		}
	}

	function F(x,y,z) { return (x & y) | ((~x) & z); }
	function G(x,y,z) { return (x & z) | (y & (~z)); }
	function H(x,y,z) { return (x ^ y ^ z); }
	function I(x,y,z) { return (y ^ (x | (~z))); }

	function FF(a,b,c,d,x,s,ac) {
		a = AddUnsigned(a, AddUnsigned(AddUnsigned(F(b, c, d), x), ac));
		return AddUnsigned(RotateLeft(a, s), b);
	};

	function GG(a,b,c,d,x,s,ac) {
		a = AddUnsigned(a, AddUnsigned(AddUnsigned(G(b, c, d), x), ac));
		return AddUnsigned(RotateLeft(a, s), b);
	};

	function HH(a,b,c,d,x,s,ac) {
		a = AddUnsigned(a, AddUnsigned(AddUnsigned(H(b, c, d), x), ac));
		return AddUnsigned(RotateLeft(a, s), b);
	};

	function II(a,b,c,d,x,s,ac) {
		a = AddUnsigned(a, AddUnsigned(AddUnsigned(I(b, c, d), x), ac));
		return AddUnsigned(RotateLeft(a, s), b);
	};

	function ConvertToWordArray(string) {
		var lWordCount;
		var lMessageLength = string.length;
		var lNumberOfWords_temp1=lMessageLength + 8;
		var lNumberOfWords_temp2=(lNumberOfWords_temp1-(lNumberOfWords_temp1 % 64))/64;
		var lNumberOfWords = (lNumberOfWords_temp2+1)*16;
		var lWordArray=Array(lNumberOfWords-1);
		var lBytePosition = 0;
		var lByteCount = 0;
		while ( lByteCount < lMessageLength ) {
			lWordCount = (lByteCount-(lByteCount % 4))/4;
			lBytePosition = (lByteCount % 4)*8;
			lWordArray[lWordCount] = (lWordArray[lWordCount] | (string.charCodeAt(lByteCount)<<lBytePosition));
			lByteCount++;
		}
		lWordCount = (lByteCount-(lByteCount % 4))/4;
		lBytePosition = (lByteCount % 4)*8;
		lWordArray[lWordCount] = lWordArray[lWordCount] | (0x80<<lBytePosition);
		lWordArray[lNumberOfWords-2] = lMessageLength<<3;
		lWordArray[lNumberOfWords-1] = lMessageLength>>>29;
		return lWordArray;
	};

	function WordToHex(lValue) {
		var WordToHexValue="",WordToHexValue_temp="",lByte,lCount;
		for (lCount = 0;lCount<=3;lCount++) {
			lByte = (lValue>>>(lCount*8)) & 255;
			WordToHexValue_temp = "0" + lByte.toString(16);
			WordToHexValue = WordToHexValue + WordToHexValue_temp.substr(WordToHexValue_temp.length-2,2);
		}
		return WordToHexValue;
	};

	if (typeof utf8_encoding == 'undefined') utf8_encoding = false;

	// to make the return value as the same as PHP returns
	// convert from UTF16 string
	if (utf8_encoding) string = UTF8.fromUTF16(string);

	var x=Array();
	var k,AA,BB,CC,DD,a,b,c,d;
	var S11=7, S12=12, S13=17, S14=22;
	var S21=5, S22=9 , S23=14, S24=20;
	var S31=4, S32=11, S33=16, S34=23;
	var S41=6, S42=10, S43=15, S44=21;

	x = ConvertToWordArray(string);

	a = 0x67452301; b = 0xEFCDAB89; c = 0x98BADCFE; d = 0x10325476;

	for (k=0;k<x.length;k+=16) {
		AA=a; BB=b; CC=c; DD=d;
		a=FF(a,b,c,d,x[k+0], S11,0xD76AA478);
		d=FF(d,a,b,c,x[k+1], S12,0xE8C7B756);
		c=FF(c,d,a,b,x[k+2], S13,0x242070DB);
		b=FF(b,c,d,a,x[k+3], S14,0xC1BDCEEE);
		a=FF(a,b,c,d,x[k+4], S11,0xF57C0FAF);
		d=FF(d,a,b,c,x[k+5], S12,0x4787C62A);
		c=FF(c,d,a,b,x[k+6], S13,0xA8304613);
		b=FF(b,c,d,a,x[k+7], S14,0xFD469501);
		a=FF(a,b,c,d,x[k+8], S11,0x698098D8);
		d=FF(d,a,b,c,x[k+9], S12,0x8B44F7AF);
		c=FF(c,d,a,b,x[k+10],S13,0xFFFF5BB1);
		b=FF(b,c,d,a,x[k+11],S14,0x895CD7BE);
		a=FF(a,b,c,d,x[k+12],S11,0x6B901122);
		d=FF(d,a,b,c,x[k+13],S12,0xFD987193);
		c=FF(c,d,a,b,x[k+14],S13,0xA679438E);
		b=FF(b,c,d,a,x[k+15],S14,0x49B40821);
		a=GG(a,b,c,d,x[k+1], S21,0xF61E2562);
		d=GG(d,a,b,c,x[k+6], S22,0xC040B340);
		c=GG(c,d,a,b,x[k+11],S23,0x265E5A51);
		b=GG(b,c,d,a,x[k+0], S24,0xE9B6C7AA);
		a=GG(a,b,c,d,x[k+5], S21,0xD62F105D);
		d=GG(d,a,b,c,x[k+10],S22,0x2441453);
		c=GG(c,d,a,b,x[k+15],S23,0xD8A1E681);
		b=GG(b,c,d,a,x[k+4], S24,0xE7D3FBC8);
		a=GG(a,b,c,d,x[k+9], S21,0x21E1CDE6);
		d=GG(d,a,b,c,x[k+14],S22,0xC33707D6);
		c=GG(c,d,a,b,x[k+3], S23,0xF4D50D87);
		b=GG(b,c,d,a,x[k+8], S24,0x455A14ED);
		a=GG(a,b,c,d,x[k+13],S21,0xA9E3E905);
		d=GG(d,a,b,c,x[k+2], S22,0xFCEFA3F8);
		c=GG(c,d,a,b,x[k+7], S23,0x676F02D9);
		b=GG(b,c,d,a,x[k+12],S24,0x8D2A4C8A);
		a=HH(a,b,c,d,x[k+5], S31,0xFFFA3942);
		d=HH(d,a,b,c,x[k+8], S32,0x8771F681);
		c=HH(c,d,a,b,x[k+11],S33,0x6D9D6122);
		b=HH(b,c,d,a,x[k+14],S34,0xFDE5380C);
		a=HH(a,b,c,d,x[k+1], S31,0xA4BEEA44);
		d=HH(d,a,b,c,x[k+4], S32,0x4BDECFA9);
		c=HH(c,d,a,b,x[k+7], S33,0xF6BB4B60);
		b=HH(b,c,d,a,x[k+10],S34,0xBEBFBC70);
		a=HH(a,b,c,d,x[k+13],S31,0x289B7EC6);
		d=HH(d,a,b,c,x[k+0], S32,0xEAA127FA);
		c=HH(c,d,a,b,x[k+3], S33,0xD4EF3085);
		b=HH(b,c,d,a,x[k+6], S34,0x4881D05);
		a=HH(a,b,c,d,x[k+9], S31,0xD9D4D039);
		d=HH(d,a,b,c,x[k+12],S32,0xE6DB99E5);
		c=HH(c,d,a,b,x[k+15],S33,0x1FA27CF8);
		b=HH(b,c,d,a,x[k+2], S34,0xC4AC5665);
		a=II(a,b,c,d,x[k+0], S41,0xF4292244);
		d=II(d,a,b,c,x[k+7], S42,0x432AFF97);
		c=II(c,d,a,b,x[k+14],S43,0xAB9423A7);
		b=II(b,c,d,a,x[k+5], S44,0xFC93A039);
		a=II(a,b,c,d,x[k+12],S41,0x655B59C3);
		d=II(d,a,b,c,x[k+3], S42,0x8F0CCC92);
		c=II(c,d,a,b,x[k+10],S43,0xFFEFF47D);
		b=II(b,c,d,a,x[k+1], S44,0x85845DD1);
		a=II(a,b,c,d,x[k+8], S41,0x6FA87E4F);
		d=II(d,a,b,c,x[k+15],S42,0xFE2CE6E0);
		c=II(c,d,a,b,x[k+6], S43,0xA3014314);
		b=II(b,c,d,a,x[k+13],S44,0x4E0811A1);
		a=II(a,b,c,d,x[k+4], S41,0xF7537E82);
		d=II(d,a,b,c,x[k+11],S42,0xBD3AF235);
		c=II(c,d,a,b,x[k+2], S43,0x2AD7D2BB);
		b=II(b,c,d,a,x[k+9], S44,0xEB86D391);
		a=AddUnsigned(a,AA);
		b=AddUnsigned(b,BB);
		c=AddUnsigned(c,CC);
		d=AddUnsigned(d,DD);
	}
	var temp = WordToHex(a)+WordToHex(b)+WordToHex(c)+WordToHex(d);

	// to make the return value as the same as PHP returns
	// convert to UTF16 string and return;
	//return UTF8.toUTF16(temp.toLowerCase());
	return utf8_encoding ? UTF8.toUTF16(temp.toLowerCase()) : temp.toLowerCase();
};


var UTF8 = {
	encode: function(input) {
		/* Why do we need converting functions between UTF16 and UTF8? 
		   Because the output of PHP utf8_encode() and Javascript UTF8 function output are different from each other.
		   
		   JavaScript strings are UCS-2 encoded but can represent Unicode code points
		   outside the Basic Multilingual Pane (U+0000 - U+D7FF and U+E000 - U+FFFF)
		   using two 16 bit numbers (a UTF-16 surrogate pair), 
		   the first of which must be in the range U+D800 - U+DFFF. */
		input = this.fromUTF16(input);

		input = input.replace(/\r\n/g,"\n");
		var output = "";
		for (var n = 0; n < input.length; n++) {
			var c = input.charCodeAt(n);

			if (c < 128) {
				output += String.fromCharCode(c);
			} else if ((c > 127) && (c < 2048)) {
/*
				output += String.fromCharCode((c >> 6) | 192);
				output += String.fromCharCode((c & 63) | 128);
*/
				output += String.fromCharCode((c >> 6) | 192, (c & 63) | 128);
			} else {
/*
				output += String.fromCharCode((c >> 12) | 224);
				output += String.fromCharCode(((c >> 6) & 63) | 128); // 192 rather than 128?
																	  //  see php_utf8_encode() in ext/xml/xml.c of php source.
				output += String.fromCharCode((c & 63) | 128);
*/
				output += String.fromCharCode((c >> 12) | 224, ((c >> 6) & 63) | 128 /* 192 */, (c & 63) | 128);
			}
		}
		return output;
	},

	decode: function(input) {
		var output = "";
		var i = 0;
		var c1 = c2 = c3 = c4 = 0;
		while ( i < input.length ) {
			c1 = input.charCodeAt(i);

			if (c1 < 128) {
				output += String.fromCharCode(c1);
				i++;
			} else if((c1 > 191) && (c1 < 224)) {
				c2 = input.charCodeAt(i+1);
				output += String.fromCharCode(((c1 & 31) << 6) | (c2 & 63));
				i += 2;
			} else {
				c2 = input.charCodeAt(i+1);
				c3 = input.charCodeAt(i+2);
				output += String.fromCharCode(((c1 & 15) << 12) | ((c2 & 63) << 6) | (c3 & 63));
				i += 3;
			}
		}

		output = this.toUTF16(output);

		return output;
	},


	fromUTF16: function(input) {
		/* converts UTF16 chracters to UTF8 */
		var output, i, len, c;

		output = "";
		len = input.length;
		for(i = 0; i < len; i++) {
			c = input.charCodeAt(i);
			if ((c >= 0x0001) && (c <= 0x007F)) {
				output += input.charAt(i);
			} else if (c > 0x07FF) {
				output += String.fromCharCode(0xE0 | ((c >> 12) & 0x0F));
				output += String.fromCharCode(0x80 | ((c >>  6) & 0x3F));
				output += String.fromCharCode(0x80 | ((c >>  0) & 0x3F));
			} else {
				output += String.fromCharCode(0xC0 | ((c >>  6) & 0x1F));
				output += String.fromCharCode(0x80 | ((c >>  0) & 0x3F));
			}
		}
		return output;
	},

	toUTF16: function(input) {
		/* converts UTF8 characters to UTF16 */
		var output, i, len, c;
		var char2, char3;

		output = "";
		len = input.length;
		i = 0;
		while(i < len) {
			c = input.charCodeAt(i++);
			switch(c >> 4) { 
				case 0:
				case 1:
				case 2:
				case 3:
				case 4:
				case 5:
				case 6:
				case 7:
					// 0xxxxxxx
					output += input.charAt(i-1);
					break;
				case 12:
				case 13:
					// 110x xxxx   10xx xxxx
					char2 = input.charCodeAt(i++);
					output += String.fromCharCode(((c & 0x1F) << 6) | (char2 & 0x3F));
					break;
				case 14:
					// 1110 xxxx  10xx xxxx  10xx xxxx
					char2 = input.charCodeAt(i++);
					char3 = input.charCodeAt(i++);
					output += String.fromCharCode(((c & 0x0F) << 12) |
								   ((char2 & 0x3F) << 6) |
								   ((char3 & 0x3F) << 0));
					break;
			}
		}

		return output;
	}



};


var BASE64 = {
	base64: "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=",

	encode: function(input) {
		if (!input) {
			return false;
		}
		var output = "";
		var chr1, chr2, chr3;
		var enc1, enc2, enc3, enc4;
		var i = 0;
		do {
			chr1 = input.charCodeAt(i++);
			chr2 = input.charCodeAt(i++);
			chr3 = input.charCodeAt(i++);
			enc1 = chr1 >> 2;
			enc2 = ((chr1 & 3) << 4) | (chr2 >> 4);
			enc3 = ((chr2 & 15) << 2) | (chr3 >> 6);
			enc4 = chr3 & 63;
			if (isNaN(chr2)) enc3 = enc4 = 64;
			else if (isNaN(chr3)) enc4 = 64;
			output += this.base64.charAt(enc1) + this.base64.charAt(enc2) + this.base64.charAt(enc3) + this.base64.charAt(enc4);
		} while (i < input.length);
		return output;
	},

	decode: function(input) {
		if(!input) return false;
		input = input.replace(/[^A-Za-z0-9\+\/\=]/g, "");
		var output = "";
		var enc1, enc2, enc3, enc4;
		var i = 0;
		do {
			enc1 = this.base64.indexOf(input.charAt(i++));
			enc2 = this.base64.indexOf(input.charAt(i++));
			enc3 = this.base64.indexOf(input.charAt(i++));
			enc4 = this.base64.indexOf(input.charAt(i++));
			output += String.fromCharCode((enc1 << 2) | (enc2 >> 4));
			if (enc3 != 64) output += String.fromCharCode(((enc2 & 15) << 4) | (enc3 >> 2));
			if (enc4 != 64) output += String.fromCharCode(((enc3 & 3) << 6) | enc4);
		} while (i < input.length);
		return output;
	}
};