<?php
session_start();

// 로그인 확인
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

// DB 연결
include('connect.php');

// 변수 초기화
$user_id = $_SESSION['user_id']; // 세션에서 가져온 사용자 ID
$message = "";

// 만약 폼이 전송되었다면 (POST 요청)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // 입력값 가져오기
    $email = trim($_POST['email']);
    $is_admin = trim($_POST['is_admin']);  // 1이면 관리자, 0이면 일반회원

    // 비밀번호(옵션)
    $password = trim($_POST['password']);
    $confirmPassword = trim($_POST['confirm_password']);

    // 사진(옵션)
    $photo = isset($_FILES['photo']) ? $_FILES['photo'] : null;

    // 이메일 검증
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $message = "유효한 이메일 주소를 입력하세요.";
    } else {
        // 만약 비밀번호가 입력되었다면(변경 시도)
        $hashedPassword = null;
        if (!empty($password) || !empty($confirmPassword)) {
            if ($password !== $confirmPassword) {
                $message = "비밀번호가 일치하지 않습니다.";
            } else {
                $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
            }
        }

        // 사진 업로드 처리
        $photoPath = null;
        if ($photo && $photo['error'] === UPLOAD_ERR_OK) {
            $uploadDir = 'uploads/photos/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0777, true);
            }
            $photoPath = $uploadDir . uniqid() . "_" . basename($photo['name']);

            if (!move_uploaded_file($photo['tmp_name'], $photoPath)) {
                $message = "사진 업로드 실패.";
            }
        }

        // 에러 메시지가 없다면 DB 업데이트 진행
        if (empty($message)) {
            try {
                // 기존 데이터 가져오기
                $sqlSelect = "SELECT photo FROM zb_users WHERE user_id = :user_id";
                $stmtSelect = $conn->prepare($sqlSelect);
                $stmtSelect->bindParam(':user_id', $user_id);
                $stmtSelect->execute();
                $oldData = $stmtSelect->fetch(PDO::FETCH_ASSOC);

                $oldPhotoPath = $oldData['photo'];

                // 업데이트할 필드 동적 구성
                $fieldsToUpdate = "email = :email, is_admin = :is_admin";
                if ($hashedPassword) {
                    $fieldsToUpdate .= ", password = :password";
                }
                if ($photoPath) {
                    $fieldsToUpdate .= ", photo = :photo";
                }

                $sqlUpdate = "UPDATE zb_users 
                              SET $fieldsToUpdate 
                              WHERE user_id = :user_id";
                $stmtUpdate = $conn->prepare($sqlUpdate);

                $stmtUpdate->bindValue(':email', $email);
                $stmtUpdate->bindValue(':is_admin', $is_admin);

                if ($hashedPassword) {
                    $stmtUpdate->bindValue(':password', $hashedPassword);
                }
                if ($photoPath) {
                    $stmtUpdate->bindValue(':photo', $photoPath);
                }
                $stmtUpdate->bindValue(':user_id', $user_id);

                if ($stmtUpdate->execute()) {
                    // 기존 사진 삭제(선택)
                    if ($photoPath && $oldPhotoPath && file_exists($oldPhotoPath)) {
                        // unlink($oldPhotoPath);
                    }
                    
                    // 업데이트 성공 시 관리자 / 일반회원 분기 처리
                    if ($is_admin == 1) {
                        // 관리자
                        echo "<script>
                                alert('관리자님 정보가 변경이 되었습니다.');
                                window.location.href = 'admin_dashboard_pc.php';
                              </script>";
                        exit;
                    } else {
                        // 일반회원
                        // $user_id 님 정보가 변경되었다는 메시지
                        echo "<script>
                                alert('{$user_id}님 정보가 변경이 되었습니다.');
                                window.location.href = 'user_dashboard_pc.php';
                              </script>";
                        exit;
                    }
                    
                } else {
                    $message = "회원정보 업데이트 실패.";
                }
            } catch (Exception $e) {
                $message = "오류 발생: " . $e->getMessage();
            }
        }
    }
}

// DB에서 사용자 정보 다시 조회 (최신 정보 반영)
$sql = "SELECT user_id, email, is_admin, photo 
        FROM zb_users 
        WHERE user_id = :user_id";
$stmt = $conn->prepare($sql);
$stmt->bindParam(':user_id', $user_id);
$stmt->execute();
$userData = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$userData) {
    die("유효하지 않은 사용자입니다.");
}
?>
<!DOCTYPE html>
<html lang="ko">
<head>
    <meta charset="UTF-8">
    <title>회원정보 수정</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        /* 공통 스타일 */
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            padding: 0;
            background-color: #f9f9f9;
        }
        .container {
            max-width: 480px;
            margin: 0 auto;
            padding: 20px;
        }
        form {
            background: #ffffff;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            margin-top: 15px;
        }
        input, select {
            width: 100%;
            padding: 10px;
            margin-top: 5px;
            border: 1px solid #ccc;
            border-radius: 4px;
            font-size: 16px;
            box-sizing: border-box;
        }
        button {
            width: 100%;
            padding: 12px;
            background: #007BFF;
            color: white;
            border: none;
            border-radius: 4px;
            font-size: 16px;
            cursor: pointer;
            margin-top: 20px;
        }
        button:hover {
            background: #0056b3;
        }
        .message {
            margin-top: 15px;
            color: red;
            font-weight: bold;
        }

        /* 모바일 친화적 디자인을 위한 미디어 쿼리 */
        @media screen and (max-width: 480px) {
            .container {
                padding: 10px;
            }
            form {
                padding: 15px;
            }
            label {
                margin-top: 10px;
            }
            button {
                margin-top: 15px;
            }
        }
    </style>
</head>
<body>
<div class="container">
    <h1>회원정보 수정</h1>
    <!-- 알림 메시지 표시 -->
    <?php if (!empty($message)): ?>
        <div class="message"><?= htmlspecialchars($message) ?></div>
    <?php endif; ?>

    <form action="" method="POST" enctype="multipart/form-data">
        <!-- user_id는 수정 불가(표시만) -->
        <label for="user_id">User ID (변경 불가)</label>
        <input type="text" id="user_id" name="user_id"
               value="<?= htmlspecialchars($userData['user_id']) ?>" readonly>

        <label for="email">Email</label>
        <input type="email" id="email" name="email" 
               value="<?= htmlspecialchars($userData['email']) ?>" required>

        <label for="is_admin">Role</label>
        <select id="is_admin" name="is_admin" required>
            <option value="0" <?= $userData['is_admin'] == 0 ? 'selected' : '' ?>>일반회원</option>
            <option value="1" <?= $userData['is_admin'] == 1 ? 'selected' : '' ?>>관리자</option>
        </select>

        <!-- 비밀번호 변경(옵션) -->
        <label for="password">새 비밀번호 (변경 시에만 입력)</label>
        <input type="password" id="password" name="password">

        <label for="confirm_password">새 비밀번호 확인</label>
        <input type="password" id="confirm_password" name="confirm_password">

        <!-- 기존 프로필 사진이 있다면 미리보기 -->
        <?php if (!empty($userData['photo'])): ?>
            <label>현재 등록된 사진</label>
            <img src="<?= htmlspecialchars($userData['photo']) ?>"
                 alt="Profile Photo"
                 style="max-width: 100%; height: auto;">
        <?php endif; ?>

        <label for="photo">프로필 사진 변경</label>
        <input type="file" id="photo" name="photo" accept="image/*">

        <button type="submit">정보 수정</button>
    </form>
</div>
</body>
</html>
