<?php
include_once('./_common.php');

/**
 * 그누보드 버전 체크 기능
 * GitHub에서 최신 버전을 가져와서 현재 버전과 비교
 */

// GitHub에서 그누보드 최신 버전 정보 가져오기
function get_gnuboard_latest_version() {
    $github_url = 'https://raw.githubusercontent.com/gnuboard/gnuboard5/master/version.php';
    
    // cURL을 사용하여 GitHub에서 파일 내용 가져오기
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $github_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (compatible; Gnuboard Version Checker)');
    
    $content = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code === 200 && $content !== false) {
        // G5_GNUBOARD_VER 값 추출
        if (preg_match("/define\s*\(\s*['\"]G5_GNUBOARD_VER['\"]\s*,\s*['\"]([^'\"]+)['\"]\s*\)/", $content, $matches)) {
            return trim($matches[1]);
        }
    }
    
    return false;
}

// 버전 비교 함수
function compare_gnuboard_version($current_version, $latest_version) {
    return version_compare($current_version, $latest_version, '<');
}

// 버전 체크 알림 HTML 생성
function generate_version_check_notice($current_version, $latest_version) {
    $notice_html = '
    <div class="version-check-notice">
        <div class="version-check-header">
            <div class="version-check-icon">⚠️</div>
            <div class="version-check-content">
                <strong>그누보드 새 버전이 있습니다!</strong><br>
                현재 버전: <span class="current-version">' . htmlspecialchars($current_version) . '</span><br>
                최신 버전: <span class="latest-version">' . htmlspecialchars($latest_version) . '</span>
            </div>
            <button type="button" class="version-check-close" onclick="hideVersionNotice()">×</button>
        </div>
        <div class="version-check-actions">
            <a href="https://github.com/gnuboard/gnuboard5/releases" target="_blank" class="version-check-btn">
                업데이트 확인
            </a>
        </div>
    </div>';
    
    return $notice_html;
}


function check_gnuboard_version_update() {
    global $config;
    
    // 관리자 페이지에서만 실행
    if (!defined('G5_IS_ADMIN') || !G5_IS_ADMIN) {
        return;
    }
    
    // 세션을 통한 중복 체크 방지 (하루에 한 번만 체크)
    $today = date('Y-m-d');
    if (isset($_SESSION['version_check_date']) && $_SESSION['version_check_date'] === $today) {
        return;
    }
    
    // 현재 버전 가져오기
    $current_version = G5_GNUBOARD_VER;
    
    // GitHub에서 최신 버전 가져오기
    $latest_version = get_gnuboard_latest_version();
    
    if ($latest_version && compare_gnuboard_version($current_version, $latest_version)) {
        // 새 버전이 있을 때만 알림 표시
        $notice_html = generate_version_check_notice($current_version, $latest_version);
        
        // CSS 스타일 추가
        echo '<style>
        .version-check-notice {
            position: fixed;
            top: 20px;
            right: 20px;
            background: #ffffff;
            color: #333333;
            padding: 15px;
            border-radius: 8px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
            border: 1px solid #e5e7eb;
            border-left: 4px solid #2563eb;
            z-index: 9999;
            max-width: 350px;
            font-family: "Malgun Gothic", "맑은 고딕", sans-serif;
            font-size: 13px;
            line-height: 1.4;
            animation: slideInRight 0.5s ease-out;
        }
        
        .version-check-header {
            display: flex;
            align-items: flex-start;
            gap: 10px;
            margin-bottom: 10px;
        }
        
        .version-check-icon {
            font-size: 20px;
            flex-shrink: 0;
        }
        
        .version-check-content {
            flex: 1;
        }
        
        .version-check-content strong {
            display: block;
            margin-bottom: 5px;
            font-size: 14px;
        }
        
        .current-version {
            background: #f3f4f6;
            color: #6b7280;
            padding: 2px 6px;
            border-radius: 3px;
            font-weight: bold;
            display: inline-block;
            margin-bottom: 4px;
        }
        
        .latest-version {
            background: #2563eb;
            color: white;
            padding: 2px 6px;
            border-radius: 3px;
            font-weight: bold;
        }
        
        .version-check-actions {
            display: flex;
            justify-content: center;
            margin-top: 5px;
        }
        
        .version-check-btn {
            background: #2563eb;
            color: white !important;
            text-decoration: none;
            padding: 8px 16px;
            border-radius: 4px;
            font-size: 12px;
            border: 1px solid #2563eb;
            transition: all 0.3s ease;
            width: 100%;
            text-align: center;
        }
        
        .version-check-btn:hover {
            background: #1d4ed8;
            border-color: #1d4ed8;
            transform: translateY(-1px);
        }
        
        .version-check-close {
            background: none;
            border: none;
            color: #6b7280;
            font-size: 16px;
            cursor: pointer;
            padding: 0;
            width: 20px;
            height: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            transition: background 0.3s ease;
        }
        
        .version-check-close:hover {
            background: #f3f4f6;
            color: #374151;
        }
        
        @keyframes slideInRight {
            from {
                transform: translateX(100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }
        
        @keyframes slideOutRight {
            from {
                transform: translateX(0);
                opacity: 1;
            }
            to {
                transform: translateX(100%);
                opacity: 0;
            }
        }
        
        .version-check-notice.hiding {
            animation: slideOutRight 0.3s ease-in forwards;
        }
        </style>';
        
        // JavaScript 함수 추가
        echo '<script>
        function hideVersionNotice() {
            var notice = document.querySelector(".version-check-notice");
            if (notice) {
                notice.classList.add("hiding");
                setTimeout(function() {
                    notice.remove();
                }, 300);
            }
        }
        
        // 20초 후 자동으로 숨기기
        setTimeout(function() {
            var notice = document.querySelector(".version-check-notice");
            if (notice && !notice.classList.contains("hiding")) {
                hideVersionNotice();
            }
        }, 20000);
        </script>';
        
        // 알림 HTML 출력
        echo $notice_html;
    }
    
    // 오늘 체크했다고 세션에 저장
    $_SESSION['version_check_date'] = $today;
}

check_gnuboard_version_update();