<?php
if (!defined('_GNUBOARD_')) exit; // 개별 페이지 접근 불가

// ========================================
// 통합 푸시 발송 시스템 (핵심 인프라)
// ========================================

/**
 * 통합 푸시 발송 함수 (모든 발송의 중심)
 * @param mixed $recipients 수신자 (member_ids, subscriptions, 빈 배열=전체)
 * @param string $title 푸시 제목
 * @param string $message 푸시 메시지
 * @param array $options 발송 옵션
 * @return array 발송 결과
 */
function send_push_unified($recipients, $title, $message, $options = array()) {
    $config = get_pushmanager_config();
    
    if (!$config || !$config['api_key']) {
        return array(
            'success' => false,
            'message' => 'API 키가 설정되지 않았습니다.',
            'code' => 'NO_API_KEY'
        );
    }
    
    // 1단계: 수신자 처리 및 구독 정보 조회
    $subscriptions = process_recipients($recipients, $options);
    
    if (empty($subscriptions)) {
        return array(
            'success' => false,
            'message' => '발송할 구독자가 없습니다.',
            'code' => 'NO_SUBSCRIBERS',
            'details' => array(
                'recipients_type' => get_recipients_type($recipients),
                'filter_applied' => !empty($options['notification_type'])
            )
        );
    }
    
    // 2단계: 페이로드 구성
    $payload = build_push_payload($title, $message, $options);
    
    // 3단계: 옵션 구성 (Node.js 서버 형식에 맞춤)
    $push_options = build_push_options($options);
    
    // 4단계: API 호출
    $api_url = rtrim($config['api_server_url'], '/') . '/api/push/multiple';
    $post_data = array(
        'subscriptions' => $subscriptions,
        'payload' => $payload,
        'options' => (object)$push_options
    );
    
    $result = send_api_request($api_url, $post_data, 'POST', $config['api_key']);
    
    // 5단계: 결과 처리 및 로깅
    return process_push_result($result, $subscriptions, $title, $message, $options);
}

/**
 * 수신자 처리 (모든 타입 지원)
 */
function process_recipients($recipients, $options = array()) {
    // 이미 구독 배열인 경우 (하위 호환성)
    if (is_array($recipients) && isset($recipients[0]['endpoint'])) {
        return $recipients;
    }
    
    // 회원 ID 배열인 경우
    if (is_array($recipients) && !empty($recipients)) {
        $filters = array('member_ids' => $recipients);
        return get_filtered_subscriptions($filters, $options);
    }
    
    // 문자열 하나인 경우 (단일 회원)
    if (is_string($recipients)) {
        $filters = array('member_ids' => array($recipients));
        return get_filtered_subscriptions($filters, $options);
    }
    
    // 빈 값인 경우 전체 구독자 (필터 옵션 적용)
    if (empty($recipients)) {
        return get_filtered_subscriptions(array(), $options);
    }
    
    return array();
}

/**
 * 필터링된 구독 정보 조회 (알림 설정 확인 포함)
 */
function get_filtered_subscriptions($filters = array(), $options = array()) {
    // 옵션에서 필터 추출
    $filter_options = array('level', 'gender', 'created_after');
    foreach ($filter_options as $filter_key) {
        if (isset($options[$filter_key])) {
            $filters[$filter_key] = $options[$filter_key];
        }
    }
    
    // 알림 타입이 지정된 경우 사용자 설정 확인
    $notification_type = $options['notification_type'] ?? null;
    
    if ($notification_type && $notification_type !== 'system') {
        return get_subscriptions_with_notification_check($filters, $notification_type);
    }
    
    // 시스템 알림이거나 타입이 없으면 바로 조회
    return get_active_subscriptions($filters);
}

/**
 * 알림 설정 확인 후 구독 정보 조회
 */
function get_subscriptions_with_notification_check($filters, $notification_type) {
    $all_subscriptions = get_active_subscriptions($filters);
    $valid_subscriptions = array();
    $skipped_members = array();
    
    // 알림 타입 검증
    $valid_types = array('comment', 'message', 'inquiry', 'system', 'newpost');
    if (!in_array($notification_type, $valid_types)) {
        return array();
    }
    
    foreach ($all_subscriptions as $subscription) {
        $member_id = $subscription['member_id'];
        
        // 비회원이거나 시스템 알림인 경우 무조건 포함
        if (!$member_id || $notification_type === 'system') {
            $valid_subscriptions[] = $subscription;
            continue;
        }
        
        // 회원의 알림 설정 확인
        if (is_notification_enabled($member_id, $notification_type)) {
            $valid_subscriptions[] = $subscription;
        } else {
            $skipped_members[] = $member_id;
        }
    }
    
    // 스킵된 회원이 있으면 로그 기록
    if (!empty($skipped_members)) {
        $skipped_list = implode(', ', array_unique($skipped_members));
        log_pushmanager_activity(
            'skip', 
            '알림 설정으로 제외됨', 
            "알림 타입: {$notification_type}, 제외된 회원: {$skipped_list}"
        );
    }
    
    return $valid_subscriptions;
}

/**
 * 푸시 페이로드 구성
 */
function build_push_payload($title, $message, $options = array()) {
    $payload = array(
        'title' => $title,
        'body' => $message,
        'message' => $message
    );
    
    // 기본 이미지 설정
    $payload['icon'] = $options['icon'] ?? get_push_asset_url('icon');
    $payload['badge'] = $options['badge'] ?? get_push_asset_url('badge');
    
    // 선택적 필드들
    $optional_fields = array(
        'image', 'url', 'click_action', 'tag', 'renotify', 
        'require_interaction', 'requireInteraction', 'silent',
        'actions', 'data', 'dir', 'lang', 'vibrate',
        'bo_table', 'wr_id', 'comment_id', 'me_id', 'mb_id', 'qa_id'
    );
    
    foreach ($optional_fields as $field) {
        if (isset($options[$field])) {
            $payload[$field] = $options[$field];
        }
    }
    
    // click_action을 url로도 설정 (하위 호환성)
    if (isset($options['click_action']) && !isset($payload['url'])) {
        $payload['url'] = $options['click_action'];
    }
    
    return $payload;
}

/**
 * 푸시 옵션 구성 (Node.js webpush 라이브러리용)
 */
function build_push_options($options = array()) {
    $push_options = array();
    
    // webpush 라이브러리 전용 옵션들
    $webpush_options = array('TTL', 'urgency', 'headers', 'vapidDetails');
    
    foreach ($webpush_options as $option) {
        if (isset($options[$option])) {
            $push_options[$option] = $options[$option];
        }
    }
    
    // 기본 TTL 설정
    if (!isset($push_options['TTL'])) {
        $push_options['TTL'] = 2419200; // 4주
    }
    
    return $push_options;
}

/**
 * 푸시 발송 결과 처리
 */
function process_push_result($api_result, $subscriptions, $title, $message, $options = array()) {
    $total_count = count($subscriptions);
    
    if (!$api_result['success']) {
        log_pushmanager_activity('error', $title, $message, $total_count, 0, $total_count, $api_result['message']);
        
        return array(
            'success' => false,
            'message' => $api_result['message'],
            'code' => $api_result['code'] ?? 'API_ERROR',
            'details' => array(
                'total_subscriptions' => $total_count,
                'api_error' => true
            )
        );
    }
    
    // Node.js 응답 데이터 파싱
    $response_data = $api_result['data'] ?? array();
    $success_count = $response_data['sent'] ?? 0;
    $error_count = $response_data['failed'] ?? 0;
    
    // 로그 기록
    log_pushmanager_activity('send', $title, $message, $total_count, $success_count, $error_count);
    
    // 푸시 발송 이력 저장 (성공한 경우에만)
    if ($success_count > 0) {
        // 구독 정보에서 member_id만 추출해서 전달
        $member_ids = array();
        foreach ($subscriptions as $subscription) {
            if (isset($subscription['member_id']) && $subscription['member_id']) {
                $member_ids[] = $subscription['member_id'];
            }
        }
        
        if (!empty($member_ids)) {
            save_push_history($member_ids, $title, $message, $options);
        }
    }

    // 통합 응답 형식
    return array(
        'success' => true,
        'message' => "푸시 발송 완료: {$success_count}건 성공, {$error_count}건 실패",
        'data' => array(
            'total' => $total_count,
            'sent' => $success_count,
            'failed' => $error_count,
            'success_count' => $success_count,
            'error_count' => $error_count,
            'success_rate' => $total_count > 0 ? round(($success_count / $total_count) * 100, 2) : 0,
            'payload_size' => $response_data['payloadSize'] ?? 0,
            'processing_time' => $response_data['processingTime'] ?? 0,
            'error_stats' => $response_data['errorStats'] ?? array()
        ),
        'details' => array(
            'notification_type' => $options['notification_type'] ?? 'system',
            'api_processing_time' => $response_data['processingTime'] ?? 0
        )
    );
}

/**
 * 수신자 타입 식별
 */
function get_recipients_type($recipients) {
    if (is_array($recipients) && isset($recipients[0]['endpoint'])) {
        return 'subscriptions';
    } elseif (is_array($recipients)) {
        return 'member_ids';
    } elseif (is_string($recipients)) {
        return 'single_member';
    } else {
        return 'all';
    }
}

// ========================================
// 간편 발송 함수들
// ========================================

/**
 * 관리자용 시스템 알림 발송 (설정 무시)
 */
function send_admin_notification($recipients, $title, $message, $options = array()) {
    $options['notification_type'] = 'system';
    $options['icon'] = $options['icon'] ?? get_push_asset_url('icon');
    $options['badge'] = $options['badge'] ?? get_push_asset_url('badge');
    
    return send_push_unified($recipients, $title, $message, $options);
}

/**
 * 댓글 알림 발송 (사용자 설정 확인)
 */
function send_comment_alert($recipients, $title, $message, $options = array()) {
    $options['notification_type'] = 'comment';
    return send_push_unified($recipients, $title, $message, $options);
}

/**
 * 쪽지 알림 발송 (사용자 설정 확인)
 */
function send_message_alert($recipients, $title, $message, $options = array()) {
    $options['notification_type'] = 'message';
    $options['require_interaction'] = true;
    return send_push_unified($recipients, $title, $message, $options);
}

/**
 * 레벨별 발송
 */
function send_push_by_level($level, $title, $message, $options = array()) {
    $options['level'] = $level;
    return send_push_unified(array(), $title, $message, $options);
}

/**
 * 성별 발송
 */
function send_push_by_gender($gender, $title, $message, $options = array()) {
    $options['gender'] = $gender;
    return send_push_unified(array(), $title, $message, $options);
}

// ========================================
// 하위 호환성 함수들 (기존 함수를 통합 함수 호출로 변경)
// ========================================

/**
 * 기존 send_push_notification 함수 (통합 함수 호출)
 */
function send_push_notification($title, $message, $options = array()) {
    $recipients = array();
    
    if (isset($options['member_ids'])) {
        $recipients = $options['member_ids'];
        unset($options['member_ids']);
    }
    
    return send_push_unified($recipients, $title, $message, $options);
}

/**
 * 기존 send_multiple_push_notifications 함수 (통합 함수 호출)
 */
function send_multiple_push_notifications($subscriptions, $payload, $options = array()) {
    $title = $payload['title'] ?? '';
    $message = $payload['body'] ?? $payload['message'] ?? '';
    
    $merged_options = array_merge($options, $payload);
    unset($merged_options['title'], $merged_options['body'], $merged_options['message']);
    
    return send_push_unified($subscriptions, $title, $message, $merged_options);
}

/**
 * 기존 send_notification 함수 (통합 함수 호출)
 */
function send_notification($notification_type, $member_ids, $title, $message, $options = array()) {
    $options['notification_type'] = $notification_type;
    return send_push_unified($member_ids, $title, $message, $options);
}

// ========================================
// 기존 특정 알림 함수들은 DEPRECATED - 훅에서 처리
// ========================================

/**
 * @deprecated 이 함수는 더 이상 사용하지 않습니다. 훅에서 메시지 포맷팅 후 send_push_unified를 직접 호출하세요.
 */
function send_new_post_notification($bo_table, $write_data) {
    // 하위 호환성을 위해 남겨두지만 사용 권장하지 않음
    $board_info = get_board_info($bo_table);
    if (!$board_info) {
        return array('success' => false, 'message' => '게시판 정보를 찾을 수 없습니다.');
    }
    
    $notification_users = get_board_notification_admins($bo_table);
    if (empty($notification_users)) {
        return array('success' => false, 'message' => '알림 설정된 사용자가 없습니다.');
    }
    
    // 간단한 기본 메시지로 발송 (포맷팅 없음)
    $title = "[{$board_info['bo_subject']}] 새글 알림";
    $message = "새글이 등록되었습니다: " . ($write_data['wr_subject'] ?? '제목 없음');
    
    $options = array(
        'notification_type' => 'system',
        'bo_table' => $bo_table,
        'wr_id' => $write_data['wr_id'] ?? 0
    );
    
    return send_push_unified($notification_users, $title, $message, $options);
}

/**
 * @deprecated 이 함수는 더 이상 사용하지 않습니다. 훅에서 메시지 포맷팅 후 send_push_unified를 직접 호출하세요.
 */
function send_comment_notification($bo_table, $wr_id, $comment_data) {
    // 하위 호환성을 위해 남겨두지만 사용 권장하지 않음
    $write_table = $GLOBALS['g5']['write_prefix'] . $bo_table;
    $sql = "SELECT wr_subject, mb_id FROM {$write_table} WHERE wr_id = " . intval($wr_id);
    $original_post = sql_fetch($sql);
    
    if (!$original_post || empty($original_post['mb_id'])) {
        return array('success' => false, 'message' => '원글이 없거나 비회원 글입니다.');
    }
    
    $title = "댓글 알림";
    $message = "새 댓글이 등록되었습니다: " . $original_post['wr_subject'];
    
    $options = array(
        'notification_type' => 'comment',
        'bo_table' => $bo_table,
        'wr_id' => $wr_id,
        'comment_id' => $comment_data['wr_id'] ?? 0
    );
    
    return send_push_unified(array($original_post['mb_id']), $title, $message, $options);
}

/**
 * @deprecated 이 함수는 더 이상 사용하지 않습니다. 훅에서 메시지 포맷팅 후 send_push_unified를 직접 호출하세요.
 */
function send_memo_notification($sender, $recipient, $me_id, $memo = '') {
    // 하위 호환성을 위해 남겨두지만 사용 권장하지 않음
    $title = "새 쪽지 알림";
    $message = ($sender['mb_nick'] ?: $sender['mb_name']) . "님으로부터 쪽지가 도착했습니다.";
    
    $options = array(
        'notification_type' => 'message',
        'me_id' => $me_id,
        'mb_id' => $sender['mb_id']
    );
    
    return send_push_unified(array($recipient['mb_id']), $title, $message, $options);
}

// ========================================
// 유틸리티 함수 (핵심 인프라)
// ========================================

/**
 * 푸시 에셋 URL 가져오기 (아이콘/배지)
 */
function get_push_asset_url($type = 'icon') {
    $config = get_pushmanager_config();
    
    $config_key = $type === 'badge' ? 'badge_icon_url' : 'push_icon_url';
    $default_file = $type === 'badge' ? 'push-badge.png' : 'push-icon.png';
    
    if ($config && !empty($config[$config_key])) {
        if (strpos($config[$config_key], 'http://') === 0 || strpos($config[$config_key], 'https://') === 0) {
            return $config[$config_key];
        } else {
            return G5_URL . '/' . ltrim($config[$config_key], '/');
        }
    }
    
    return get_site_origin() . '/img/' . $default_file;
}

/**
 * 알림 설정 확인 함수
 */
function is_notification_enabled($member_id, $notification_type) {
    if (!$member_id) {
        return true; // 비회원은 항상 허용
    }
    
    if ($notification_type === 'system') {
        return true; // 시스템 알림은 항상 허용
    }
    
    $valid_types = array('comment', 'message', 'inquiry', 'newpost');
    if (!in_array($notification_type, $valid_types)) {
        return false;
    }
    
    $user_settings = get_user_notification_settings($member_id);
    $setting_key = $notification_type . '_notification';
    
    if (!isset($user_settings[$setting_key])) {
        return true; // 설정이 없으면 기본값: 활성화
    }
    
    return (bool)$user_settings[$setting_key];
}

/**
 * 현재 사이트의 Origin URL 가져오기
 */
function get_site_origin() {
    $push_config = get_pushmanager_config();
    if ($push_config && !empty($push_config['site_url'])) {
        return rtrim($push_config['site_url'], '/');
    }
    
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    
    return $protocol . $host;
}

/**
 * API 요청용 공통 헤더 생성
 */
function get_api_headers($api_key = '') {
    $headers = array(
        'Content-Type: application/json',
        'User-Agent: GnuBoard5-PushManager/1.0',
        'Origin: ' . get_site_origin()
    );
    
    if (!empty($api_key)) {
        $headers[] = 'X-API-Key: ' . $api_key;
    }
    
    return $headers;
}

/**
 * Push Manager 설정 조회 (캐싱 적용)
 */
function get_pushmanager_config() {
    static $config_cache = null;
    
    if ($config_cache === null) {
        $sql = "SELECT * FROM pushmanager_config ORDER BY id DESC LIMIT 1";
        $result = sql_query($sql, false);
        
        if ($result && sql_num_rows($result) > 0) {
            $config_cache = sql_fetch_array($result);
        } else {
            $config_cache = false;
        }
    }
    
    return $config_cache;
}

/**
 * 공통 HTTP 응답 처리
 */
function parse_api_response($response, $http_code, $error) {
    if ($error) {
        return array(
            'success' => false,
            'message' => 'cURL 오류: ' . $error
        );
    }
    
    $decoded_response = json_decode($response, true);
    
    if ($http_code >= 200 && $http_code < 300) {
        return array(
            'success' => true,
            'data' => $decoded_response['data'] ?? $decoded_response ?? array(),
            'message' => $decoded_response['message'] ?? '성공'
        );
    } else {
        return array(
            'success' => false,
            'message' => $decoded_response['message'] ?? 'HTTP 오류: ' . $http_code,
            'code' => $decoded_response['code'] ?? 'HTTP_ERROR'
        );
    }
}

/**
 * 통합된 API 요청 함수
 */
function send_api_request($url, $data = array(), $method = 'POST', $api_key = '') {
    $ch = curl_init();
    
    curl_setopt_array($ch, array(
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_HTTPHEADER => get_api_headers($api_key)
    ));
    
    if ($method === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    } elseif ($method === 'GET' && !empty($data)) {
        $url .= '?' . http_build_query($data);
        curl_setopt($ch, CURLOPT_URL, $url);
    }
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    return parse_api_response($response, $http_code, $error);
}

/**
 * API 서버에 사이트 등록 및 키 발급 요청
 */
function request_api_registration($config) {
    $api_url = rtrim($config['api_server_url'], '/') . '/api/register';
    
    $post_data = array(
        'site_name' => $config['site_name'],
        'site_url' => $config['site_url'],
        'admin_email' => $config['admin_email']
    );
    
    return send_api_request($api_url, $post_data);
}

/**
 * API 서버 연결 테스트
 */
function test_api_connection($config) {
    $api_url = rtrim($config['api_server_url'], '/') . '/health';
    return send_api_request($api_url, array(), 'GET');
}

/**
 * 실제 IP 주소 가져오기
 */
function get_real_ip() {
    $ip_keys = array(
        'HTTP_CLIENT_IP',
        'HTTP_X_FORWARDED_FOR',
        'HTTP_X_FORWARDED',
        'HTTP_X_CLUSTER_CLIENT_IP',
        'HTTP_FORWARDED_FOR',
        'HTTP_FORWARDED',
        'REMOTE_ADDR'
    );
    
    foreach ($ip_keys as $key) {
        if (array_key_exists($key, $_SERVER) === true) {
            foreach (explode(',', $_SERVER[$key]) as $ip) {
                $ip = trim($ip);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                    return $ip;
                }
            }
        }
    }
    
    return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
}

/**
 * 구독 정보 저장 (회원별 독립 관리)
 */
function save_push_subscription($subscription_data) {
    global $member;
    
    $endpoint = addslashes($subscription_data['endpoint']);
    $p256dh_key = addslashes($subscription_data['keys']['p256dh']);
    $auth_key = addslashes($subscription_data['keys']['auth']);
    $member_id = $member['mb_id'] ?? null;
    $user_agent = addslashes($_SERVER['HTTP_USER_AGENT'] ?? '');
    $ip_address = get_real_ip();
    
    if (!$member_id) {
        return false; // 비회원인 경우 구독 정보를 저장하지 않음
    }
    
    // 현재 회원의 현재 endpoint 구독 정보 확인
    $sql = "SELECT id FROM pushmanager_subscriptions 
            WHERE member_id = '{$member_id}' AND endpoint = '{$endpoint}'";
    $existing = sql_fetch($sql);
    
    if ($existing) {
        // 동일한 회원의 동일한 기기인 경우 업데이트
        $sql = "UPDATE pushmanager_subscriptions SET 
                p256dh_key = '{$p256dh_key}',
                auth_key = '{$auth_key}',
                user_agent = '{$user_agent}',
                ip_address = '{$ip_address}',
                is_active = 1,
                updated_at = NOW()
                WHERE id = {$existing['id']}";
    } else {
        // 새로운 회원+기기 조합인 경우 신규 삽입
        $sql = "INSERT INTO pushmanager_subscriptions 
                (member_id, endpoint, p256dh_key, auth_key, user_agent, ip_address, is_active, created_at) 
                VALUES 
                ('{$member_id}', '{$endpoint}', '{$p256dh_key}', '{$auth_key}', '{$user_agent}', '{$ip_address}', 1, NOW())";
    }
    
    return sql_query($sql, false) ? true : false;
}

/**
 * 구독 쿼리 실행 및 결과 포맷팅 (공통 함수)
 */
function execute_subscription_query($sql) {
    $result = sql_query($sql);
    $subscriptions = array();
    
    while ($row = sql_fetch_array($result)) {
        $subscriptions[] = array(
            'endpoint' => $row['endpoint'],
            'keys' => array(
                'p256dh' => $row['p256dh_key'],
                'auth' => $row['auth_key']
            ),
            'member_id' => $row['member_id']
        );
    }
    
    return $subscriptions;
}

/**
 * 활성 구독자 조회 (통합 필터 지원)
 */
function get_active_subscriptions($filters = array()) {
    global $g5;
    
    $where_conditions = array();
    $joins = array();
    $use_alias = false;
    
    // 회원 ID 필터
    if (isset($filters['member_ids']) && is_array($filters['member_ids'])) {
        $member_ids = array_map('addslashes', $filters['member_ids']);
        $member_ids_condition = "member_id IN ('" . implode("','", $member_ids) . "')";
        $where_conditions[] = $member_ids_condition;
    }
    
    // 날짜 필터
    if (isset($filters['created_after'])) {
        $where_conditions[] = "created_at >= '" . addslashes($filters['created_after']) . "'";
    }
    
    // 레벨 필터
    if (isset($filters['level'])) {
        $level = intval($filters['level']);
        $joins[] = "INNER JOIN {$g5['member_table']} m ON ps.member_id = m.mb_id";
        $where_conditions[] = "m.mb_level = {$level}";
        $use_alias = true;
    }
    
    // 성별 필터
    if (isset($filters['gender'])) {
        $gender = addslashes($filters['gender']);
        if (empty($joins)) {
            $joins[] = "INNER JOIN {$g5['member_table']} m ON ps.member_id = m.mb_id";
        }
        $where_conditions[] = "m.mb_sex = '{$gender}'";
        $use_alias = true;
    }
    
    // 활성 구독자 조건 추가
    if ($use_alias) {
        $where_conditions[] = "ps.is_active = 1";
        $select_fields = "ps.endpoint, ps.p256dh_key, ps.auth_key, ps.member_id";
        $table_name = "pushmanager_subscriptions ps";
        $order_by = "ps.created_at DESC";
    } else {
        $where_conditions[] = "is_active = 1";
        $select_fields = "endpoint, p256dh_key, auth_key, member_id";
        $table_name = "pushmanager_subscriptions";
        $order_by = "created_at DESC";
    }
    
    $sql = "SELECT {$select_fields}
            FROM {$table_name} 
            " . implode(' ', $joins) . "
            WHERE " . implode(' AND ', $where_conditions) . " 
            ORDER BY {$order_by}";
    
    return execute_subscription_query($sql);
}

/**
 * 구독자 통계 조회 (한 번에 모든 통계 수집)
 */
function get_subscription_stats() {
    $stats = array(
        'total' => 0,
        'active' => 0,
        'inactive' => 0,
        'today' => 0,
        'this_week' => 0,
        'member' => 0,
        'guest' => 0
    );
    
    $sql = "SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active,
                SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as inactive,
                SUM(CASE WHEN DATE(created_at) = CURDATE() THEN 1 ELSE 0 END) as today,
                SUM(CASE WHEN WEEK(created_at) = WEEK(NOW()) AND YEAR(created_at) = YEAR(NOW()) THEN 1 ELSE 0 END) as this_week,
                SUM(CASE WHEN member_id IS NOT NULL THEN 1 ELSE 0 END) as member
            FROM pushmanager_subscriptions";
    
    $result = sql_fetch($sql);
    if ($result) {
        $stats['total'] = intval($result['total']);
        $stats['active'] = intval($result['active']);
        $stats['inactive'] = intval($result['inactive']);
        $stats['today'] = intval($result['today']);
        $stats['this_week'] = intval($result['this_week']);
        $stats['member'] = intval($result['member']);
        $stats['guest'] = intval($result['total'] - $result['member']);
    }
    
    return $stats;
}

/**
 * 활동 로그 기록
 */
function log_pushmanager_activity($type, $title = null, $message = null, $target_count = null, $success_count = null, $error_count = null, $error_message = null) {
    $values = array(
        'type' => "'" . addslashes($type) . "'",
        'title' => $title ? "'" . addslashes($title) . "'" : 'NULL',
        'message' => $message ? "'" . addslashes($message) . "'" : 'NULL',
        'target_count' => $target_count !== null ? intval($target_count) : 'NULL',
        'success_count' => $success_count !== null ? intval($success_count) : 'NULL',
        'error_count' => $error_count !== null ? intval($error_count) : 'NULL',
        'error_message' => $error_message ? "'" . addslashes($error_message) . "'" : 'NULL'
    );
    
    $sql = "INSERT INTO pushmanager_logs 
            (type, title, message, target_count, success_count, error_count, error_message, created_at) 
            VALUES 
            ({$values['type']}, {$values['title']}, {$values['message']}, {$values['target_count']}, {$values['success_count']}, {$values['error_count']}, {$values['error_message']}, NOW())";
    
    return sql_query($sql, false) ? true : false;
}

// ========================================
// 사용자 설정 관리 함수들
// ========================================

/**
 * 현재 사용자 식별자 가져오기
 */
function get_current_user_identifier() {
    global $member;
    
    if ($member['mb_id']) {
        return array(
            'type' => 'member',
            'id' => $member['mb_id'],
            'member_id' => $member['mb_id'],
            'session_id' => null
        );
    } else {
        $session_id = session_id();
        if (empty($session_id)) {
            session_start();
            $session_id = session_id();
        }
        
        return array(
            'type' => 'guest',
            'id' => $session_id,
            'member_id' => null,
            'session_id' => $session_id
        );
    }
}

/**
 * 회원의 게시판별 알림 설정 조회
 * @param string $member_id 회원 ID
 * @return array 설정된 게시판 목록 배열
 */
function get_board_notification_settings($member_id) {
    if (!$member_id) {
        return array();
    }
    
    $member_id = addslashes($member_id);
    
    // pushmanager_board_notifications 테이블에서 활성화된 게시판 알림 설정 조회
    $sql = "SELECT bo_table, is_active, created_at, updated_at 
            FROM pushmanager_board_notifications 
            WHERE member_id = '{$member_id}' 
            AND is_active = 1 
            ORDER BY created_at ASC";
    
    $result = sql_query($sql);
    $board_settings = array();
    
    if ($result) {
        while ($row = sql_fetch_array($result)) {
            $board_settings[] = array(
                'bo_table' => $row['bo_table'],
                'is_active' => intval($row['is_active']),
                'created_at' => $row['created_at'],
                'updated_at' => $row['updated_at']
            );
        }
    }
    
    return $board_settings;
}

/**
 * 사용자 알림 설정 조회
 */
function get_user_notification_settings($member_id = null, $session_id = null) {
    if (!$member_id && !$session_id) {
        $user = get_current_user_identifier();
        $member_id = $user['member_id'];
        $session_id = $user['session_id'];
    }
    
    $where_conditions = array();
    if ($member_id) {
        $where_conditions[] = "member_id = '" . addslashes($member_id) . "'";
    } elseif ($session_id) {
        $where_conditions[] = "session_id = '" . addslashes($session_id) . "'";
    } else {
        return array(
            'message_notification' => 1,
            'comment_notification' => 1,
            'inquiry_notification' => 1,
            'newpost_notification' => 1
        );

    }
    
    $sql = "SELECT message_notification, comment_notification, inquiry_notification, newpost_notification 
            FROM pushmanager_user_settings 
            WHERE " . implode(' AND ', $where_conditions) . " 
            LIMIT 1";
    
    $result = sql_fetch($sql);
    
    if ($result) {
        return array(
            'message_notification' => intval($result['message_notification']),
            'comment_notification' => intval($result['comment_notification']),
            'inquiry_notification' => intval($result['inquiry_notification']),
            'newpost_notification' => intval($result['newpost_notification'])
        );
    } else {
        return array(
            'message_notification' => 1,
            'comment_notification' => 1,
            'inquiry_notification' => 1,
            'newpost_notification' => 1
        );
    }
}

/**
 * 사용자 알림 설정 저장
 */
function save_user_notification_settings($settings, $member_id = null, $session_id = null) {
    if (!$member_id && !$session_id) {
        $user = get_current_user_identifier();
        $member_id = $user['member_id'];
        $session_id = $user['session_id'];
    }
    
    $message_notification = isset($settings['message_notification']) ? intval($settings['message_notification']) : 1;
    $comment_notification = isset($settings['comment_notification']) ? intval($settings['comment_notification']) : 1;
    $inquiry_notification = isset($settings['inquiry_notification']) ? intval($settings['inquiry_notification']) : 1;
    $newpost_notification = isset($settings['newpost_notification']) ? intval($settings['newpost_notification']) : 1;
    
    $where_conditions = array();
    if ($member_id) {
        $where_conditions[] = "member_id = '" . addslashes($member_id) . "'";
    } elseif ($session_id) {
        $where_conditions[] = "session_id = '" . addslashes($session_id) . "'";
    } else {
        return false;
    }
    
    $existing_sql = "SELECT id FROM pushmanager_user_settings WHERE " . implode(' AND ', $where_conditions);
    $existing = sql_fetch($existing_sql);
    
    if ($existing) {
        $sql = "UPDATE pushmanager_user_settings SET 
                message_notification = {$message_notification},
                comment_notification = {$comment_notification},
                inquiry_notification = {$inquiry_notification},
                newpost_notification = {$newpost_notification},
                updated_at = NOW()
                WHERE id = {$existing['id']}";
    } else {
        $member_id_value = $member_id ? "'" . addslashes($member_id) . "'" : 'NULL';
        $session_id_value = $session_id ? "'" . addslashes($session_id) . "'" : 'NULL';
        
        $sql = "INSERT INTO pushmanager_user_settings 
                (member_id, session_id, message_notification, comment_notification, inquiry_notification, newpost_notification, created_at) 
                VALUES 
                ({$member_id_value}, {$session_id_value}, {$message_notification}, {$comment_notification}, {$inquiry_notification}, {$newpost_notification}, NOW())";
    }
    
    return sql_query($sql, false) ? true : false;
}

/**
 * 사용자의 활성 구독 기기 목록 조회
 */
function get_user_devices($member_id = null, $session_id = null) {
    if (!$member_id && !$session_id) {
        $user = get_current_user_identifier();
        $member_id = $user['member_id'];
        $session_id = $user['session_id'];
    }
    
    $where_conditions = array();
    if ($member_id) {
        $where_conditions[] = "member_id = '" . addslashes($member_id) . "'";
    } else {
        $ip_address = get_real_ip();
        $where_conditions[] = "member_id IS NULL";
        $where_conditions[] = "ip_address = '" . addslashes($ip_address) . "'";
    }
    
    $sql = "SELECT id, endpoint, user_agent, ip_address, is_active, created_at, updated_at
            FROM pushmanager_subscriptions 
            WHERE " . implode(' AND ', $where_conditions) . "
            ORDER BY created_at DESC";
    
    $result = sql_query($sql);
    $devices = array();
    
    while ($row = sql_fetch_array($result)) {
        $device_info = parse_user_agent($row['user_agent']);
        
        $devices[] = array(
            'id' => $row['id'],
            'endpoint' => $row['endpoint'],
            'user_agent' => $row['user_agent'],
            'device_name' => $device_info['device_name'],
            'device_icon' => $device_info['device_icon'],
            'device_description' => $device_info['device_description'],
            'ip_address' => $row['ip_address'],
            'is_active' => intval($row['is_active']),
            'status' => $device_info['status'],
            'created_at' => $row['created_at'],
            'updated_at' => $row['updated_at']
        );
    }
    
    return $devices;
}

/**
 * User Agent 파싱하여 기기 정보 추출
 */
function parse_user_agent($user_agent) {
    $device_name = 'Unknown Device';
    $device_icon = '🖥️';
    $device_description = $user_agent;
    $status = '● 온라인';
    
    if (empty($user_agent)) {
        return array(
            'device_name' => $device_name,
            'device_icon' => $device_icon,
            'device_description' => 'User Agent 정보 없음',
            'status' => '● 오프라인'
        );
    }
    
    // Windows
    if (strpos($user_agent, 'Windows') !== false) {
        if (strpos($user_agent, 'Chrome') !== false) {
            $device_name = 'Chrome on Windows';
            $device_icon = '💻';
        } elseif (strpos($user_agent, 'Firefox') !== false) {
            $device_name = 'Firefox on Windows';
            $device_icon = '💻';
        } elseif (strpos($user_agent, 'Edge') !== false) {
            $device_name = 'Edge on Windows';
            $device_icon = '💻';
        } else {
            $device_name = 'Windows Device';
            $device_icon = '💻';
        }
    }
    // macOS
    elseif (strpos($user_agent, 'Macintosh') !== false || strpos($user_agent, 'Mac OS X') !== false) {
        if (strpos($user_agent, 'Chrome') !== false) {
            $device_name = 'Chrome on macOS';
            $device_icon = '🖥️';
        } elseif (strpos($user_agent, 'Safari') !== false && strpos($user_agent, 'Chrome') === false) {
            $device_name = 'Safari on macOS';
            $device_icon = '🖥️';
        } elseif (strpos($user_agent, 'Firefox') !== false) {
            $device_name = 'Firefox on macOS';
            $device_icon = '🖥️';
        } else {
            $device_name = 'macOS Device';
            $device_icon = '🖥️';
        }
    }
    // iPhone
    elseif (strpos($user_agent, 'iPhone') !== false) {
        if (strpos($user_agent, 'Safari') !== false) {
            $device_name = 'Safari on iPhone';
            $device_icon = '📱';
        } elseif (strpos($user_agent, 'Chrome') !== false) {
            $device_name = 'Chrome on iPhone';
            $device_icon = '📱';
        } else {
            $device_name = 'iPhone';
            $device_icon = '📱';
        }
    }
    // Android
    elseif (strpos($user_agent, 'Android') !== false) {
        if (strpos($user_agent, 'Chrome') !== false) {
            $device_name = 'Chrome on Android';
            $device_icon = '🤖';
        } elseif (strpos($user_agent, 'Firefox') !== false) {
            $device_name = 'Firefox on Android';
            $device_icon = '🤖';
        } else {
            $device_name = 'Android Device';
            $device_icon = '🤖';
        }
    }
    // iPad
    elseif (strpos($user_agent, 'iPad') !== false) {
        $device_name = 'Safari on iPad';
        $device_icon = '📱';
    }
    // Linux
    elseif (strpos($user_agent, 'Linux') !== false) {
        $device_name = 'Linux Device';
        $device_icon = '🐧';
    }
    
    if (strlen($user_agent) > 80) {
        $device_description = substr($user_agent, 0, 77) . '...';
    }
    
    return array(
        'device_name' => $device_name,
        'device_icon' => $device_icon,
        'device_description' => $device_description,
        'status' => $status
    );
}

/**
 * 특정 기기의 알림 활성화/비활성화
 */
function toggle_device_notification($subscription_id, $is_active) {
    $subscription_id = intval($subscription_id);
    $is_active = $is_active ? 1 : 0;
    
    $sql = "UPDATE pushmanager_subscriptions SET 
            is_active = {$is_active},
            updated_at = NOW()
            WHERE id = {$subscription_id}";
    
    return sql_query($sql, false) ? true : false;
}

// ========================================
// 게시판 알림 관리 함수들
// ========================================

/**
 * 특정 게시판의 알림 설정된 사용자 목록 조회
 */
function get_board_notification_admins($bo_table) {
    $bo_table = addslashes($bo_table);
    
    $sql = "SELECT DISTINCT bn.member_id 
            FROM pushmanager_board_notifications bn 
            INNER JOIN pushmanager_subscriptions ps ON bn.member_id = ps.member_id 
            WHERE bn.bo_table = '{$bo_table}' 
            AND bn.is_active = 1 
            AND ps.is_active = 1";
    
    $result = sql_query($sql);
    $users = array();
    
    while ($row = sql_fetch_array($result)) {
        $users[] = $row['member_id'];
    }
    
    return $users;
}

/**
 * 게시판 정보 조회 (간단 버전)
 */
function get_board_info($bo_table) {
    global $g5;
    
    $bo_table = addslashes($bo_table);
    
    $sql = "SELECT bo_table, bo_subject, bo_admin, bo_list_level, bo_write_level 
            FROM {$g5['board_table']} 
            WHERE bo_table = '{$bo_table}'";
    
    return sql_fetch($sql);
}

// ========================================
// AJAX 처리 함수들
// ========================================

/**
 * 공통 JSON 응답 처리
 */
function send_json_response($success, $message, $data = null, $http_code = 200) {
    header('Content-Type: application/json; charset=utf-8');
    
    if ($http_code !== 200) {
        http_response_code($http_code);
    }
    
    $response = array(
        'success' => $success,
        'message' => $message
    );
    
    if ($data !== null) {
        $response['data'] = $data;
    }
    
    echo json_encode($response);
    exit;
}

/**
 * 푸시 구독 등록 (AJAX 용)
 */
function handle_push_subscription() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        send_json_response(false, 'POST 메서드만 허용됩니다.', null, 405);
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['subscription'])) {
        send_json_response(false, '구독 정보가 필요합니다.', null, 400);
    }
    
    $subscription = $input['subscription'];
    
    if (!isset($subscription['endpoint']) || !isset($subscription['keys']['p256dh']) || !isset($subscription['keys']['auth'])) {
        send_json_response(false, '구독 정보가 올바르지 않습니다.', null, 400);
    }
    
    if (save_push_subscription($subscription)) {
        send_json_response(true, '구독이 등록되었습니다.');
    } else {
        send_json_response(false, '구독 등록에 실패했습니다.', null, 500);
    }
}

/**
 * 푸시 구독 해제 (AJAX 용)
 */
function handle_push_unsubscription() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        send_json_response(false, 'POST 메서드만 허용됩니다.', null, 405);
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['endpoint'])) {
        send_json_response(false, '엔드포인트가 필요합니다.', null, 400);
    }
    
    if (unsubscribe_push($input['endpoint'])) {
        send_json_response(true, '구독이 해제되었습니다.');
    } else {
        send_json_response(false, '구독 해제에 실패했습니다.', null, 500);
    }
}

/**
 * 사용자 기기 목록 조회
 */
function handle_get_user_devices() {
    try {
        $devices = get_user_devices();
        
        echo json_encode(array(
            'success' => true,
            'data' => array(
                'devices' => $devices,
                'count' => count($devices)
            )
        ));
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(array(
            'success' => false, 
            'message' => '기기 목록 조회 중 오류가 발생했습니다: ' . $e->getMessage()
        ));
    }
    
    exit;
}

/**
 * VAPID 공개키 가져오기 (AJAX 용)
 */
function get_vapid_public_key() {
    $config = get_pushmanager_config();
    
    if (!$config || !$config['vapid_public_key']) {
        send_json_response(false, 'VAPID 공개키가 설정되지 않았습니다.', null, 404);
    }
    
    send_json_response(true, '성공', array('vapid_public_key' => $config['vapid_public_key']));
}

/**
 * 알림 설정 저장 AJAX 처리
 */
function handle_save_notification_settings() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        send_json_response(false, 'POST 메서드만 허용됩니다.', null, 405);
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        $input = $_POST;
    }
    
    $settings = array(
        'message_notification' => isset($input['message_notification']) ? intval($input['message_notification']) : 0,
        'comment_notification' => isset($input['comment_notification']) ? intval($input['comment_notification']) : 0,
        'inquiry_notification' => isset($input['inquiry_notification']) ? intval($input['inquiry_notification']) : 0,
        'newpost_notification' => isset($input['newpost_notification']) ? intval($input['newpost_notification']) : 0
    );
    
    $device_settings = array();
    if (isset($input['devices']) && is_array($input['devices'])) {
        foreach ($input['devices'] as $device) {
            if (isset($device['id']) && isset($device['enabled'])) {
                $device_settings[] = array(
                    'id' => intval($device['id']),
                    'enabled' => intval($device['enabled'])
                );
            }
        }
    }
    
    $settings_saved = save_user_notification_settings($settings);
    
    $devices_saved = true;
    foreach ($device_settings as $device) {
        if (!toggle_device_notification($device['id'], $device['enabled'])) {
            $devices_saved = false;
        }
    }
    
    if ($settings_saved && $devices_saved) {
        send_json_response(true, '알림 설정이 저장되었습니다.');
    } else {
        send_json_response(false, '설정 저장 중 오류가 발생했습니다.', null, 500);
    }
}

/**
 * 현재 알림 설정 조회 AJAX 처리
 */
function handle_get_notification_settings() {
    $settings = get_user_notification_settings();
    $devices = get_user_devices();
    
    send_json_response(true, '성공', array(
        'settings' => $settings,
        'devices' => $devices
    ));
}

/**
 * 기기별 알림 토글 AJAX 처리
 */
function handle_toggle_device_notification() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        send_json_response(false, 'POST 메서드만 허용됩니다.', null, 405);
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['device_id']) || !isset($input['is_active'])) {
        send_json_response(false, '기기 ID와 활성화 상태가 필요합니다.', null, 400);
    }
    
    $device_id = intval($input['device_id']);
    $is_active = intval($input['is_active']);
    
    if (toggle_device_notification($device_id, $is_active)) {
        send_json_response(true, '기기 알림 설정이 변경되었습니다.');
    } else {
        send_json_response(false, '기기 알림 설정 변경에 실패했습니다.', null, 500);
    }
}

// ========================================
// 통계 및 로그 함수들
// ========================================

/**
 * 최근 푸시 로그 조회
 */
function get_recent_push_logs($limit = 10) {
    $limit = intval($limit);
    
    $sql = "SELECT * FROM pushmanager_logs 
            ORDER BY created_at DESC 
            LIMIT {$limit}";
    
    $result = sql_query($sql);
    $logs = array();
    
    while ($row = sql_fetch_array($result)) {
        $logs[] = $row;
    }
    
    return $logs;
}

/**
 * 푸시 설정 상태 확인
 */
function check_pushmanager_status() {
    $status = array(
        'installed' => false,
        'configured' => false,
        'api_key_issued' => false,
        'active_subscriptions' => 0
    );
    
    $sql = "SHOW TABLES LIKE 'pushmanager_config'";
    $result = sql_query($sql, false);
    if (sql_num_rows($result) > 0) {
        $status['installed'] = true;
        
        $config = get_pushmanager_config();
        if ($config && $config['site_name'] && $config['api_server_url']) {
            $status['configured'] = true;
            
            if ($config['api_key']) {
                $status['api_key_issued'] = true;
                
                $stats = get_subscription_stats();
                $status['active_subscriptions'] = $stats['active'];
            }
        }
    }
    
    return $status;
}

// ========================================
// 하위 호환성 함수들 (래퍼 함수)
// ========================================

function get_push_icon_url() {
    return get_push_asset_url('icon');
}

function get_push_badge_url() {
    return get_push_asset_url('badge');
}

function get_subscriptions_by_level($level) {
    return get_active_subscriptions(array('level' => $level));
}

function get_subscriptions_by_gender($gender) {
    return get_active_subscriptions(array('gender' => $gender));
}

function get_subscriptions_by_member_ids($member_ids) {
    return get_active_subscriptions(array('member_ids' => $member_ids));
}

function get_all_active_subscriptions() {
    return get_active_subscriptions();
}

// ========================================
// 유지보수 및 관리 함수들
// ========================================

/**
 * 회원 탈퇴할 때 알림 설정 정리
 */
function cleanup_member_notification_data($member_id) {
    $member_id = addslashes($member_id);
    
    $sql1 = "DELETE FROM pushmanager_user_settings WHERE member_id = '{$member_id}'";
    $sql2 = "UPDATE pushmanager_subscriptions SET is_active = 0, member_id = NULL WHERE member_id = '{$member_id}'";
    $sql3 = "DELETE FROM pushmanager_push_history WHERE recipient_id = '{$member_id}' OR sender_id = '{$member_id}'";
    
    $result1 = sql_query($sql1, false);
    $result2 = sql_query($sql2, false);
    $result3 = sql_query($sql3, false);
    
    return $result1 && $result2 && $result3;
}

/**
 * 게시판 삭제시 관련 알림 설정 정리
 */
function cleanup_board_notifications($bo_table) {
    $bo_table = addslashes($bo_table);
    
    $sql = "DELETE FROM pushmanager_board_notifications WHERE bo_table = '{$bo_table}'";
    return sql_query($sql, false) ? true : false;
}

/**
 * 푸시 매니저 초기화 (테스트용)
 */
function reset_pushmanager() {
    $tables = array('pushmanager_logs', 'pushmanager_subscriptions', 'pushmanager_config');
    
    foreach ($tables as $table) {
        $sql = "DROP TABLE IF EXISTS {$table}";
        if (!sql_query($sql, false)) {
            return false;
        }
    }
    
    return true;
}

// ========================================
// 테스트 및 디버깅 함수들
// ========================================

/**
 * 새글 알림 테스트 발송 (훅에서 포맷팅된 메시지로)
 */
function test_board_notification($bo_table, $test_member_id) {
    $board_info = get_board_info($bo_table);
    
    // 간단한 테스트 메시지 (실제로는 훅에서 포맷팅된 메시지 사용)
    $title = "[{$board_info['bo_subject']}] 테스트 알림";
    $message = "새글 알림 테스트입니다.";
    
    $options = array(
        'notification_type' => 'system',
        'bo_table' => $bo_table,
        'wr_id' => 999999
    );
    
    return send_push_unified(array($test_member_id), $title, $message, $options);
}

/**
 * API 응답 디버깅을 위한 헬퍼 함수
 */
function debug_api_response($api_response) {
    if (defined('G5_DEBUG') && G5_DEBUG) {        
        if (isset($api_response['data'])) {
            // 디버그 정보 로깅
        }
    }
}

/**
 * 푸시 알림 발송 테스트 함수
 */
function test_push_notification($title = '테스트 알림', $message = '테스트 메시지입니다.') {
    global $member;
    
    if (!$member['mb_id']) {
        return array('success' => false, 'message' => '로그인이 필요합니다.');
    }
    
    $options = array(
        'notification_type' => 'system',
        'require_interaction' => false,
        'tag' => 'test_' . time()
    );
    
    return send_push_unified(array($member['mb_id']), $title, $message, $options);
}

/**
 * 알림 발송 통계
 */
function get_notification_stats($notification_type = null) {
    $where_condition = '';
    if ($notification_type && is_valid_notification_type($notification_type)) {
        $where_condition = "WHERE type = '" . addslashes($notification_type) . "'";
    }
    
    $sql = "SELECT 
                type,
                COUNT(*) as total_count,
                SUM(target_count) as total_targets,
                SUM(success_count) as total_success,
                SUM(error_count) as total_errors,
                DATE(created_at) as date
            FROM pushmanager_logs 
            {$where_condition}
            GROUP BY type, DATE(created_at)
            ORDER BY created_at DESC
            LIMIT 30";
    
    $result = sql_query($sql);
    $stats = array();
    
    while ($row = sql_fetch_array($result)) {
        $stats[] = array(
            'type' => $row['type'],
            'date' => $row['date'],
            'total_count' => intval($row['total_count']),
            'total_targets' => intval($row['total_targets']),
            'total_success' => intval($row['total_success']),
            'total_errors' => intval($row['total_errors']),
            'success_rate' => $row['total_targets'] > 0 ? round(($row['total_success'] / $row['total_targets']) * 100, 2) : 0
        );
    }
    
    return $stats;
}

/**
 * 알림 타입 유효성 검사
 */
function is_valid_notification_type($notification_type) {
    $valid_types = array('comment', 'message', 'inquiry', 'system', 'newpost');
    return in_array($notification_type, $valid_types);
}

/**
 * 여러 회원의 알림 설정 일괄 확인
 */
function check_notification_settings($member_ids, $notification_type) {
    $result = array(
        'enabled' => array(),
        'disabled' => array(),
        'notification_type' => $notification_type
    );
    
    foreach ($member_ids as $member_id) {
        if (is_notification_enabled($member_id, $notification_type)) {
            $result['enabled'][] = $member_id;
        } else {
            $result['disabled'][] = $member_id;
        }
    }
    
    return $result;
}

/**
 * 푸시 설정 검증 함수
 */
function validate_push_config($config) {
    $errors = array();
    
    if (empty($config['site_name'])) {
        $errors[] = '사이트명이 필요합니다.';
    }
    
    if (empty($config['site_url']) || !filter_var($config['site_url'], FILTER_VALIDATE_URL)) {
        $errors[] = '올바른 사이트 URL이 필요합니다.';
    }
    
    if (empty($config['admin_email']) || !filter_var($config['admin_email'], FILTER_VALIDATE_EMAIL)) {
        $errors[] = '올바른 관리자 이메일이 필요합니다.';
    }
    
    if (empty($config['api_server_url']) || !filter_var($config['api_server_url'], FILTER_VALIDATE_URL)) {
        $errors[] = '올바른 API 서버 URL이 필요합니다.';
    }
    
    return $errors;
}

/**
 * 구독 정보 검증 함수
 */
function validate_subscription_data($subscription_data) {
    if (!isset($subscription_data['endpoint']) || empty($subscription_data['endpoint'])) {
        return false;
    }
    
    if (!isset($subscription_data['keys']['p256dh']) || empty($subscription_data['keys']['p256dh'])) {
        return false;
    }
    
    if (!isset($subscription_data['keys']['auth']) || empty($subscription_data['keys']['auth'])) {
        return false;
    }
    
    return true;
}

/**
 * 푸시 매니저 상태 진단
 */
function diagnose_pushmanager() {
    $diagnosis = array(
        'config' => array(),
        'database' => array(),
        'api' => array(),
        'subscriptions' => array(),
        'overall_status' => 'unknown'
    );
    
    // 설정 확인
    $config = get_pushmanager_config();
    if ($config) {
        $diagnosis['config']['status'] = 'ok';
        $diagnosis['config']['message'] = '설정이 정상적으로 로드되었습니다.';
        $validation_errors = validate_push_config($config);
        if (!empty($validation_errors)) {
            $diagnosis['config']['status'] = 'warning';
            $diagnosis['config']['errors'] = $validation_errors;
        }
    } else {
        $diagnosis['config']['status'] = 'error';
        $diagnosis['config']['message'] = '설정을 찾을 수 없습니다.';
    }
    
    // 데이터베이스 확인
    $required_tables = array('pushmanager_config', 'pushmanager_subscriptions', 'pushmanager_logs', 'pushmanager_user_settings', 'pushmanager_board_notifications');
    $missing_tables = array();
    
    foreach ($required_tables as $table) {
        $sql = "SHOW TABLES LIKE '{$table}'";
        $result = sql_query($sql, false);
        if (!$result || sql_num_rows($result) == 0) {
            $missing_tables[] = $table;
        }
    }
    
    if (empty($missing_tables)) {
        $diagnosis['database']['status'] = 'ok';
        $diagnosis['database']['message'] = '모든 필수 테이블이 존재합니다.';
    } else {
        $diagnosis['database']['status'] = 'error';
        $diagnosis['database']['message'] = '누락된 테이블: ' . implode(', ', $missing_tables);
        $diagnosis['database']['missing_tables'] = $missing_tables;
    }
    
    // API 연결 확인
    if ($config && $config['api_server_url']) {
        $api_test = test_api_connection($config);
        if ($api_test['success']) {
            $diagnosis['api']['status'] = 'ok';
            $diagnosis['api']['message'] = 'API 서버 연결이 정상입니다.';
        } else {
            $diagnosis['api']['status'] = 'error';
            $diagnosis['api']['message'] = 'API 서버 연결 실패: ' . $api_test['message'];
        }
    } else {
        $diagnosis['api']['status'] = 'warning';
        $diagnosis['api']['message'] = 'API 서버 URL이 설정되지 않았습니다.';
    }
    
    // 구독자 상태 확인
    $stats = get_subscription_stats();
    $diagnosis['subscriptions']['stats'] = $stats;
    
    if ($stats['active'] > 0) {
        $diagnosis['subscriptions']['status'] = 'ok';
        $diagnosis['subscriptions']['message'] = "활성 구독자 {$stats['active']}명";
    } else {
        $diagnosis['subscriptions']['status'] = 'warning';
        $diagnosis['subscriptions']['message'] = '활성 구독자가 없습니다.';
    }
    
    // 전체 상태 판단
    $error_count = 0;
    $warning_count = 0;
    
    foreach ($diagnosis as $key => $section) {
        if (is_array($section) && isset($section['status'])) {
            if ($section['status'] === 'error') {
                $error_count++;
            } elseif ($section['status'] === 'warning') {
                $warning_count++;
            }
        }
    }
    
    if ($error_count > 0) {
        $diagnosis['overall_status'] = 'error';
    } elseif ($warning_count > 0) {
        $diagnosis['overall_status'] = 'warning';
    } else {
        $diagnosis['overall_status'] = 'ok';
    }
    
    return $diagnosis;
}

// ========================================
// 1:1 문의 알림 관련 함수
// ========================================

/**
 * 1:1 문의 답변 알림 발송 함수 (간단 버전)
 */
function send_inquiry_reply_notification($member_id, $qa_subject, $qa_id) {
    // 1:1 문의는 별도 포맷이 없으므로 기본 메시지 사용
    $title = "1:1 문의 답변 알림";
    $message = "문의하신 \"{$qa_subject}\"에 대한 답변이 등록되었습니다.";
    
    $options = array(
        'notification_type' => 'inquiry',
        'click_action' => get_site_origin() . "/bbs/qa_view.php?qa_id={$qa_id}",
        'require_interaction' => true,
        'tag' => 'inquiry_' . $qa_id,
        'qa_id' => $qa_id
    );
    
    return send_push_unified(array($member_id), $title, $message, $options);
}

// ========================================
// 기타 유틸리티 함수들
// ========================================

/**
 * 시간대별 알림 발송 제한 확인
 */
function is_notification_time_allowed() {
    $current_hour = intval(date('H'));
    
    // 밤 10시 ~ 아침 8시는 알림 발송 제한 (선택사항)
    if ($current_hour >= 22 || $current_hour < 8) {
        return false;
    }
    
    return true;
}

/**
 * 발송 빈도 제한 확인 (스팸 방지)
 */
function check_send_rate_limit($member_id, $notification_type, $limit_minutes = 5) {
    if (!$member_id) {
        return true; // 비회원은 제한하지 않음
    }
    
    $sql = "SELECT COUNT(*) as count FROM pushmanager_logs 
            WHERE type = '" . addslashes($notification_type) . "' 
            AND created_at >= DATE_SUB(NOW(), INTERVAL {$limit_minutes} MINUTE)";
    
    $result = sql_fetch($sql);
    
    // 5분 내에 같은 타입 알림이 3개 이상이면 제한
    return intval($result['count']) < 3;
}

/**
 * 메시지 길이 확인 및 조정
 */
function validate_push_message($title, $message) {
    $errors = array();
    
    if (empty($title)) {
        $errors[] = '제목이 필요합니다.';
    } elseif (mb_strlen($title, 'UTF-8') > 100) {
        $errors[] = '제목은 100자 이하여야 합니다.';
    }
    
    if (empty($message)) {
        $errors[] = '메시지가 필요합니다.';
    } elseif (mb_strlen($message, 'UTF-8') > 500) {
        $errors[] = '메시지는 500자 이하여야 합니다.';
    }
    
    return $errors;
}

/**
 * 푸시 알림 우선순위 설정
 */
function get_notification_priority($notification_type) {
    $priorities = array(
        'system' => 'high',
        'inquiry' => 'high',
        'message' => 'normal',
        'comment' => 'normal'
    );
    
    return isset($priorities[$notification_type]) ? $priorities[$notification_type] : 'normal';
}

/**
 * 배치 발송을 위한 구독자 그룹 분할
 */
function chunk_subscriptions($subscriptions, $chunk_size = 1000) {
    return array_chunk($subscriptions, $chunk_size);
}

/**
 * 발송 결과 요약 생성
 */
function generate_send_summary($results) {
    $total_sent = 0;
    $total_failed = 0;
    $total_targets = 0;
    
    foreach ($results as $result) {
        if (isset($result['data'])) {
            $total_sent += $result['data']['sent'] ?? 0;
            $total_failed += $result['data']['failed'] ?? 0;
            $total_targets += $result['data']['total'] ?? 0;
        }
    }
    
    return array(
        'total_targets' => $total_targets,
        'total_sent' => $total_sent,
        'total_failed' => $total_failed,
        'success_rate' => $total_targets > 0 ? round(($total_sent / $total_targets) * 100, 2) : 0,
        'batch_count' => count($results)
    );
}

// ========================================
// 정리 및 종료
// ========================================

/**
* 오래된 로그 정리 (30일 이상)
*/
function cleanup_old_logs($days = 30) {
   $sql = "DELETE FROM pushmanager_logs 
           WHERE created_at < DATE_SUB(NOW(), INTERVAL {$days} DAY)";
   
   return sql_query($sql, false) ? true : false;
}

/**
* 비활성 구독 정리 (90일 이상 미사용)
*/
function cleanup_inactive_subscriptions($days = 90) {
   $sql = "DELETE FROM pushmanager_subscriptions 
           WHERE is_active = 0 
           AND updated_at < DATE_SUB(NOW(), INTERVAL {$days} DAY)";
   
   return sql_query($sql, false) ? true : false;
}

// ========================================
// 푸시 발송 이력 관리 함수들
// ========================================

/**
 * 푸시 발송 이력 저장
 */
function save_push_history($recipients, $title, $message, $options = array()) {
    $notification_type = $options['notification_type'] ?? 'system';
    $sender_id = $options['sender_id'] ?? null;
    $related_table = $options['bo_table'] ?? null;
    $related_id = $options['wr_id'] ?? $options['me_id'] ?? $options['qa_id'] ?? $options['comment_id'] ?? null;
    
    $member_ids = array();
    
    // 수신자 타입에 따라 처리
    if (is_array($recipients)) {
        // 구독 정보 배열인 경우 (endpoint, keys 포함)
        if (!empty($recipients) && isset($recipients[0]['endpoint'])) {
            foreach ($recipients as $subscription) {
                if (isset($subscription['member_id']) && $subscription['member_id']) {
                    $member_ids[] = $subscription['member_id'];
                }
            }
        }
        // 단순 회원 ID 배열인 경우
        else {
            foreach ($recipients as $recipient) {
                if (is_string($recipient) && !empty($recipient)) {
                    $member_ids[] = $recipient;
                }
            }
        }
    }
    // 단일 회원 ID인 경우
    elseif (is_string($recipients) && !empty($recipients)) {
        $member_ids = array($recipients);
    }
    
    // 빈 배열인 경우 (전체 발송) 실제 발송된 구독자들의 member_id 조회
    if (empty($member_ids)) {
        $subscriptions = get_active_subscriptions();
        foreach ($subscriptions as $subscription) {
            if ($subscription['member_id']) {
                $member_ids[] = $subscription['member_id'];
            }
        }
    }
    
    // 중복 제거
    $member_ids = array_unique($member_ids);
    
    $success_count = 0;
    
    foreach ($member_ids as $recipient_id) {
        if (!$recipient_id) continue; // 비회원 제외
        
        $values = array(
            'recipient_id' => "'" . addslashes($recipient_id) . "'",
            'sender_id' => $sender_id ? "'" . addslashes($sender_id) . "'" : 'NULL',
            'notification_type' => "'" . addslashes($notification_type) . "'",
            'push_content' => "'" . addslashes($message) . "'",
            'related_table' => $related_table ? "'" . addslashes($related_table) . "'" : 'NULL',
            'related_id' => $related_id ? intval($related_id) : 'NULL'
        );
        
        $sql = "INSERT INTO pushmanager_push_history 
                (recipient_id, sender_id, notification_type, push_content, related_table, related_id, sent_at) 
                VALUES 
                ({$values['recipient_id']}, {$values['sender_id']}, {$values['notification_type']}, {$values['push_content']}, {$values['related_table']}, {$values['related_id']}, NOW())";
        
        if (sql_query($sql, false)) {
            $success_count++;
        }
    }
    
    return $success_count;
}

/**
 * 사용자의 푸시 알림 이력 조회
 */
function get_user_push_history($member_id, $limit = 20, $offset = 0) {
    $member_id = addslashes($member_id);
    $limit = intval($limit);
    $offset = intval($offset);
    
    $sql = "SELECT ph.*, m.mb_nick, m.mb_name 
            FROM pushmanager_push_history ph 
            LEFT JOIN " . $GLOBALS['g5']['member_table'] . " m ON ph.sender_id = m.mb_id 
            WHERE ph.recipient_id = '{$member_id}' 
            ORDER BY ph.sent_at DESC 
            LIMIT {$offset}, {$limit}";
    
    $result = sql_query($sql);
    $history = array();
    
    while ($row = sql_fetch_array($result)) {
        $history[] = array(
            'id' => $row['id'],
            'sender_name' => $row['mb_nick'] ?: $row['mb_name'] ?: '시스템',
            'notification_type' => $row['notification_type'],
            'push_content' => $row['push_content'],
            'related_table' => $row['related_table'],
            'related_id' => $row['related_id'],
            'is_read' => intval($row['is_read']),
            'sent_at' => $row['sent_at'],
            'read_at' => $row['read_at']
        );
    }
    
    return $history;
}

/**
 * 사용자의 읽지 않은 알림 개수 조회
 */
function get_unread_push_count($member_id) {
    $member_id = addslashes($member_id);
    
    $sql = "SELECT COUNT(*) as count FROM pushmanager_push_history 
            WHERE recipient_id = '{$member_id}' AND is_read = 0";
    
    $result = sql_fetch($sql);
    return intval($result['count']);
}

/**
 * 푸시 알림을 읽음으로 표시
 */
function mark_push_as_read($history_id, $member_id = null) {
    $history_id = intval($history_id);
    
    $where_condition = "id = {$history_id}";
    if ($member_id) {
        $member_id = addslashes($member_id);
        $where_condition .= " AND recipient_id = '{$member_id}'";
    }
    
    $sql = "UPDATE pushmanager_push_history 
            SET is_read = 1, read_at = NOW() 
            WHERE {$where_condition} AND is_read = 0";
    
    return sql_query($sql, false) ? true : false;
}

/**
 * 여러 푸시 알림을 일괄 읽음 처리
 */
function mark_multiple_push_as_read($history_ids, $member_id = null) {
    if (empty($history_ids) || !is_array($history_ids)) {
        return false;
    }
    
    $ids = array_map('intval', $history_ids);
    $ids_str = implode(',', $ids);
    
    $where_condition = "id IN ({$ids_str})";
    if ($member_id) {
        $member_id = addslashes($member_id);
        $where_condition .= " AND recipient_id = '{$member_id}'";
    }
    
    $sql = "UPDATE pushmanager_push_history 
            SET is_read = 1, read_at = NOW() 
            WHERE {$where_condition} AND is_read = 0";
    
    return sql_query($sql, false) ? true : false;
}

/**
 * 사용자의 모든 알림을 읽음 처리
 */
function mark_all_push_as_read($member_id) {
    $member_id = addslashes($member_id);
    
    $sql = "UPDATE pushmanager_push_history 
            SET is_read = 1, read_at = NOW() 
            WHERE recipient_id = '{$member_id}' AND is_read = 0";
    
    return sql_query($sql, false) ? true : false;
}

/**
 * 푸시 알림 이력 삭제 (사용자별)
 */
function delete_push_history($history_id, $member_id = null) {
    $history_id = intval($history_id);
    
    $where_condition = "id = {$history_id}";
    if ($member_id) {
        $member_id = addslashes($member_id);
        $where_condition .= " AND recipient_id = '{$member_id}'";
    }
    
    $sql = "DELETE FROM pushmanager_push_history WHERE {$where_condition}";
    
    return sql_query($sql, false) ? true : false;
}

/**
 * 오래된 푸시 이력 정리 (90일 이상)
 */
function cleanup_old_push_history($days = 90) {
    $days = intval($days);
    
    $sql = "DELETE FROM pushmanager_push_history 
            WHERE sent_at < DATE_SUB(NOW(), INTERVAL {$days} DAY)";
    
    return sql_query($sql, false) ? true : false;
}

/**
* 푸시매니저 라이브러리 버전 정보
*/
function get_pushmanager_version() {
   return array(
       'version' => '1.0.0',
       'build_date' => '2024-12-19',
       'features' => array(
           'message_formatting',
           'user_notifications_settings',
           'board_notifications',
           'pwa_support',
           'icon_management',
           'api_integration'
       )
   );
}
?>