<?php
if (!defined('_GNUBOARD_')) exit;

if (defined('PUSHMANAGER_HOOK_LOADED')) {
    return;
}
define('PUSHMANAGER_HOOK_LOADED', true);

class PushManagerHook {
    private $debug_mode = false;
    private $log_file = '';
    private $executed_actions = array();
    private $new_posts = array();
    private static $global_initialized = false;
    private $config = null;
    
    public static function getInstance() {
        static $instance = null;
        if (null === $instance) {
            $instance = new self();
        }
        return $instance;
    }

    public function __construct() {
        if (self::$global_initialized) return;
        self::$global_initialized = true;

        $this->load_config();
        $this->add_hooks();
    }

    private function load_config() {
        try {
            if (function_exists('get_pushmanager_config')) {
                $this->config = get_pushmanager_config();
            } else {
                // 직접 데이터베이스에서 설정 조회
                $sql = "SELECT * FROM pushmanager_config ORDER BY id DESC LIMIT 1";
                $result = sql_query($sql, false);
                if ($result && sql_num_rows($result) > 0) {
                    $this->config = sql_fetch_array($result);
                }
            }
        } catch (Exception $e) {
            $this->config = null;
        }
    }

    private function get_message_format($type) {
        if (!$this->config) {
            // 기본 포맷 반환
            $defaults = array(
                'newpost_title' => '[{BOARD_NAME}] 새글 알림',
                'newpost_message' => '{SENDER_NICKNAME}님이 새글을 작성했습니다.\n제목: {POST_TITLE}',
                'comment_title' => '댓글 알림',
                'comment_message' => '{SENDER_NICKNAME}님이 댓글을 남겼습니다.\n원글: {POST_TITLE}\n댓글: {COMMENT_CONTENT}',
                'memo_title' => '쪽지 알림',
                'memo_message' => '{SENDER_NICKNAME}님으로부터 쪽지가 도착했습니다.\n내용: {MEMO_CONTENT}'
            );
            return isset($defaults[$type]) ? $defaults[$type] : '';
        }
        
        return isset($this->config[$type]) ? $this->config[$type] : '';
    }

    private function replace_variables($text, $variables) {
        if (empty($text) || empty($variables)) {
            return $text;
        }
        
        // 변수 치환
        foreach ($variables as $key => $value) {
            $text = str_replace('{' . $key . '}', $value, $text);
        }
        
        // 줄바꿈 복원 (마지막에 한 번만)
        $text = str_replace('\\n', "\n", $text);
        
        return $text;
    }

    private function get_board_name($bo_table) {
        try {
            $sql = "SELECT bo_subject FROM " . $GLOBALS['g5']['board_table'] . " WHERE bo_table = '" . addslashes($bo_table) . "'";
            $result = sql_fetch($sql);
            return $result ? $result['bo_subject'] : $bo_table;
        } catch (Exception $e) {
            return $bo_table;
        }
    }

    private function get_site_name() {
        if ($this->config && isset($this->config['site_name'])) {
            return $this->config['site_name'];
        }
        
        // 그누보드 기본 설정에서 사이트명 가져오기
        global $config;
        return isset($config['cf_title']) ? $config['cf_title'] : '사이트';
    }

    private function truncate_text($text, $max_length = 100) {
        if (mb_strlen($text, 'UTF-8') <= $max_length) {
            return $text;
        }
        
        return mb_substr($text, 0, $max_length, 'UTF-8') . '...';
    }

    private function load_pushmanager_library() {
        if (function_exists('send_push_unified')) {
            return true;
        }
        
        $push_lib_path = G5_LIB_PATH.'/pushmanager.lib.php';
        if (file_exists($push_lib_path)) {
            include_once($push_lib_path);
            $loaded = function_exists('send_push_unified');
            return $loaded;
        }
        
        return false;
    }

    public function add_hooks() {
        add_event('write_update_after', array($this, 'handle_write_after'), 10, 5);
        add_event('board_new_post', array($this, 'handle_new_post_direct'), 10, 3);
        
        add_event('comment_update_after', array($this, 'handle_comment'), 10, 5);
        add_event('comment_insert_after', array($this, 'handle_comment_insert'), 10, 5);
        
        add_event('memo_form_update_after', array($this, 'handle_memo'), 10, 5);
        add_event('memo_insert_after', array($this, 'handle_memo_direct'), 10, 5);
    }

    public function handle_write_after($board, $wr_id, $w, $qstr, $redirect_url) {
        if ($w !== 'c' && $w !== '') {
            return;
        }
        
        $bo_table = is_array($board) ? $board['bo_table'] : $board;
        
        if (empty($bo_table) || empty($wr_id)) {
            return;
        }
        
        $this->process_new_post($bo_table, $wr_id);
    }

    public function handle_new_post_direct($bo_table, $wr_id, $write_data = null) {
        $this->process_new_post($bo_table, $wr_id, $write_data);
    }

    private function process_new_post($bo_table, $wr_id, $write_data = null) {
        if (!$this->validate_new_post($bo_table, $wr_id)) {
            return;
        }
        
        if (!$this->load_pushmanager_library()) {
            return;
        }
        
        $this->send_new_post_notification($bo_table, $wr_id, $write_data);
    }

    private function validate_new_post($bo_table, $wr_id) {
        try {
            $write_table = $GLOBALS['g5']['write_prefix'] . $bo_table;
            $sql = "SELECT wr_reply, wr_parent, wr_is_comment FROM {$write_table} WHERE wr_id = " . intval($wr_id);
            $result = sql_fetch($sql);
            
            if (!$result) {
                return false;
            }
            
            if (!empty($result['wr_reply']) || (isset($result['wr_is_comment']) && $result['wr_is_comment'])) {
                return false;
            }
            
            if ($result['wr_parent'] != $wr_id) {
                return false;
            }
            
            $new_post_key = $bo_table . '_' . $wr_id;
            if (isset($this->new_posts[$new_post_key])) {
                return false;
            }
            
            $this->new_posts[$new_post_key] = time();
            return true;
            
        } catch (Exception $e) {
            return false;
        }
    }

    private function send_new_post_notification($bo_table, $wr_id, $write_data = null) {
        try {
            if (!$write_data) {
                $write_data = $this->get_write_data($bo_table, $wr_id);
            }
            
            if (!$write_data) {
                return;
            }
            
            // 메시지 포맷팅 적용
            $variables = array(
                'BOARD_NAME' => $this->get_board_name($bo_table),
                'SENDER_ID' => $write_data['wr_name'],
                'SENDER_NICKNAME' => $write_data['wr_name'],
                'POST_TITLE' => $write_data['wr_subject'],
                'POST_CONTENT' => $this->truncate_text(strip_tags($write_data['wr_content']), 100),
                'SITE_NAME' => $this->get_site_name()
            );
            
            $title_format = $this->get_message_format('newpost_title');
            $message_format = $this->get_message_format('newpost_message');
            
            $title = $this->replace_variables($title_format, $variables);
            $message = $this->replace_variables($message_format, $variables);
            
            $notification_users = get_board_notification_admins($bo_table);
            if (empty($notification_users)) {
                return;
            }
            
            $author_mb_id = $write_data['mb_id'] ?? '';
            if (!empty($author_mb_id)) {
                $notification_users = array_values(array_diff($notification_users, array($author_mb_id)));
            }

            $options = array(
                'notification_type' => 'newpost',
                'sender_id' => $write_data['mb_id'] ?? null,
                'click_action' => get_site_origin() . "/bbs/board.php?bo_table={$bo_table}&wr_id={$wr_id}",
                'require_interaction' => true,
                'tag' => 'new_post_' . $bo_table . '_' . $wr_id,
                'renotify' => true,
                'bo_table' => $bo_table,
                'wr_id' => $wr_id
            );

            return send_push_unified($notification_users, $title, $message, $options);

            
            
        } catch (Exception $e) {
            // 오류 로깅
        }
    }

    private function get_write_data($bo_table, $wr_id) {
        try {
            $write_table = $GLOBALS['g5']['write_prefix'] . $bo_table;
            $sql = "SELECT * FROM {$write_table} WHERE wr_id = " . intval($wr_id);
            $result = sql_fetch($sql);
            
            if ($result) {
                $result['wr_id'] = $wr_id;
            }
            
            return $result;
            
        } catch (Exception $e) {
            return null;
        }
    }

    public function handle_comment($board, $wr_id, $w, $qstr, $redirect_url) {
        if ($w !== 'c') {
            return;
        }
        
        $this->process_comment($board, $wr_id);
    }

    public function handle_comment_insert($board, $wr_id, $comment_data) {
        $this->process_comment($board, $wr_id, $comment_data);
    }

    private function process_comment($board, $wr_id, $comment_data = null) {
        global $g5, $member;
        
        try {
            if (!$this->load_pushmanager_library()) {
                return;
            }

            $bo_table = $board['bo_table'];
            
            $comment_id = $this->find_latest_comment_id($bo_table, $wr_id);
            if (!$comment_id) {
                return;
            }
            
            if (!$comment_data) {
                $comment_data = $this->get_comment_data($bo_table, $comment_id);
            }
            
            if (!$comment_data) {
                return;
            }
            
            // 원글 정보 가져오기
            $original_post = $this->get_write_data($bo_table, $wr_id);
            if (!$original_post) {
                return;
            }
            
            // 메시지 포맷팅 적용
            $variables = array(
                'BOARD_NAME' => $this->get_board_name($bo_table),
                'SENDER_ID' => $comment_data['wr_name'],
                'SENDER_NICKNAME' => $comment_data['wr_name'],
                'POST_TITLE' => $original_post['wr_subject'],
                'POST_AUTHOR' => $original_post['wr_name'],
                'COMMENT_CONTENT' => $this->truncate_text(strip_tags($comment_data['wr_content']), 100),
                'SITE_NAME' => $this->get_site_name()
            );
            
            $title_format = $this->get_message_format('comment_title');
            $message_format = $this->get_message_format('comment_message');
            
            $title = $this->replace_variables($title_format, $variables);
            $message = $this->replace_variables($message_format, $variables);
            
            if ($original_post['mb_id'] !== ($comment_data['mb_id'] ?? '')) {
                $click_url = get_site_origin() . "/bbs/board.php?bo_table={$bo_table}&wr_id={$wr_id}";
                $options = array(
                    'notification_type' => 'comment',
                    'sender_id' => $comment_data['mb_id'] ?? null,
                    'click_action' => $click_url,
                    'require_interaction' => false,
                    'tag' => 'comment_' . $bo_table . '_' . $wr_id,
                    'bo_table' => $bo_table,
                    'wr_id' => $wr_id,
                    'comment_id' => $comment_id
                );
                return send_push_unified(array($original_post['mb_id']), $title, $message, $options);
            }
            
        } catch (Exception $e) {
            // 오류 로깅
        }
    }

    private function find_latest_comment_id($bo_table, $wr_id) {
        try {
            $write_table = $GLOBALS['g5']['write_prefix'] . $bo_table;
            $sql = "SELECT wr_id FROM {$write_table} 
                    WHERE wr_parent = " . intval($wr_id) . " 
                    AND wr_is_comment = 1 
                    ORDER BY wr_id DESC 
                    LIMIT 1";
            
            $result = sql_fetch($sql);
            return $result ? $result['wr_id'] : null;
            
        } catch (Exception $e) {
            return null;
        }
    }

    private function get_comment_data($bo_table, $comment_id) {
        try {
            $write_table = $GLOBALS['g5']['write_prefix'] . $bo_table;
            $sql = "SELECT * FROM {$write_table} WHERE wr_id = " . intval($comment_id);
            $result = sql_fetch($sql);
            
            return $result;
            
        } catch (Exception $e) {
            return null;
        }
    }

    public function handle_memo($member_list, $str_nick_list, $redirect_url, $memo = '') {
        $this->process_memo($member_list, $str_nick_list, $memo);
    }

    public function handle_memo_direct($sender, $recipient, $me_id, $memo = '') {
        // 메시지 포맷팅 적용
        $memo_data = $this->get_memo_data($me_id);
        
        $variables = array(
            'SENDER_ID' => $sender['mb_id'],
            'SENDER_NICKNAME' => $sender['mb_nick'],
            'RECEIVER_ID' => $recipient['mb_id'],
            'RECEIVER_NICKNAME' => $recipient['mb_nick'],
            'MEMO_SUBJECT' => $memo_data ? $memo_data['me_subject'] : '',
            'MEMO_CONTENT' => $memo_data ? $this->truncate_text(strip_tags($memo_data['me_memo']), 100) : $this->truncate_text(strip_tags($memo), 100),
            'SITE_NAME' => $this->get_site_name()
        );
        
        $title_format = $this->get_message_format('memo_title');
        $message_format = $this->get_message_format('memo_message');
        
        $title = $this->replace_variables($title_format, $variables);
        $message = $this->replace_variables($message_format, $variables);
        
        $options = array(
            'notification_type' => 'message',
            'sender_id' => $sender['mb_id'],
            'click_action' => get_site_origin() . "/bbs/memo_view.php?me_id={$me_id}&kind=recv",
            'require_interaction' => true,
            'tag' => 'memo_' . $me_id,
            'me_id' => $me_id,
            'mb_id' => $sender['mb_id']
        );
        return send_push_unified(array($recipient['mb_id']), $title, $message, $options);
    }

    private function get_memo_data($me_id) {
        try {
            if (empty($me_id)) return null;
            
            $sql = "SELECT * FROM " . $GLOBALS['g5']['memo_table'] . " WHERE me_id = " . intval($me_id);
            $result = sql_fetch($sql);
            
            return $result;
            
        } catch (Exception $e) {
            return null;
        }
    }

    private function process_memo($member_list, $str_nick_list, $memo) {
        global $member;
        
        try {
            if (!$member['mb_id']) {
                return;
            }

            if (!$this->load_pushmanager_library()) {
                return;
            }

            if (!$memo && isset($_POST['me_memo'])) {
                $memo = $_POST['me_memo'];
            }

            if (isset($member_list['id']) && is_array($member_list['id'])) {
                for ($i = 0; $i < count($member_list['id']); $i++) {
                    $this->process_memo_recipient($member_list, $i, $member, $memo);
                }
            }
            
        } catch (Exception $e) {
            // 오류 로깅
        }
    }

    private function process_memo_recipient($member_list, $index, $sender, $memo) {
        try {
            $mb_id = $member_list['id'][$index];
            $me_id = isset($member_list['me_id'][$index]) ? (int) $member_list['me_id'][$index] : 0;

            if (($mb_id === $sender['mb_id']) || !$me_id) {
                return;
            }

            $recipient = get_member($mb_id, '*', true);
            if (!($recipient && $recipient['mb_id'])) {
                return;
            }

            // 메시지 포맷팅 적용
            $variables = array(
                'SENDER_ID' => $sender['mb_id'],
                'SENDER_NICKNAME' => $sender['mb_nick'],
                'RECEIVER_ID' => $recipient['mb_id'],
                'RECEIVER_NICKNAME' => $recipient['mb_nick'],
                'MEMO_SUBJECT' => isset($_POST['me_subject']) ? $_POST['me_subject'] : '',
                'MEMO_CONTENT' => $this->truncate_text(strip_tags($memo), 100),
                'SITE_NAME' => $this->get_site_name()
            );
            
            $title_format = $this->get_message_format('memo_title');
            $message_format = $this->get_message_format('memo_message');
            
            $title = $this->replace_variables($title_format, $variables);
            $message = $this->replace_variables($message_format, $variables);

            $me_id = isset($member_list['me_id'][$index]) ? (int) $member_list['me_id'][$index] : 0;
            $mb_id = $member_list['id'][$index];

            $options = array(
                'notification_type' => 'message',
                'sender_id' => $sender['mb_id'],
                'click_action' => get_site_origin() . "/bbs/memo_view.php?me_id={$me_id}&kind=recv",
                'require_interaction' => true,
                'tag' => 'memo_' . $me_id,
                'me_id' => $me_id,
                'mb_id' => $sender['mb_id']
            );
            return send_push_unified(array($mb_id), $title, $message, $options);
            
        } catch (Exception $e) {
            // 오류 로깅
        }
    }

    public function test_message_formatting() {
        // 메시지 포맷팅 테스트 함수
        $test_variables = array(
            'BOARD_NAME' => '자유게시판',
            'SENDER_ID' => 'testuser',
            'SENDER_NICKNAME' => '테스트사용자',
            'POST_TITLE' => '테스트 글입니다',
            'POST_CONTENT' => '이것은 테스트 내용입니다.',
            'COMMENT_CONTENT' => '테스트 댓글입니다.',
            'MEMO_CONTENT' => '테스트 쪽지입니다.',
            'SITE_NAME' => $this->get_site_name()
        );
        
        $results = array();
        
        // 새글 알림 테스트
        $newpost_title = $this->replace_variables($this->get_message_format('newpost_title'), $test_variables);
        $newpost_message = $this->replace_variables($this->get_message_format('newpost_message'), $test_variables);
        $results['newpost'] = array('title' => $newpost_title, 'message' => $newpost_message);
        
        // 댓글 알림 테스트
        $comment_title = $this->replace_variables($this->get_message_format('comment_title'), $test_variables);
        $comment_message = $this->replace_variables($this->get_message_format('comment_message'), $test_variables);
        $results['comment'] = array('title' => $comment_title, 'message' => $comment_message);
        
        // 쪽지 알림 테스트
        $memo_title = $this->replace_variables($this->get_message_format('memo_title'), $test_variables);
        $memo_message = $this->replace_variables($this->get_message_format('memo_message'), $test_variables);
        $results['memo'] = array('title' => $memo_title, 'message' => $memo_message);
        
        return $results;
    }
}

if (!isset($GLOBALS['pushmanager_hook_initialized'])) {
    $GLOBALS['pushmanager_hook_initialized'] = true;
    $GLOBALS['pushmanager_hook'] = PushManagerHook::getInstance();
}

function show_push_debug_log() {
    global $pushmanager_hook;
    if (isset($pushmanager_hook)) {
        echo '<pre>' . htmlspecialchars($pushmanager_hook->get_debug_log()) . '</pre>';
    }
}

function clear_push_debug_log() {
    global $pushmanager_hook;
    if (isset($pushmanager_hook)) {
        return $pushmanager_hook->clear_debug_log();
    }
    return false;
}

function test_push_library() {
    global $pushmanager_hook;
    if (isset($pushmanager_hook)) {
        return $pushmanager_hook->test_library_functions();
    }
    return array();
}

function test_push_message_formatting() {
    global $pushmanager_hook;
    if (isset($pushmanager_hook)) {
        return $pushmanager_hook->test_message_formatting();
    }
    return array();
}
?>