/**
 * Push Manager Core - 최적화된 통합 JavaScript 라이브러리
 * GnuBoard5용 푸시 알림 관리 시스템 (중복 요청 해결 버전)
 * 
 * 최적화 포인트:
 * - 중복 초기화 방지
 * - 단일 구독 프로세스
 * - 실시간 UI 업데이트 (페이지 새로고침 제거)
 * - API 요청 최적화 및 모니터링
 */

// ========================================
// 전역 변수 및 상수
// ========================================

const PUSHMANAGER_CONFIG = {
    version: 'beta 0.0.3',
    ajaxUrl: g5_url + '/plugin/pushmanager/pushmanager_ajax.php',
    serviceWorkerPath: g5_url + '/plugin/pushmanager/js/service-worker.js',
    debug: false,
    showToast: true,
    autoInit: true,
    cachePrefix: 'pushmanager-',
    apiTimeout: 10000
};

let globalPushManager = null;
let statusCheckInterval = null;

// ========================================
// API 요청 모니터링 클래스
// ========================================

class APIRequestMonitor {
    constructor() {
        this.requests = new Map();
        this.requestCount = 0;
        this.duplicateRequests = new Set();
    }

    logRequest(action, data) {
        this.requestCount++;
        const requestKey = `${action}_${JSON.stringify(data)}`;
        const requestId = `${action}_${Date.now()}_${this.requestCount}`;
        
        // 중복 요청 감지
        if (this.duplicateRequests.has(requestKey)) {
            console.warn(`[중복 API 요청 감지] ${action}`, data);
        } else {
            this.duplicateRequests.add(requestKey);
            // 5초 후 중복 체크 리셋
            setTimeout(() => this.duplicateRequests.delete(requestKey), 5000);
        }
        
        this.requests.set(requestId, {
            action,
            data,
            timestamp: Date.now(),
            status: 'pending'
        });

        if (PUSHMANAGER_CONFIG.debug) {
            console.log(`[API Request ${this.requestCount}] ${action}`, data);
        }
        
        return requestId;
    }

    logResponse(requestId, response, error = null) {
        const request = this.requests.get(requestId);
        if (request) {
            request.status = error ? 'error' : 'success';
            request.response = response;
            request.error = error;
            request.duration = Date.now() - request.timestamp;

            if (PUSHMANAGER_CONFIG.debug) {
                console.log(`[API Response] ${request.action} (${request.duration}ms)`, 
                           error ? error : response);
            }
        }
    }

    getRequestStats() {
        const stats = {
            total: this.requests.size,
            pending: 0,
            success: 0,
            error: 0
        };

        this.requests.forEach(req => {
            stats[req.status]++;
        });

        return stats;
    }
}

const apiMonitor = new APIRequestMonitor();

// ========================================
// 유틸리티 함수들
// ========================================

class PushManagerUtils {
    /**
     * ArrayBuffer를 Base64로 변환
     */
    static arrayBufferToBase64(buffer) {
        const bytes = new Uint8Array(buffer);
        let binary = '';
        for (let i = 0; i < bytes.byteLength; i++) {
            binary += String.fromCharCode(bytes[i]);
        }
        return window.btoa(binary);
    }

    /**
     * URL-safe base64를 Uint8Array로 변환
     */
    static urlBase64ToUint8Array(base64String) {
        const padding = '='.repeat((4 - base64String.length % 4) % 4);
        const base64 = (base64String + padding)
            .replace(/-/g, '+')
            .replace(/_/g, '/');
            
        const rawData = window.atob(base64);
        const outputArray = new Uint8Array(rawData.length);
        
        for (let i = 0; i < rawData.length; ++i) {
            outputArray[i] = rawData.charCodeAt(i);
        }
        
        return outputArray;
    }

    /**
     * 디바이스 및 브라우저 감지
     */
    static detectDevice() {
        const userAgent = navigator.userAgent;
        const isIOS = /iPad|iPhone|iPod/.test(userAgent);
        const isAndroid = /Android/.test(userAgent);
        const isSafari = /Safari/.test(userAgent) && !/Chrome/.test(userAgent);
        const isChrome = /Chrome/.test(userAgent);
        const isEdge = /Edge/.test(userAgent);
        const isFirefox = /Firefox/.test(userAgent);

        return { 
            isIOS, 
            isAndroid, 
            isSafari, 
            isChrome, 
            isEdge, 
            isFirefox,
            isMobile: isIOS || isAndroid,
            isDesktop: !isIOS && !isAndroid
        };
    }

    /**
     * 브라우저별 디바이스 아이콘 반환
     */
    static getDeviceIcon() {
        const device = this.detectDevice();
        
        if (device.isIOS) return '📱';
        if (device.isAndroid) return '🤖';
        if (device.isChrome) return '🌐';
        if (device.isSafari) return '🧭';
        if (device.isFirefox) return '🦊';
        if (device.isEdge) return '🔷';
        
        return '🖥️';
    }

    /**
     * 브라우저별 디바이스 이름 반환
     */
    static getDeviceName() {
        const device = this.detectDevice();
        const userAgent = navigator.userAgent;
        
        let deviceName = 'Unknown Device';
        let browserName = '';
        
        // 브라우저 감지
        if (device.isChrome) browserName = 'Chrome';
        else if (device.isSafari) browserName = 'Safari';
        else if (device.isFirefox) browserName = 'Firefox';
        else if (device.isEdge) browserName = 'Edge';
        else browserName = 'Browser';
        
        // 플랫폼 감지
        if (device.isIOS) {
            if (/iPhone/.test(userAgent)) deviceName = `iPhone (${browserName})`;
            else if (/iPad/.test(userAgent)) deviceName = `iPad (${browserName})`;
            else deviceName = `iOS Device (${browserName})`;
        } else if (device.isAndroid) {
            deviceName = `Android Device (${browserName})`;
        } else if (/Windows/.test(userAgent)) {
            deviceName = `Windows PC (${browserName})`;
        } else if (/Mac/.test(userAgent)) {
            deviceName = `Mac (${browserName})`;
        } else if (/Linux/.test(userAgent)) {
            deviceName = `Linux (${browserName})`;
        } else {
            deviceName = `Desktop (${browserName})`;
        }
        
        return deviceName;
    }

    /**
     * 토스트 메시지 표시
     */
    static showToast(message, type = 'info', duration = 5000) {
        // 기존 토스트 제거
        const existingToast = document.getElementById('pushmanager-toast');
        if (existingToast) {
            existingToast.remove();
        }
        
        // 새 토스트 생성
        const toast = document.createElement('div');
        toast.id = 'pushmanager-toast';
        toast.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 16px 24px;
            border-radius: 12px;
            color: white;
            font-weight: 600;
            z-index: 10000;
            max-width: 400px;
            word-wrap: break-word;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.2);
            backdrop-filter: blur(10px);
            ${this.getToastStyles(type)}
            transform: translateX(100%);
            transition: transform 0.3s ease;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        `;
        
        toast.textContent = message;
        document.body.appendChild(toast);
        
        // 애니메이션으로 표시
        setTimeout(() => {
            toast.style.transform = 'translateX(0)';
        }, 100);
        
        // 자동 제거
        setTimeout(() => {
            if (toast.parentNode) {
                toast.style.transform = 'translateX(100%)';
                setTimeout(() => {
                    if (toast.parentNode) {
                        toast.remove();
                    }
                }, 300);
            }
        }, duration);
    }

    /**
     * 토스트 스타일 반환
     */
    static getToastStyles(type) {
        const styles = {
            success: 'background: linear-gradient(135deg, #10b981, #059669);',
            error: 'background: linear-gradient(135deg, #ef4444, #dc2626);',
            warning: 'background: linear-gradient(135deg, #f59e0b, #d97706);',
            info: 'background: linear-gradient(135deg, #3b82f6, #2563eb);'
        };
        
        return styles[type] || styles.info;
    }

    /**
     * 최적화된 API 요청 (중복 요청 방지)
     */
    static async apiRequest(action, data = {}, method = 'POST') {
        const requestId = apiMonitor.logRequest(action, data);
        
        try {
            const url = `${PUSHMANAGER_CONFIG.ajaxUrl}?action=${action}`;
            const controller = new AbortController();
            const timeoutId = setTimeout(() => controller.abort(), PUSHMANAGER_CONFIG.apiTimeout);
            
            const options = {
                method: method,
                headers: {
                    'Content-Type': 'application/json',
                },
                signal: controller.signal
            };
            
            if (method === 'POST') {
                options.body = JSON.stringify(data);
            }
            
            const response = await fetch(url, options);
            clearTimeout(timeoutId);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const result = await response.json();
            apiMonitor.logResponse(requestId, result);
            
            return result;
            
        } catch (error) {
            apiMonitor.logResponse(requestId, null, error);
            
            if (error.name === 'AbortError') {
                throw new Error('요청 시간이 초과되었습니다.');
            }
            throw error;
        }
    }

    /**
     * 로그 출력
     */
    static log(...args) {
        if (PUSHMANAGER_CONFIG.debug) {
            console.log('[PushManager]', ...args);
        }
    }
}

// ========================================
// 최적화된 Push Manager 클래스
// ========================================

class GnuBoard5PushManager {
    constructor(options = {}) {
        this.options = {
            ...PUSHMANAGER_CONFIG,
            ...options
        };
        
        this.registration = null;
        this.subscription = null;
        this.isSupported = false;
        this.isSubscribed = false;
        this.vapidPublicKey = null;
        
        // 중복 방지 플래그들
        this.isInitializing = false;
        this.isSubscribing = false;
        this.isUnsubscribing = false;
        
        // 이벤트 리스너 저장
        this.eventListeners = new Map();
        
        this.log('Push Manager 인스턴스 생성');
    }

    /**
     * 최적화된 초기화 (중복 방지)
     */
    async init() {
        if (this.isInitializing) {
            this.log('초기화가 이미 진행 중입니다.');
            return false;
        }

        this.isInitializing = true;

        try {
            this.log('초기화 시작');
            
            // 브라우저 지원 확인
            if (!this.checkSupport()) {
                this.log('Push notifications not supported');
                this.updateUI();
                return false;
            }
            
            // Service Worker 등록
            await this.registerServiceWorker();
            
            // VAPID 키 미리 로드 (구독 시점이 아닌 초기화 시점에서)
            try {
                await this.loadVapidKey();
                this.log('VAPID 키 로드 완료');
            } catch (error) {
                this.log('VAPID key loading failed, continuing without it:', error);
            }
            
            // 기존 구독 확인
            await this.checkExistingSubscription();
            
            // UI 업데이트
            this.updateUI();
            
            // 이벤트 리스너 설정
            this.setupEventListeners();
            
            this.log('초기화 완료');
            return true;
            
        } catch (error) {
            this.log('초기화 실패:', error);
            return false;
        } finally {
            this.isInitializing = false;
        }
    }

    /**
     * 브라우저 지원 확인
     */
    checkSupport() {
        this.isSupported = (
            'serviceWorker' in navigator &&
            'PushManager' in window &&
            'Notification' in window
        );
        
        return this.isSupported;
    }

    /**
     * Service Worker 등록
     */
    async registerServiceWorker() {
        try {
            this.registration = await navigator.serviceWorker.register(
                this.options.serviceWorkerPath,
                { scope: '/plugin/pushmanager/js/' }
            );
            
            this.log('Service Worker 등록 성공:', this.registration.scope);
            
            // Service Worker 업데이트 확인
            this.registration.addEventListener('updatefound', () => {
                const newWorker = this.registration.installing;
                if (newWorker) {
                    newWorker.addEventListener('statechange', () => {
                        if (newWorker.state === 'installed' && navigator.serviceWorker.controller) {
                            this.showToast('새로운 버전이 있습니다. 페이지를 새로고침해주세요.', 'info');
                        }
                    });
                }
            });
            
            return this.registration;
            
        } catch (error) {
            this.log('Service Worker 등록 실패:', error);
            throw error;
        }
    }

    /**
     * VAPID 공개키 로드
     */
    async loadVapidKey() {
        if (this.options.vapidPublicKey) {
            this.vapidPublicKey = this.options.vapidPublicKey;
            return this.vapidPublicKey;
        }
        
        if (this.vapidPublicKey) {
            return this.vapidPublicKey; // 이미 로드됨
        }
        
        try {
            const response = await PushManagerUtils.apiRequest('get_vapid_key', {}, 'GET');
            if (response.success && response.data.vapidPublicKey) {
                this.vapidPublicKey = response.data.vapidPublicKey;
                this.log('VAPID 공개키 로드 완료');
                return this.vapidPublicKey;
            } else {
                throw new Error(response.message || 'VAPID 키 로드 실패');
            }
        } catch (error) {
            this.log('VAPID 키 로드 실패:', error);
            throw error;
        }
    }

    /**
     * 기존 구독 확인
     */
    async checkExistingSubscription() {
        if (!this.registration) return;
        
        try {
            this.subscription = await this.registration.pushManager.getSubscription();
            this.isSubscribed = !!this.subscription;
            
            if (this.subscription) {
                this.log('기존 구독 발견');
                // 서버 상태 검증은 선택적으로만 수행 (불필요한 요청 방지)
                // await this.verifySubscriptionOnServer();
            }
            
        } catch (error) {
            this.log('구독 확인 중 오류:', error);
        }
    }

    /**
     * 최적화된 푸시 구독 (단일 요청)
     */
    async subscribe() {
        // 중복 구독 방지
        if (this.isSubscribing) {
            this.log('구독이 이미 진행 중입니다.');
            return this.subscription;
        }

        if (this.isSubscribed && this.subscription) {
            this.log('이미 구독되어 있습니다.');
            return this.subscription;
        }

        this.isSubscribing = true;

        try {
            // 권한 요청
            const permission = await Notification.requestPermission();
            if (permission !== 'granted') {
                throw new Error('푸시 알림 권한이 거부되었습니다.');
            }
            
            if (!this.registration) {
                throw new Error('Service Worker가 등록되지 않았습니다.');
            }
            
            // VAPID 키가 없으면 다시 시도 (초기화에서 실패했을 경우)
            if (!this.vapidPublicKey) {
                try {
                    await this.loadVapidKey();
                } catch (error) {
                    this.log('VAPID 키 재로드 실패, 없이 진행:', error);
                }
            }
            
            // 구독 옵션 설정
            const subscribeOptions = {
                userVisibleOnly: true
            };
            
            // VAPID 키가 있는 경우에만 추가
            if (this.vapidPublicKey) {
                subscribeOptions.applicationServerKey = PushManagerUtils.urlBase64ToUint8Array(this.vapidPublicKey);
                this.log('VAPID 키를 사용하여 구독');
            } else {
                this.log('VAPID 키 없이 구독 (서버에서 처리)');
            }
            
            // 구독 생성
            this.subscription = await this.registration.pushManager.subscribe(subscribeOptions);
            this.isSubscribed = true;
            
            this.log('푸시 구독 생성 완료');
            
            // 서버에 구독 정보 등록 (단일 요청)
            await this.registerSubscriptionOnServer();
            
            // UI 업데이트 (페이지 새로고침 없이)
            this.updateUI();
            await this.updateDeviceListUI();
            
            // 성공 알림
            this.showToast('푸시 알림 구독이 완료되었습니다!', 'success');
            
            // 이벤트 발생
            this.emit('subscribe', this.subscription);
            
            return this.subscription;
            
        } catch (error) {
            this.log('구독 실패:', error);
            this.showToast('푸시 알림 구독에 실패했습니다: ' + error.message, 'error');
            throw error;
        } finally {
            this.isSubscribing = false;
        }
    }

    /**
     * 최적화된 푸시 구독 해제
     */
    async unsubscribe() {
        if (this.isUnsubscribing) {
            this.log('구독 해제가 이미 진행 중입니다.');
            return true;
        }

        if (!this.subscription) {
            this.log('활성 구독이 없습니다.');
            return true;
        }

        this.isUnsubscribing = true;

        try {
            // 클라이언트에서 구독 해제
            const successful = await this.subscription.unsubscribe();
            
            if (successful) {
                this.log('푸시 구독 해제 완료');
                
                // 서버에서 구독 해제
                await this.unregisterSubscriptionOnServer();
                
                this.subscription = null;
                this.isSubscribed = false;
                
                // UI 업데이트 (페이지 새로고침 없이)
                this.updateUI();
                await this.updateDeviceListUI();
                
                // 성공 알림
                this.showToast('푸시 알림 구독이 해제되었습니다.', 'info');
                
                // 이벤트 발생
                this.emit('unsubscribe');
            }
            
            return successful;
            
        } catch (error) {
            this.log('구독 해제 실패:', error);
            this.showToast('구독 해제에 실패했습니다: ' + error.message, 'error');
            throw error;
        } finally {
            this.isUnsubscribing = false;
        }
    }

    /**
     * 서버에 구독 정보 등록
     */
    async registerSubscriptionOnServer() {
        if (!this.subscription) {
            throw new Error('등록할 구독이 없습니다.');
        }
        
        try {
            const subscriptionData = this.subscriptionToObject(this.subscription);
            const response = await PushManagerUtils.apiRequest('subscribe', {
                subscription: subscriptionData
            });
            
            if (!response.success) {
                throw new Error(response.message);
            }
            
            this.log('서버에 구독 정보 등록 완료');
            return response;
            
        } catch (error) {
            this.log('서버 구독 등록 실패:', error);
            throw error;
        }
    }

    /**
     * 서버에서 구독 해제
     */
    async unregisterSubscriptionOnServer() {
        if (!this.subscription) {
            return true; // 이미 해제됨
        }
        
        try {
            const response = await PushManagerUtils.apiRequest('unsubscribe', {
                endpoint: this.subscription.endpoint
            });
            
            if (!response.success) {
                throw new Error(response.message);
            }
            
            this.log('서버에서 구독 해제 완료');
            return response;
            
        } catch (error) {
            this.log('서버 구독 해제 실패:', error);
            throw error;
        }
    }

    /**
     * 기기 목록 UI 실시간 업데이트
     */
    async updateDeviceListUI() {
        try {
            const response = await PushManagerUtils.apiRequest('get_user_devices', {}, 'GET');
            
            if (response.success && response.data.devices) {
                this.renderDeviceList(response.data.devices);
            }
        } catch (error) {
            this.log('기기 목록 업데이트 실패:', error);
        }
    }

    /**
     * 기기 목록 렌더링
     */
    renderDeviceList(devices) {
        const deviceGrid = document.querySelector('.device-grid');
        const emptyState = document.querySelector('.empty-state');
        const sectionTitle = document.querySelector('.section-title');
        const testSection = document.querySelector('.test-section');
        
        // 기기 개수 업데이트
        if (sectionTitle && sectionTitle.textContent.includes('등록된 기기')) {
            sectionTitle.textContent = `등록된 기기 (${devices.length}개)`;
        }
        
        if (devices.length === 0) {
            // 기기가 없는 경우
            if (deviceGrid) deviceGrid.style.display = 'none';
            if (testSection) testSection.style.display = 'none';
            if (emptyState) emptyState.style.display = 'block';
        } else {
            // 기기가 있는 경우
            if (emptyState) emptyState.style.display = 'none';
            if (testSection) testSection.style.display = 'block';
            
            if (deviceGrid) {
                deviceGrid.style.display = 'grid';
                deviceGrid.innerHTML = this.generateDeviceHTML(devices);
                
                // 새로 생성된 버튼들에 이벤트 리스너 재설정
                this.setupDeviceButtonEvents();
            }
        }
    }

    /**
     * 기기 HTML 생성
     */
    generateDeviceHTML(devices) {
        return devices.map(device => `
            <div class="device-item">
                <div class="device-header">
                    <div class="device-info">
                        <span class="device-icon">${device.device_icon || '🖥️'}</span>
                        <div class="device-details">
                            <h5>${this.escapeHtml(device.device_name || 'Unknown Device')}</h5>
                            <small>${this.escapeHtml((device.device_description || '').substr(0, 80))}</small>
                            <small class="device-status ${device.is_active ? 'status-active' : 'status-inactive'}">
                                ${device.is_active ? '● 활성' : '● 비활성'} | 
                                등록일: ${new Date(device.created_at).toLocaleDateString('ko-KR', { month: 'numeric', day: 'numeric', hour: 'numeric', minute: 'numeric' })}
                            </small>
                        </div>
                    </div>
                    <div class="device-controls">
                        ${device.is_active ? `
                            <button type="button" class="btn-test" data-device-id="${device.id}">
                                테스트
                            </button>
                        ` : ''}
                        <label class="switch">
                            <input type="checkbox" name="device_settings[${device.id}]" value="1"
                                   ${device.is_active ? 'checked' : ''}>
                            <span class="slider"></span>
                        </label>
                    </div>
                </div>
            </div>
        `).join('');
    }

    /**
     * HTML 이스케이프
     */
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    /**
     * 기기 버튼 이벤트 설정
     */
    setupDeviceButtonEvents() {
        const testButtons = document.querySelectorAll('.btn-test[data-device-id]');
        testButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                const deviceId = e.target.getAttribute('data-device-id');
                this.sendTestPush(deviceId, e.target);
            });
        });
    }

    /**
     * 구독 객체를 일반 객체로 변환
     */
    subscriptionToObject(subscription) {
        return {
            endpoint: subscription.endpoint,
            keys: {
                p256dh: PushManagerUtils.arrayBufferToBase64(subscription.getKey('p256dh')),
                auth: PushManagerUtils.arrayBufferToBase64(subscription.getKey('auth'))
            }
        };
    }

    /**
     * UI 업데이트
     */
    updateUI() {
        const elements = this.getUIElements();
        
        if (!this.isSupported) {
            this.updatePushStatus('❌ 이 브라우저는 푸시 알림을 지원하지 않습니다.', 
                                '최신 버전의 Chrome, Firefox, Edge를 사용해주세요.', 'error');
            this.hideAllPushButtons();
            return;
        }
        
        const permission = Notification.permission;
        
        if (permission === 'denied') {
            this.updatePushStatus('🚫 푸시 알림 권한이 거부되었습니다.', 
                                '브라우저 설정에서 알림 권한을 허용해주세요.', 'error');
            this.hideAllPushButtons();
            return;
        }
        
        if (this.isSubscribed) {
            this.updatePushStatus('✅ 푸시 알림이 활성화되었습니다.', 
                                '모든 알림을 정상적으로 받을 수 있습니다.', 'success');
            this.showPushButton('unsubscribe');
        } else {
            this.updatePushStatus('⚠️ 푸시 알림이 구독되지 않았습니다.', 
                                '구독하기 버튼을 클릭하여 알림을 활성화하세요.', 'warning');
            this.showPushButton('subscribe');
        }
    }

    /**
     * UI 요소 가져오기
     */
    getUIElements() {
        return {
            status: document.getElementById('push-status'),
            statusDetail: document.getElementById('push-status-detail'),
            subscribeBtn: document.getElementById('push-subscribe-btn'),
            unsubscribeBtn: document.getElementById('push-unsubscribe-btn')
        };
    }

    /**
     * 푸시 상태 업데이트
     */
    updatePushStatus(statusText, detailText, type) {
        const elements = this.getUIElements();
        
        if (elements.status) {
            elements.status.innerHTML = `<div class="push-status-indicator status-${type}"></div>${statusText}`;
        }
        
        if (elements.statusDetail) {
            elements.statusDetail.textContent = detailText;
        }
    }

    /**
     * 푸시 버튼 표시/숨김
     */
    showPushButton(type) {
        const elements = this.getUIElements();
        
        if (type === 'subscribe') {
            if (elements.subscribeBtn) {
                elements.subscribeBtn.style.display = 'flex';
                elements.subscribeBtn.disabled = false;
            }
            if (elements.unsubscribeBtn) {
                elements.unsubscribeBtn.style.display = 'none';
            }
        } else if (type === 'unsubscribe') {
            if (elements.subscribeBtn) {
                elements.subscribeBtn.style.display = 'none';
            }
            if (elements.unsubscribeBtn) {
                elements.unsubscribeBtn.style.display = 'flex';
                elements.unsubscribeBtn.disabled = false;
            }
        }
    }

    /**
     * 모든 푸시 버튼 숨기기
     */
    hideAllPushButtons() {
        const elements = this.getUIElements();
        
        if (elements.subscribeBtn) elements.subscribeBtn.style.display = 'none';
        if (elements.unsubscribeBtn) elements.unsubscribeBtn.style.display = 'none';
    }

    /**
     * 이벤트 리스너 설정
     */
    setupEventListeners() {
        const elements = this.getUIElements();
        
        // 구독 버튼
        if (elements.subscribeBtn) {
            elements.subscribeBtn.addEventListener('click', async (e) => {
                e.preventDefault();
                await this.handleSubscribeClick(elements.subscribeBtn);
            });
        }
        
        // 구독 해제 버튼
        if (elements.unsubscribeBtn) {
            elements.unsubscribeBtn.addEventListener('click', async (e) => {
                e.preventDefault();
                if (confirm('정말 푸시 알림 구독을 해제하시겠습니까?')) {
                    await this.handleUnsubscribeClick(elements.unsubscribeBtn);
                }
            });
        }
        
        // 폼 처리
        this.setupFormHandling();
        
        // 스위치 애니메이션
        this.setupSwitchAnimations();

        // 전체 테스트 버튼
        this.setupTestAllButton();
    }

    /**
     * 최적화된 구독 버튼 클릭 처리 (페이지 새로고침 제거)
     */
    async handleSubscribeClick(button) {
        if (button.disabled || this.isSubscribing) return;
        
        const originalText = button.innerHTML;
        
        try {
            button.innerHTML = '<div class="loading-spinner"></div> 구독 중...';
            button.disabled = true;
            
            await this.subscribe();
            
            button.innerHTML = '✅ 완료';
            
            // 페이지 새로고침 대신 UI만 업데이트
            setTimeout(() => {
                button.innerHTML = originalText;
                button.disabled = false;
                this.updateUI();
            }, 2000);
            
        } catch (error) {
            this.log('구독 버튼 처리 실패:', error);
            
            button.innerHTML = '❌ 실패';
            
            setTimeout(() => {
                button.innerHTML = originalText;
                button.disabled = false;
            }, 3000);
        }
    }

    /**
     * 최적화된 구독 해제 버튼 클릭 처리 (페이지 새로고침 제거)
     */
    async handleUnsubscribeClick(button) {
        if (button.disabled || this.isUnsubscribing) return;
        
        const originalText = button.innerHTML;
        
        try {
            button.innerHTML = '<div class="loading-spinner"></div> 해제 중...';
            button.disabled = true;
            
            await this.unsubscribe();
            
            button.innerHTML = '✅ 완료';
            
            // 페이지 새로고침 대신 UI만 업데이트
            setTimeout(() => {
                button.innerHTML = originalText;
                button.disabled = false;
                this.updateUI();
            }, 2000);
            
        } catch (error) {
            this.log('구독 해제 버튼 처리 실패:', error);
            
            button.innerHTML = '❌ 실패';
            
            setTimeout(() => {
                button.innerHTML = originalText;
                button.disabled = false;
            }, 3000);
        }
    }

    /**
     * 폼 처리 설정
     */
    setupFormHandling() {
        const form = document.querySelector('form[action*="notification_setting_update.php"]');
        const saveBtn = document.getElementById('saveBtn');
        
        if (form && saveBtn) {
            form.addEventListener('submit', function(e) {
                if (!saveBtn.disabled) {
                    saveBtn.textContent = '저장 중...';
                    saveBtn.disabled = true;
                }
            });
        }
    }

    /**
     * 스위치 애니메이션 설정
     */
    setupSwitchAnimations() {
        const switches = document.querySelectorAll('.switch input');
        switches.forEach(switchInput => {
            switchInput.addEventListener('change', function() {
                this.parentElement.style.transform = 'scale(0.95)';
                setTimeout(() => {
                    this.parentElement.style.transform = 'scale(1)';
                }, 100);
            });
        });
    }

    /**
     * 전체 테스트 버튼 설정
     */
    setupTestAllButton() {
        const testAllBtn = document.querySelector('.btn-test-all');
        if (testAllBtn) {
            testAllBtn.addEventListener('click', (e) => {
                e.preventDefault();
                this.sendTestToAll();
            });
        }
    }

    /**
     * 테스트 푸시 발송
     */
    async sendTestPush(deviceId, button) {
        if (button && button.disabled) return;
        
        const originalText = button ? button.textContent : '';
        
        try {
            if (button) {
                button.textContent = '발송 중...';
                button.disabled = true;
                button.classList.add('sending');
            }
            
            const response = await fetch('./test_push.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `device_id=${deviceId}&action=test_single`
            });
            
            const data = await response.json();
            
            if (data.success) {
                if (button) {
                    button.textContent = '✅ 발송됨';
                    button.classList.remove('sending');
                    button.classList.add('success');
                    
                    setTimeout(() => {
                        button.textContent = originalText;
                        button.disabled = false;
                        button.classList.remove('success');
                    }, 3000);
                }
                
                this.showToast('테스트 알림이 발송되었습니다.', 'success');
            } else {
                throw new Error(data.message || '알 수 없는 오류');
            }
            
        } catch (error) {
            this.log('테스트 푸시 실패:', error);
            
            if (button) {
                button.textContent = '❌ 실패';
                button.classList.remove('sending');
                button.classList.add('error');
                
                setTimeout(() => {
                    button.textContent = originalText;
                    button.disabled = false;
                    button.classList.remove('error');
                }, 3000);
            }
            
            this.showToast('테스트 푸시 발송에 실패했습니다: ' + error.message, 'error');
        }
    }

    /**
     * 모든 활성 기기에 테스트 푸시 발송
     */
    async sendTestToAll() {
        const button = document.querySelector('.btn-test-all');
        if (!button || button.disabled) return;
        
        const originalText = button.textContent;
        
        try {
            button.textContent = '📤 발송 중...';
            button.disabled = true;
            
            const response = await fetch('./test_push.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=test_all'
            });
            
            const data = await response.json();
            
            if (data.success) {
                button.textContent = `✅ 모든 기기에 발송됨 (${data.count}개)`;
                this.showToast(`${data.count}개 기기에 테스트 알림을 발송했습니다.`, 'success');
                
                setTimeout(() => {
                    button.textContent = originalText;
                    button.disabled = false;
                }, 5000);
            } else {
                throw new Error(data.message || '알 수 없는 오류');
            }
            
        } catch (error) {
            this.log('전체 테스트 푸시 실패:', error);
            
            button.textContent = '❌ 발송 실패';
            
            setTimeout(() => {
                button.textContent = originalText;
                button.disabled = false;
            }, 3000);
            
            this.showToast('테스트 푸시 발송에 실패했습니다: ' + error.message, 'error');
        }
    }

    /**
     * 현재 상태 정보 반환
     */
    getStatus() {
        return {
            isSupported: this.isSupported,
            isSubscribed: this.isSubscribed,
            permission: this.isSupported ? Notification.permission : 'unsupported',
            subscription: this.subscription,
            registration: this.registration,
            vapidPublicKey: this.vapidPublicKey,
            isInitializing: this.isInitializing,
            isSubscribing: this.isSubscribing,
            isUnsubscribing: this.isUnsubscribing
        };
    }

    /**
     * 토스트 메시지 표시 (인스턴스 메서드)
     */
    showToast(message, type = 'info', duration = 5000) {
        if (!this.options.showToast) return;
        PushManagerUtils.showToast(message, type, duration);
    }

    /**
     * 이벤트 발생
     */
    emit(eventName, data = null) {
        const listeners = this.eventListeners.get(eventName) || [];
        listeners.forEach(listener => {
            try {
                listener(data);
            } catch (error) {
                this.log('이벤트 리스너 오류:', error);
            }
        });
    }

    /**
     * 이벤트 리스너 추가
     */
    on(eventName, listener) {
        if (!this.eventListeners.has(eventName)) {
            this.eventListeners.set(eventName, []);
        }
        this.eventListeners.get(eventName).push(listener);
    }

    /**
     * 이벤트 리스너 제거
     */
    off(eventName, listener) {
        const listeners = this.eventListeners.get(eventName) || [];
        const index = listeners.indexOf(listener);
        if (index > -1) {
            listeners.splice(index, 1);
        }
    }

    /**
     * 로그 출력
     */
    log(...args) {
        PushManagerUtils.log(...args);
    }

    /**
     * 정리 작업
     */
    destroy() {
        // 인터벌 정리
        if (statusCheckInterval) {
            clearInterval(statusCheckInterval);
            statusCheckInterval = null;
        }
        
        // 이벤트 리스너 정리
        this.eventListeners.clear();
        
        this.log('Push Manager 정리 완료');
    }
}

// ========================================
// 싱글톤 Push Manager Core 클래스
// ========================================

class PushManagerCore {
    constructor() {
        this.instance = null;
        this.isInitialized = false;
        this.isInitializing = false;
        this.initPromise = null;
    }

    /**
     * 싱글톤 인스턴스 반환 (중복 초기화 방지)
     */
    getInstance(options = {}) {
        if (!this.instance) {
            this.instance = new GnuBoard5PushManager(options);
        }
        return this.instance;
    }

    /**
     * 중복 방지 초기화
     */
    async init(options = {}) {
        // 이미 초기화 완료된 경우
        if (this.isInitialized) {
            return this.instance;
        }

        // 초기화가 진행 중인 경우 기존 Promise 반환
        if (this.isInitializing && this.initPromise) {
            return this.initPromise;
        }

        this.isInitializing = true;

        // 초기화 Promise 생성 및 저장
        this.initPromise = this._performInit(options);

        try {
            const result = await this.initPromise;
            this.isInitialized = true;
            return result;
        } catch (error) {
            this.isInitializing = false;
            this.initPromise = null;
            throw error;
        } finally {
            this.isInitializing = false;
        }
    }

    /**
     * 실제 초기화 수행
     */
    async _performInit(options) {
        try {
            this.instance = this.getInstance(options);
            const result = await this.instance.init();
            
            if (result) {
                this.setupGlobalEvents();
                PushManagerUtils.log('Push Manager Core 초기화 완료');
            }
            
            return this.instance;
        } catch (error) {
            PushManagerUtils.log('Push Manager Core 초기화 실패:', error);
            throw error;
        }
    }

    /**
     * 전역 이벤트 설정
     */
    setupGlobalEvents() {
        // 페이지 언로드 시 정리
        window.addEventListener('beforeunload', () => {
            if (this.instance) {
                this.instance.destroy();
            }
        });

        // PWA 관련 이벤트
        window.addEventListener('appinstalled', () => {
            PushManagerUtils.log('PWA 앱 설치 완료');
        });

        // Service Worker 메시지 수신
        if ('serviceWorker' in navigator) {
            navigator.serviceWorker.addEventListener('message', (event) => {
                PushManagerUtils.log('Service Worker 메시지:', event.data);
            });
        }
    }

    /**
     * 빠른 접근 메서드들
     */
    async subscribe() {
        const instance = await this.init();
        return await instance.subscribe();
    }

    async unsubscribe() {
        const instance = await this.init();
        return await instance.unsubscribe();
    }

    getStatus() {
        if (!this.instance) {
            return {
                isSupported: false,
                isSubscribed: false,
                permission: 'unknown',
                subscription: null,
                registration: null,
                vapidPublicKey: null,
                isInitialized: this.isInitialized,
                isInitializing: this.isInitializing
            };
        }
        
        return {
            ...this.instance.getStatus(),
            isInitialized: this.isInitialized,
            isInitializing: this.isInitializing
        };
    }

    showToast(message, type, duration) {
        PushManagerUtils.showToast(message, type, duration);
    }

    async sendTestPush(deviceId, button) {
        const instance = await this.init();
        return instance.sendTestPush(deviceId, button);
    }

    async sendTestToAll() {
        const instance = await this.init();
        return instance.sendTestToAll();
    }

    // 유틸리티 메서드들
    detectDevice() {
        return PushManagerUtils.detectDevice();
    }

    getDeviceIcon() {
        return PushManagerUtils.getDeviceIcon();
    }

    getDeviceName() {
        return PushManagerUtils.getDeviceName();
    }

    log(...args) {
        PushManagerUtils.log(...args);
    }

    /**
     * 강제 재초기화
     */
    async reinit(options = {}) {
        this.isInitialized = false;
        this.isInitializing = false;
        this.initPromise = null;
        
        if (this.instance) {
            this.instance.destroy();
            this.instance = null;
        }
        
        return await this.init(options);
    }
}

// ========================================
// 전역 인스턴스 및 자동 초기화
// ========================================

// 전역 싱글톤 인스턴스 생성
const pushManagerCore = new PushManagerCore();

// DOM 로드 완료 시 자동 초기화 (중복 방지)
document.addEventListener('DOMContentLoaded', async function() {
    PushManagerUtils.log('DOM 로드 완료 - 자동 초기화 시작');
    
    try {
        // 이미 초기화가 시작되었는지 확인
        if (!pushManagerCore.isInitialized && !pushManagerCore.isInitializing) {
            const instance = await pushManagerCore.init({
                debug: true,
                showToast: true
            });
            
            if (instance) {
                PushManagerUtils.log('Push Manager 자동 초기화 완료');
            }
        } else {
            PushManagerUtils.log('Push Manager 이미 초기화됨 또는 진행 중');
        }
    } catch (error) {
        PushManagerUtils.log('자동 초기화 실패:', error);
    }
});

// ========================================
// 전역 함수 및 호환성 제공
// ========================================

/**
 * 전역 함수들 (기존 코드 호환성 + 중복 방지)
 */
window.initializePushManagerCommon = async function() {
    return await pushManagerCore.init();
};

window.showToast = function(message, type, duration) {
    return PushManagerUtils.showToast(message, type, duration);
};

window.sendTestPush = async function(deviceId, button) {
    return await pushManagerCore.sendTestPush(deviceId, button);
};

window.sendTestToAll = async function() {
    return await pushManagerCore.sendTestToAll();
};

window.detectDevice = function() {
    return PushManagerUtils.detectDevice();
};

window.getDeviceIcon = function() {
    return PushManagerUtils.getDeviceIcon();
};

window.getDeviceName = function() {
    return PushManagerUtils.getDeviceName();
};

// 디버그 함수들
window.debugPushManager = function() {
    const status = pushManagerCore.getStatus();
    console.log('=== Push Manager Debug Info ===');
    console.log('Push Manager Status:', status);
    console.log('API Request Stats:', apiMonitor.getRequestStats());
    console.log('Recent Requests:', Array.from(apiMonitor.requests.values()).slice(-5));
    console.log('Device Info:', {
        device: PushManagerUtils.detectDevice(),
        icon: PushManagerUtils.getDeviceIcon(),
        name: PushManagerUtils.getDeviceName()
    });
    console.log('==============================');
};

window.debugAPIRequests = function() {
    console.log('=== API Request Statistics ===');
    console.log('Stats:', apiMonitor.getRequestStats());
    console.log('Recent Requests:', Array.from(apiMonitor.requests.values()).slice(-10));
    console.log('=============================');
};

// ========================================
// 전역 객체 등록
// ========================================

// 새로운 통합 API
window.PushManager = pushManagerCore;

// 기존 호환성을 위한 객체들 (중복 방지 적용)
window.PushManagerCommon = {
    initializePushManagerCommon: async () => await pushManagerCore.init(),
    showToast: PushManagerUtils.showToast,
    sendTestPush: async (deviceId, button) => await pushManagerCore.sendTestPush(deviceId, button),
    sendTestToAll: async () => await pushManagerCore.sendTestToAll(),
    detectDevice: PushManagerUtils.detectDevice,
    getDeviceIcon: PushManagerUtils.getDeviceIcon,
    getDeviceName: PushManagerUtils.getDeviceName,
    updatePushStatus: async (statusText, detailText, type) => {
        const instance = await pushManagerCore.init();
        instance.updatePushStatus(statusText, detailText, type);
    },
    showPushButton: async (type) => {
        const instance = await pushManagerCore.init();
        instance.showPushButton(type);
    },
    hideAllPushButtons: async () => {
        const instance = await pushManagerCore.init();
        instance.hideAllPushButtons();
    },
    handleSubscribe: async (button) => {
        const instance = await pushManagerCore.init();
        return await instance.handleSubscribeClick(button);
    },
    handleUnsubscribe: async (button) => {
        const instance = await pushManagerCore.init();
        return await instance.handleUnsubscribeClick(button);
    }
};

// 기존 GnuBoard5PushManager 호환성
window.GnuBoard5PushManager = GnuBoard5PushManager;
window.gnuBoard5Push = null; // 초기화 후 설정됨

// 초기화 완료 후 기존 변수 설정 (중복 방지)
pushManagerCore.init().then(instance => {
    if (!window.gnuBoard5Push) {
        window.gnuBoard5Push = instance;
        PushManagerUtils.log('기존 호환성 변수 설정 완료');
    }
}).catch(error => {
    PushManagerUtils.log('기존 호환성 변수 설정 실패:', error);
});

// ========================================
// 그누보드5 특화 헬퍼 함수들
// ========================================

window.pushManagerHelpers = {
    /**
     * 게시판 알림 구독
     */
    async subscribeToBoard(boardId) {
        const instance = await pushManagerCore.init();
        if (!instance.isSubscribed) {
            PushManagerUtils.showToast('먼저 푸시 알림을 구독해주세요.', 'warning');
            return false;
        }
        
        try {
            const response = await PushManagerUtils.apiRequest('subscribe_board', {
                board_id: boardId
            });
            
            if (response.success) {
                PushManagerUtils.showToast(`${boardId} 게시판 알림이 설정되었습니다.`, 'success');
                return true;
            } else {
                PushManagerUtils.showToast(response.message, 'error');
                return false;
            }
        } catch (error) {
            PushManagerUtils.log('게시판 구독 오류:', error);
            PushManagerUtils.showToast('게시판 알림 설정에 실패했습니다.', 'error');
            return false;
        }
    },

    /**
     * 댓글 알림 구독
     */
    async subscribeToComments(boardId, wrId) {
        const instance = await pushManagerCore.init();
        if (!instance.isSubscribed) {
            PushManagerUtils.showToast('먼저 푸시 알림을 구독해주세요.', 'warning');
            return false;
        }
        
        try {
            const response = await PushManagerUtils.apiRequest('subscribe_comments', {
                board_id: boardId,
                wr_id: wrId
            });
            
            if (response.success) {
                PushManagerUtils.showToast('댓글 알림이 설정되었습니다.', 'success');
                return true;
            } else {
                PushManagerUtils.showToast(response.message, 'error');
                return false;
            }
        } catch (error) {
            PushManagerUtils.log('댓글 구독 오류:', error);
            PushManagerUtils.showToast('댓글 알림 설정에 실패했습니다.', 'error');
            return false;
        }
    },

    /**
     * 푸시 상태 확인
     */
    getStatus() {
        return pushManagerCore.getStatus();
    },

    /**
     * 수동 초기화
     */
    async init(options = {}) {
        return await pushManagerCore.init(options);
    },

    /**
     * 강제 재초기화
     */
    async reinit(options = {}) {
        return await pushManagerCore.reinit(options);
    }
};

// ========================================
// 성능 모니터링
// ========================================

window.pushManagerPerformance = {
    getInitTime() {
        return pushManagerCore.isInitialized ? 'Initialized' : 'Not Initialized';
    },
    
    getAPIStats() {
        return apiMonitor.getRequestStats();
    },
    
    getRecentRequests(count = 10) {
        return Array.from(apiMonitor.requests.values()).slice(-count);
    }
};

// ========================================
// 최종 로그 및 완료
// ========================================

PushManagerUtils.log(`Push Manager Core v${PUSHMANAGER_CONFIG.version} 로드 완료 (최적화 버전)`);

// 개발 모드에서 전역 정보 출력
if (PUSHMANAGER_CONFIG.debug) {
    PushManagerUtils.log('=== 사용 가능한 전역 객체 ===');
    PushManagerUtils.log('• window.PushManager (새로운 통합 API)');
    PushManagerUtils.log('• window.PushManagerCommon (기존 호환성)');
    PushManagerUtils.log('• window.gnuBoard5Push (기존 인스턴스)');
    PushManagerUtils.log('• window.pushManagerHelpers (그누보드5 특화 기능)');
    PushManagerUtils.log('• window.debugPushManager() (디버깅 함수)');
    PushManagerUtils.log('• window.debugAPIRequests() (API 요청 통계)');
    PushManagerUtils.log('• window.pushManagerPerformance (성능 모니터링)');
    PushManagerUtils.log('================================');
    
    // 초기화 상태 모니터링
    const checkInit = () => {
        if (pushManagerCore.isInitialized) {
            PushManagerUtils.log('✅ Push Manager 초기화 완료');
        } else if (pushManagerCore.isInitializing) {
            PushManagerUtils.log('🔄 Push Manager 초기화 진행 중...');
            setTimeout(checkInit, 1000);
        } else {
            PushManagerUtils.log('⏳ Push Manager 초기화 대기 중...');
            setTimeout(checkInit, 1000);
        }
    };
    
    setTimeout(checkInit, 100);
}