/**
 * PWA Components - UI 컴포넌트 모듈
 * PWA 설치 페이지, 설치 팝업, 서버 통신
 */

import { PWAUtils, PWAStorage, SkinLoader, IOSOptimizer } from './pwa-core.js';

// 전역 변수
let deferredPrompt = null;
let pwaServerSettings = null;

// ========================================
// PWA 설치 페이지 클래스
// ========================================

export class PWAInstallPage {
    constructor() {
        this.isInitialized = false;
    }

    /**
     * 설치 페이지 초기화
     */
    async init() {
        if (this.isInitialized) return;

        PWAUtils.log('설치 페이지 초기화 시작');

        try {
            // Service Worker 등록
            await PWAUtils.registerServiceWorker();

            // PWA 설치 프롬프트 설정
            this.setupInstallPrompt();

            // 브라우저별 설치 안내 표시
            this.showInstallInstructions();

            // 이벤트 설정
            this.setupEventListeners();

            this.isInitialized = true;
            PWAUtils.log('설치 페이지 초기화 완료');
        } catch (error) {
            PWAUtils.log('설치 페이지 초기화 실패:', error);
        }
    }

    /**
     * PWA 설치 프롬프트 설정
     */
    setupInstallPrompt() {
        // beforeinstallprompt 이벤트 리스너
        window.addEventListener('beforeinstallprompt', (e) => {
            PWAUtils.log('beforeinstallprompt 이벤트 발생');
            e.preventDefault();
            deferredPrompt = e;
            
            this.showInstallSection('pwa-install-section');
        });

        // 앱 설치 완료 이벤트
        window.addEventListener('appinstalled', () => {
            PWAUtils.log('PWA 앱 설치 완료');
            this.hideAllInstallSections();
            this.updatePageContent('앱이 설치되었습니다', '홈화면에서 앱을 확인하실 수 있습니다.');
            deferredPrompt = null;
        });
    }

    /**
     * 브라우저별 설치 안내 표시
     */
    showInstallInstructions() {
        const device = PWAUtils.detectDevice();
        
        // 이미 PWA 환경에서 실행 중인 경우
        if (PWAUtils.isPWAEnvironment()) {
            PWAUtils.log('이미 설치된 앱에서 실행 중');
            this.updatePageContent('앱이 설치되어 있습니다', '현재 설치된 앱에서 실행 중입니다');
            this.hideAllInstallSections();
            return;
        }
        
        // iOS Safari
        if (device.isIOS && device.isSafari) {
            PWAUtils.log('iOS Safari 감지 - 수동 설치 안내 표시');
            this.showInstallSection('ios-install-section');
        } 
        // Android Chrome
        else if (device.isAndroid && device.isChrome) {
            PWAUtils.log('Android Chrome 감지');
            this.showInstallSection('android-install-section');
            
            // PWA 프롬프트가 없으면 수동 안내 유지
            setTimeout(() => {
                const pwaSection = document.getElementById('pwa-install-section');
                if (pwaSection && pwaSection.classList.contains('hidden')) {
                    PWAUtils.log('PWA 프롬프트 미사용 - Android 수동 안내 유지');
                }
            }, 1000);
        } 
        // Chrome, Edge (PWA 지원)
        else if (device.isChrome || device.isEdge) {
            PWAUtils.log('PWA 지원 브라우저 감지');
            
            setTimeout(() => {
                const pwaSection = document.getElementById('pwa-install-section');
                if (pwaSection && pwaSection.classList.contains('hidden')) {
                    this.showInstallSection('other-browser-section');
                }
            }, 1000);
        } 
        // 기타 브라우저
        else {
            PWAUtils.log('기타 브라우저 - 호환 브라우저 안내');
            this.showInstallSection('other-browser-section');
        }
    }

    /**
     * 설치 섹션 표시
     */
    showInstallSection(sectionId) {
        const section = document.getElementById(sectionId);
        if (section) {
            section.classList.remove('hidden');
        }
    }

    /**
     * 모든 설치 섹션 숨기기
     */
    hideAllInstallSections() {
        const sections = [
            'pwa-install-section',
            'ios-install-section', 
            'android-install-section',
            'other-browser-section'
        ];
        
        sections.forEach(sectionId => {
            const section = document.getElementById(sectionId);
            if (section) {
                section.classList.add('hidden');
            }
        });
    }

    /**
     * 페이지 콘텐츠 업데이트
     */
    updatePageContent(title, description) {
        const titleElement = document.querySelector('.hero-title');
        const descElement = document.querySelector('.hero-description');
        
        if (titleElement) titleElement.textContent = title;
        if (descElement) descElement.textContent = description;
    }

    /**
     * 설치 상태 업데이트
     */
    updateInstallStatus(status) {
        const installButton = document.getElementById('install-button');
        if (installButton) {
            installButton.textContent = status;
            if (status === '설치 완료!') {
                installButton.style.background = 'linear-gradient(135deg, #10b981, #059669)';
            }
        }
    }

    /**
     * 이벤트 리스너 설정
     */
    setupEventListeners() {
        const installButton = document.getElementById('install-button');
        
        if (installButton) {
            installButton.addEventListener('click', async () => {
                await this.handleInstallClick();
            });
        }

        // 가시성 변경 처리
        document.addEventListener('visibilitychange', () => {
            if (!document.hidden && PWAUtils.isPWAEnvironment()) {
                this.updatePageContent('앱이 설치되어 있습니다', '현재 설치된 앱에서 실행 중입니다');
                this.hideAllInstallSections();
            }
        });
    }

    /**
     * 설치 버튼 클릭 처리
     */
    async handleInstallClick() {
        PWAUtils.log('설치 버튼 클릭');
        
        const device = PWAUtils.detectDevice();
        
        // iOS는 항상 안내 페이지로
        if (device.isIOS) {
            this.goToInstallPage();
            return;
        }
        
        // 네이티브 설치 가능한 경우
        if (deferredPrompt) {
            try {
                this.updateInstallStatus('설치 중...');
                
                deferredPrompt.prompt();
                const { outcome } = await deferredPrompt.userChoice;
                
                if (outcome === 'accepted') {
                    PWAUtils.log('사용자가 설치를 승인');
                    this.updateInstallStatus('설치 완료!');
                } else {
                    PWAUtils.log('사용자가 설치를 취소');
                    this.updateInstallStatus('지금 설치하기');
                }
                
                deferredPrompt = null;
            } catch (error) {
                PWAUtils.log('설치 프롬프트 오류:', error);
                this.goToInstallPage();
            }
        } else {
            // 기타 경우 안내 페이지로
            this.goToInstallPage();
        }
    }

    /**
     * 설치 안내 페이지로 이동
     */
    goToInstallPage() {
        const config = PWAUtils.config;
        window.location.href = config.get('installUrl');
    }
}

// ========================================
// PWA 설치 유도 팝업 클래스 (iOS 최적화)
// ========================================

export class PWAInstallPopup {
    constructor(options = {}) {
        this.device = PWAUtils.detectDevice();
        
        // iOS 특화 기본값
        const iOSDefaults = this.device.isIOS ? {
            title: '앱으로 설치하세요!',
            description: 'Safari 하단의 공유 버튼을 누르고 "홈 화면에 추가"를 선택하세요.',
            installButtonText: '설치 방법 보기',
            showDelay: 1500, // iOS는 더 빨리 표시
        } : {};
        
        // 기본값 설정
        this.defaultOptions = {
            title: '앱으로 설치하세요!',
            description: '더 빠르고 편리한 앱 경험을 위해 홈 화면에 설치해보세요.',
            installButtonText: '설치하기',
            laterButtonText: '나중에',
            closeIcon: '×',
            showDelay: 3000,
            dismissHours: 24,
            installUrl: PWAUtils.config.get('installUrl'),
            autoShow: true,
            useServerConfig: true,
            // 스킨 관련 옵션
            skin: 'default',
            skinBasePath: PWAUtils.config.get('skinBasePath'),
            skinFallback: 'default',
            ...iOSDefaults
        };
        
        this.options = { ...this.defaultOptions, ...options };
        this.popup = null;
        this.isInitialized = false;
        this.canInstallNatively = false;
        this.currentSkin = null;
    }

    /**
     * 팝업 초기화 (iOS 최적화)
     */
    async init() {
        if (this.isInitialized) return;

        PWAUtils.log(`설치 팝업 초기화 시작 (iOS: ${this.device.isIOS})`);

        try {
            // iOS viewport 확인
            if (this.device.isIOS) {
                PWAUtils.ensureViewportMeta();
                IOSOptimizer.injectOptimizedCSS();
                IOSOptimizer.setupEventListeners();
            }

            // 서버 설정 사용하는 경우 먼저 로드
            if (this.options.useServerConfig) {
                await this.loadServerConfig();
            }

            // 스킨 로드
            await this.loadSkin();
            
            this.createPopupHTML();
            this.bindEvents();
            
            if (this.options.autoShow) {
                this.checkAndShowPopup();
            }
            
            this.isInitialized = true;
            PWAUtils.log(`설치 팝업 초기화 완료 (iOS: ${this.device.isIOS})`);
        } catch (error) {
            PWAUtils.log('설치 팝업 초기화 실패:', error);
            // 오류 시 기본 설정으로 초기화
            this.options.useServerConfig = false;
            await this.loadSkin();
            this.createPopupHTML();
            this.bindEvents();
            this.isInitialized = true;
        }
    }

    /**
     * 스킨 로드
     */
    async loadSkin() {
        try {
            PWAUtils.log(`스킨 로딩 시작: ${this.options.skin}`);
            
            // 스킨 로드 (폴백 포함)
            this.currentSkin = await SkinLoader.loadSkinWithFallback(
                this.options.skin, 
                this.options.skinBasePath
            );
            
            // 스킨 유효성 검증
            if (!SkinLoader.validateSkin(this.currentSkin)) {
                PWAUtils.log('스킨 유효성 검증 실패, 하드코딩 스킨 사용');
                this.currentSkin = SkinLoader.getHardcodedSkin();
            }
            
            // CSS 주입
            SkinLoader.injectCSS(this.currentSkin.css, this.currentSkin.name);
            
            PWAUtils.log(`스킨 로딩 완료: ${this.currentSkin.name}`);
            
        } catch (error) {
            PWAUtils.log('스킨 로딩 실패, 하드코딩 스킨 사용:', error);
            this.currentSkin = SkinLoader.getHardcodedSkin();
            SkinLoader.injectCSS(this.currentSkin.css, this.currentSkin.name);
        }
    }

    /**
     * 서버 설정 로드
     */
    async loadServerConfig() {
        try {
            const serverData = await PWAUtils.loadPWASettings();
            pwaServerSettings = serverData;
            
            if (serverData && serverData.pwa_config) {
                // 서버 설정으로 옵션 업데이트
                this.options.title = serverData.pwa_config.title || this.options.title;
                this.options.description = serverData.pwa_config.description || this.options.description;
                this.options.installButtonText = serverData.pwa_config.installButtonText || this.options.installButtonText;
                this.options.laterButtonText = serverData.pwa_config.laterButtonText || this.options.laterButtonText;

                // 스킨 설정 적용
                if (serverData.pwa_config.skin) {
                    this.options.skin = serverData.pwa_config.skin;
                }
                
                PWAUtils.log('서버 설정으로 팝업 옵션 업데이트 완료');
            }
        } catch (error) {
            PWAUtils.log('서버 설정 로드 실패:', error);
        }
    }

    /**
     * 팝업 HTML 생성 (스킨 템플릿 사용)
     */
    createPopupHTML() {
        const popupId = 'pwa-install-popup';
        
        // 기존 팝업 제거
        const existingPopup = document.getElementById(popupId);
        if (existingPopup) {
            existingPopup.remove();
        }
        
        if (!this.currentSkin) {
            PWAUtils.log('스킨이 로드되지 않음');
            return;
        }
        
        // 템플릿 변수 설정
        const templateVariables = {
            title: this.options.title,
            description: this.options.description,
            installButtonText: this.options.installButtonText,
            laterButtonText: this.options.laterButtonText,
            closeIcon: this.options.closeIcon
        };
        
        // 템플릿 처리
        const processedHTML = SkinLoader.processTemplate(this.currentSkin.html, templateVariables);
        
        // 팝업 엘리먼트 생성
        const popup = document.createElement('div');
        popup.id = popupId;
        popup.innerHTML = processedHTML;
        
        // DOM에 추가
        document.body.appendChild(popup);
        this.popup = popup.querySelector('.pwa-install-popup') || popup;
        
        PWAUtils.log(`팝업 HTML 생성 완료 (스킨: ${this.currentSkin.name})`);
    }

    /**
     * 팝업 표시 여부 확인 및 표시 (iOS 강화)
     */
    checkAndShowPopup() {
        PWAUtils.log('팝업 표시 조건 확인 시작');
        
        // 1. PWA 환경에서 이미 실행 중인 경우 (iOS 강화 감지)
        if (PWAUtils.isPWAEnvironment()) {
            PWAUtils.log('이미 PWA 환경에서 실행 중 - 팝업 표시 안함');
            return;
        }
        
        // 2. iOS가 아닌 경우 기존 로직 사용
        if (!this.device.isIOS) {
            if (!PWAUtils.isPWASupported()) {
                PWAUtils.log('PWA 미지원 플랫폼 - 팝업 표시 안함');
                return;
            }
        }
        
        // 3. iOS Safari가 아닌 경우
        if (this.device.isIOS && !this.device.isSafari) {
            PWAUtils.log('iOS에서 Safari가 아님 - 팝업 표시 안함');
            return;
        }
        
        // 4. 이전에 팝업을 닫았는지 확인
        if (PWAStorage.isDismissed(this.options.dismissHours)) {
            const dismissInfo = PWAStorage.getDismissedInfo();
            PWAUtils.log(`팝업 무시 기간 중 - ${dismissInfo.hoursSinceDismissed.toFixed(1)}시간 경과`);
            return;
        }
        
        // 5. iOS용 특별 처리
        if (this.device.isIOS) {
            PWAUtils.log('iOS Safari 감지 - 커스텀 설치 안내 팝업 표시');
            this.options.installButtonText = '설치 방법 보기';
        }
        
        // 6. 지연 후 팝업 표시
        PWAUtils.log(`${this.options.showDelay}ms 후 팝업 표시 예정`);
        setTimeout(() => {
            this.showPopup();
        }, this.options.showDelay);
    }

    /**
     * iOS 최적화된 팝업 표시
     */
    showPopup() {
        if (!this.popup) {
            PWAUtils.log('팝업 엘리먼트가 없음 - 초기화 필요');
            return;
        }
        
        // iOS Safari 전용 처리
        if (this.device.isIOS) {
            this.forceShowPopup();
        } else {
            this.popup.classList.add('show');
        }
        
        // 업데이트된 버튼 텍스트 적용
        setTimeout(() => {
            this.updateInstallButton();
        }, 100);
        
        PWAUtils.log(`PWA 설치 팝업 표시 (iOS: ${this.device.isIOS})`);
    }

    /**
     * iOS Safari용 강제 팝업 표시
     */
    forceShowPopup() {
        if (!this.popup) return;
        
        // 클래스 추가
        this.popup.classList.add('show');
        
        // 강제 스타일 적용 (iOS Safari 버그 대응)
        this.popup.style.display = 'flex';
        this.popup.style.position = 'fixed';
        this.popup.style.bottom = '0';
        this.popup.style.left = '0';
        this.popup.style.right = '0';
        this.popup.style.zIndex = '999999';
        this.popup.style.visibility = 'visible';
        this.popup.style.opacity = '1';
        
        // 애니메이션 강제 적용
        this.popup.style.transform = 'translateY(0)';
        this.popup.style.transition = 'transform 0.3s ease, opacity 0.3s ease';
        
        PWAUtils.log('iOS Safari용 강제 팝업 스타일 적용');
    }

    /**
     * 이벤트 바인딩 (iOS 터치 최적화)
     */
    bindEvents() {
        if (!this.popup) return;
        
        // 기존 이벤트 리스너 제거 (중복 방지)
        this.unbindEvents();
        
        // 닫기 버튼
        const closeBtn = this.popup.querySelector('.pwa-close-btn');
        if (closeBtn) {
            this.closeHandler = (e) => {
                e.preventDefault();
                e.stopPropagation();
                this.hidePopup();
            };
            closeBtn.addEventListener('click', this.closeHandler);
            closeBtn.addEventListener('touchend', this.closeHandler);
        }
        
        // 설치 버튼
        const installBtn = this.popup.querySelector('.pwa-install-btn');
        if (installBtn) {
            this.installHandler = (e) => {
                e.preventDefault();
                e.stopPropagation();
                this.handleInstallClick();
            };
            installBtn.addEventListener('click', this.installHandler);
            installBtn.addEventListener('touchend', this.installHandler);
        }
        
        // 나중에 버튼
        const laterBtn = this.popup.querySelector('.pwa-later-btn');
        if (laterBtn) {
            this.laterHandler = (e) => {
                e.preventDefault();
                e.stopPropagation();
                this.dismissPopup();
            };
            laterBtn.addEventListener('click', this.laterHandler);
            laterBtn.addEventListener('touchend', this.laterHandler);
        }
        
        // PWA 관련 이벤트들 (iOS가 아닌 경우만)
        if (!this.device.isIOS && !this.pwaEventsAttached) {
            // beforeinstallprompt 이벤트
            this.beforeInstallHandler = (e) => {
                e.preventDefault();
                deferredPrompt = e;
                this.canInstallNatively = true;
                this.updateInstallButton();
            };
            window.addEventListener('beforeinstallprompt', this.beforeInstallHandler);
            
            // PWA 설치 완료 이벤트
            this.appInstalledHandler = () => {
                PWAStorage.clearDismissed();
                this.hidePopup();
            };
            window.addEventListener('appinstalled', this.appInstalledHandler);
            
            this.pwaEventsAttached = true;
        }
    }

    /**
     * 이벤트 바인딩 해제
     */
    unbindEvents() {
        if (!this.popup) return;
        
        const closeBtn = this.popup.querySelector('.pwa-close-btn');
        const installBtn = this.popup.querySelector('.pwa-install-btn');
        const laterBtn = this.popup.querySelector('.pwa-later-btn');
        
        if (closeBtn && this.closeHandler) {
            closeBtn.removeEventListener('click', this.closeHandler);
            closeBtn.removeEventListener('touchend', this.closeHandler);
        }
        if (installBtn && this.installHandler) {
            installBtn.removeEventListener('click', this.installHandler);
            installBtn.removeEventListener('touchend', this.installHandler);
        }
        if (laterBtn && this.laterHandler) {
            laterBtn.removeEventListener('click', this.laterHandler);
            laterBtn.removeEventListener('touchend', this.laterHandler);
        }
    }

    /**
     * iOS 전용 설치 버튼 업데이트
     */
    updateInstallButton() {
        const installBtn = this.popup?.querySelector('.pwa-install-btn');
        if (!installBtn) return;
        
        if (this.device.isIOS) {
            installBtn.textContent = '설치 방법 보기';
            installBtn.disabled = false;
            PWAUtils.log('iOS 설치 버튼 업데이트됨');
        } else {
            // 기존 로직 (Android/Desktop)
            if (this.canInstallNatively || deferredPrompt) {
                installBtn.textContent = this.options.installButtonText;
                installBtn.disabled = false;
            } else {
                installBtn.textContent = '자세히보기';
                installBtn.disabled = false;
            }
        }
    }

    /**
     * iOS 전용 설치 버튼 클릭 처리
     */
    handleInstallClick() {
        PWAUtils.log('설치 버튼 클릭됨');
        
        if (this.device.isIOS) {
            // iOS는 항상 설치 안내 페이지로 이동
            this.goToInstallPage();
            return;
        }
        
        // 기존 로직 (Android/Desktop)
        if (deferredPrompt) {
            this.installPWA();
        } else {
            this.goToInstallPage();
        }
    }

    /**
     * PWA 네이티브 설치 (Android/Desktop)
     */
    installPWA() {
        if (deferredPrompt) {
            const installBtn = this.popup?.querySelector('.pwa-install-btn');
            if (installBtn) {
                installBtn.disabled = true;
                installBtn.textContent = '설치 중...';
            }
            
            deferredPrompt.prompt();
            deferredPrompt.userChoice.then((choiceResult) => {
                if (choiceResult.outcome === 'accepted') {
                    PWAStorage.clearDismissed();
                    PWAUtils.log('사용자가 설치를 승인');
                } else {
                    if (installBtn) {
                        installBtn.disabled = false;
                        installBtn.textContent = this.options.installButtonText;
                    }
                    PWAUtils.log('사용자가 설치를 취소');
                }
                deferredPrompt = null;
                this.canInstallNatively = false;
            });
            
            this.hidePopup();
        } else {
            this.goToInstallPage();
        }
    }

    /**
     * 설치 안내 페이지로 이동
     */
    goToInstallPage() {
        window.location.href = this.options.installUrl;
    }

    /**
     * 팝업 숨기기 (iOS 최적화)
     */
    hidePopup() {
        if (this.popup) {
            this.popup.classList.remove('show');
            
            // iOS Safari에서 그림자가 남아있는 문제 해결
            if (this.device.isIOS) {
                // 강제로 스타일 초기화
                this.popup.style.display = 'none';
                this.popup.style.visibility = 'hidden';
                this.popup.style.opacity = '0';
                this.popup.style.transform = 'translateY(100%)';
                
                // 잠깐 후 스타일 제거 (CSS 클래스로 돌아가기)
                setTimeout(() => {
                    if (this.popup && !this.popup.classList.contains('show')) {
                        this.popup.style.display = '';
                        this.popup.style.visibility = '';
                        this.popup.style.opacity = '';
                        this.popup.style.transform = '';
                    }
                }, 100);
            }
            
            PWAUtils.log('PWA 설치 팝업 숨김');
        }
    }

    /**
     * 팝업 무시 (나중에)
     */
    dismissPopup() {
        PWAStorage.setDismissed(this.options.dismissHours);
        this.hidePopup();
        PWAUtils.log('PWA 설치 팝업 무시됨');
    }

    /**
     * 팝업 제거
     */
    destroy() {
        // 이벤트 리스너 제거
        this.unbindEvents();
        
        if (this.pwaEventsAttached) {
            window.removeEventListener('beforeinstallprompt', this.beforeInstallHandler);
            window.removeEventListener('appinstalled', this.appInstalledHandler);
            this.pwaEventsAttached = false;
        }
        
        // DOM 요소 제거
        if (this.popup) {
            this.popup.remove();
            this.popup = null;
        }
        
        // 스타일 제거
        if (this.currentSkin) {
            const styleId = `pwa-install-popup-styles-${this.currentSkin.name}`;
            const style = document.getElementById(styleId);
            if (style) {
                style.remove();
            }
        }
        
        this.isInitialized = false;
        PWAUtils.log('PWA 설치 팝업 제거됨');
    }
}

// ========================================
// 서버 통신 관련 함수들
// ========================================

/**
 * 서버 설정 전역 접근 함수
 */
export function getPWAServerSettings() {
    return pwaServerSettings;
}

/**
 * deferredPrompt 전역 접근 함수
 */
export function getDeferredPrompt() {
    return deferredPrompt;
}

/**
 * deferredPrompt 설정 함수
 */
export function setDeferredPrompt(prompt) {
    deferredPrompt = prompt;
}