/**
 * PWA Core - 핵심 기반 모듈
 * 설정 관리, 유틸리티, 스킨 로더, iOS 최적화
 */

// ========================================
// PWA 설정 관리
// ========================================

export class PWAConfig {
    static instance = null;
    
    constructor() {
        if (PWAConfig.instance) {
            return PWAConfig.instance;
        }
        
        this.config = {
            version: '0.0.5',
            installUrl: (typeof g5_url !== 'undefined' ? g5_url : '') + '/plugin/pushmanager/pwa_install.php',
            manifestUrl: (typeof g5_url !== 'undefined' ? g5_url : '') + '/plugin/pushmanager/manifest.json',
            serviceWorkerUrl: (typeof g5_url !== 'undefined' ? g5_url : '') + '/plugin/pushmanager/js/service-worker.js',
            ajaxUrl: (typeof g5_url !== 'undefined' ? g5_url : '') + '/plugin/pushmanager/pushmanager_ajax.php?action=get_pwa_config',
            skinBasePath: (typeof g5_url !== 'undefined' ? g5_url : '') + '/plugin/pushmanager/js/popup/',
            debug: false
        };
        
        PWAConfig.instance = this;
    }
    
    get(key) {
        return this.config[key];
    }
    
    set(key, value) {
        this.config[key] = value;
    }
    
    getAll() {
        return { ...this.config };
    }
}

// ========================================
// PWA 스토리지 관리
// ========================================

export class PWAStorage {
    static setDismissed(hours = 24) {
        localStorage.setItem('pwa-install-dismissed', 'true');
        localStorage.setItem('pwa-install-dismissed-time', Date.now().toString());
        PWAUtils.log(`PWA 팝업 ${hours}시간 무시 설정`);
    }
    
    static isDismissed(hours = 24) {
        const dismissed = localStorage.getItem('pwa-install-dismissed');
        const dismissedTime = localStorage.getItem('pwa-install-dismissed-time');
        
        if (!dismissed || !dismissedTime) return false;
        
        const timeDiff = Date.now() - parseInt(dismissedTime);
        const hoursDiff = timeDiff / (1000 * 60 * 60);
        
        return hoursDiff < hours;
    }
    
    static clearDismissed() {
        localStorage.removeItem('pwa-install-dismissed');
        localStorage.removeItem('pwa-install-dismissed-time');
        PWAUtils.log('PWA 무시 데이터 삭제');
    }
    
    static getDismissedInfo() {
        const dismissed = localStorage.getItem('pwa-install-dismissed');
        const dismissedTime = localStorage.getItem('pwa-install-dismissed-time');
        
        if (!dismissed || !dismissedTime) return null;
        
        const timeDiff = Date.now() - parseInt(dismissedTime);
        const hoursDiff = timeDiff / (1000 * 60 * 60);
        
        return {
            dismissed: dismissed === 'true',
            dismissedTime: parseInt(dismissedTime),
            hoursSinceDismissed: hoursDiff
        };
    }
}

// ========================================
// PWA 유틸리티 클래스 (iOS 최적화)
// ========================================

export class PWAUtils {
    static config = new PWAConfig();
    
    /**
     * 개선된 디바이스 및 브라우저 감지 (iOS 특화)
     */
    static detectDevice() {
        const userAgent = navigator.userAgent.toLowerCase();
        
        const device = {
            isIOS: /ipad|iphone|ipod/.test(userAgent),
            isAndroid: /android/.test(userAgent),
            isSafari: /safari/.test(userAgent) && !/chrome/.test(userAgent),
            isChrome: /chrome/.test(userAgent),
            isEdge: /edge/.test(userAgent),
            isFirefox: /firefox/.test(userAgent),
            isSamsungBrowser: /samsung/.test(userAgent) || /samsungbrowser/.test(userAgent),
            get isMobile() { return this.isIOS || this.isAndroid; },
            get isDesktop() { return !this.isMobile; }
        };
        
        // iOS 버전 감지
        if (device.isIOS) {
            const match = userAgent.match(/os (\d+)_/);
            device.iOSVersion = match ? parseInt(match[1]) : null;
        }
        
        return device;
    }

    /**
     * 개선된 PWA 환경 감지 (iOS 특화) - 디버깅 강화
     */
    static isPWAEnvironment() {
        const device = PWAUtils.detectDevice();
        
        // 디버깅을 위한 상세 로그
        PWAUtils.log('=== PWA 환경 감지 시작 ===');
        PWAUtils.log('디바이스:', device);
        
        // iOS PWA 환경 감지 (다중 검증)
        if (device.isIOS) {
            PWAUtils.log('iOS 디바이스 감지됨');
            
            // 1. navigator.standalone 확인 (가장 확실한 방법)
            const isStandalone = window.navigator.standalone;
            PWAUtils.log('navigator.standalone:', isStandalone);
            if (isStandalone === true) {
                PWAUtils.log('✅ iOS PWA 환경 감지: navigator.standalone = true');
                return true;
            }
            
            // 2. display-mode 미디어 쿼리 확인 (iOS 13+)
            const isDisplayModeStandalone = window.matchMedia && window.matchMedia('(display-mode: standalone)').matches;
            PWAUtils.log('display-mode standalone:', isDisplayModeStandalone);
            if (isDisplayModeStandalone) {
                PWAUtils.log('✅ iOS PWA 환경 감지: display-mode standalone');
                return true;
            }
            
            // 3. URL 파라미터 확인 (manifest의 start_url 파라미터)
            const urlParams = new URLSearchParams(window.location.search);
            const standaloneParam = urlParams.get('standalone');
            PWAUtils.log('URL standalone 파라미터:', standaloneParam);
            if (standaloneParam === 'true') {
                PWAUtils.log('✅ iOS PWA 환경 감지: URL 파라미터');
                return true;
            }
            
            // 4. 추가 iOS 감지 방법들
            const isInWebAppiOS = window.navigator.standalone !== undefined;
            const isFullscreen = window.screen.height === window.innerHeight;
            PWAUtils.log('WebApp 지원:', isInWebAppiOS);
            PWAUtils.log('전체화면 여부:', isFullscreen);
            PWAUtils.log('viewport 높이:', window.innerHeight);
            PWAUtils.log('화면 높이:', screen.height);
            
            // 5. viewport 높이 확인 (Safari 주소창이 없는 경우) - 더 관대한 기준
            if (device.iOSVersion >= 13) {
                const viewportHeight = window.innerHeight;
                const screenHeight = screen.height;
                const heightDiff = Math.abs(viewportHeight - screenHeight);
                PWAUtils.log('높이 차이:', heightDiff);
                
                // 주소창이 없으면 viewport가 화면 높이와 거의 같음 (더 관대한 기준)
                if (heightDiff < 150) {
                    PWAUtils.log('⚠️ iOS PWA 환경 의심: viewport 높이 분석 (차이:', heightDiff, ')');
                    // 이 경우는 확신이 없으므로 false 반환하지만 로그만 남김
                }
            }
            
            PWAUtils.log('❌ iOS PWA 환경 감지 실패 - 일반 Safari로 판단');
            return false;
        }
        
        // Android PWA 환경 감지
        const isAndroidStandalone = window.matchMedia && window.matchMedia('(display-mode: standalone)').matches;
        PWAUtils.log('Android display-mode standalone:', isAndroidStandalone);
        if (isAndroidStandalone) {
            PWAUtils.log('✅ Android PWA 환경 감지');
            return true;
        }
        
        // 기타 PWA 환경
        const isAndroidApp = document.referrer.includes('android-app://');
        PWAUtils.log('Android app referrer:', isAndroidApp);
        if (isAndroidApp) {
            PWAUtils.log('✅ Android App PWA 환경 감지');
            return true;
        }
        
        PWAUtils.log('❌ PWA 환경 아님 - 일반 브라우저');
        return false;
    }

    /**
     * PWA 설치 지원 여부 확인 (iOS 특화)
     */
    static isPWASupported() {
        const device = PWAUtils.detectDevice();
        
        // iOS는 항상 PWA 지원 (수동 설치)
        if (device.isIOS) {
            return true;
        }
        
        return (
            'serviceWorker' in navigator &&
            'PushManager' in window &&
            'Notification' in window &&
            ('BeforeInstallPromptEvent' in window || 
             device.isSamsungBrowser)
        );
    }

    /**
     * Service Worker 등록
     */
    static async registerServiceWorker() {
        if (!('serviceWorker' in navigator)) {
            throw new Error('Service Worker가 지원되지 않습니다.');
        }

        try {
            const registration = await navigator.serviceWorker.register(PWAUtils.config.get('serviceWorkerUrl'));
            PWAUtils.log('Service Worker 등록 성공:', registration.scope);
            return registration;
        } catch (error) {
            PWAUtils.log('Service Worker 등록 실패:', error);
            throw error;
        }
    }

    /**
     * 서버에서 PWA 설정 가져오기
     */
    static async loadPWASettings() {
        try {
            PWAUtils.log('서버 PWA 설정 로드 시작...');
            
            const response = await fetch(PWAUtils.config.get('ajaxUrl'), {
                method: 'GET',
                headers: {
                    'Accept': 'application/json',
                    'Cache-Control': 'no-cache'
                }
            });

            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }

            const data = await response.json();
            
            if (data.success && data.pwa_config) {
                PWAUtils.log('서버 PWA 설정 로드 완료:', {
                    title: data.pwa_config.title,
                    description: data.pwa_config.description
                });
                return data;
            } else {
                throw new Error('서버 응답 형식 오류: ' + (data.error || '알 수 없는 오류'));
            }
        } catch (error) {
            PWAUtils.log('서버 설정 로드 실패, 기본값 사용:', error.message);
            
            const device = PWAUtils.detectDevice();
            
            // iOS용 기본값
            if (device.isIOS) {
                return {
                    success: false,
                    pwa_config: {
                        title: '앱으로 설치하세요!',
                        description: 'Safari 하단의 공유 버튼을 누르고 "홈 화면에 추가"를 선택하세요.',
                        installButtonText: '설치 방법 보기',
                        laterButtonText: '나중에'
                    }
                };
            }
            
            // 기타 플랫폼용 기본값
            return {
                success: false,
                pwa_config: {
                    title: '앱 설치 안내',
                    description: '홈 화면에 추가하여 앱처럼 편리하게 이용하세요!',
                    installButtonText: '설치하기',
                    laterButtonText: '나중에'
                }
            };
        }
    }

    /**
     * iOS 전용 viewport 메타태그 확인
     */
    static ensureViewportMeta() {
        let viewportMeta = document.querySelector('meta[name="viewport"]');
        if (!viewportMeta) {
            viewportMeta = document.createElement('meta');
            viewportMeta.name = 'viewport';
            viewportMeta.content = 'width=device-width, initial-scale=1.0, user-scalable=no, viewport-fit=cover';
            document.head.appendChild(viewportMeta);
            PWAUtils.log('viewport 메타태그 추가됨');
        }
    }

    /**
     * PWA 환경에 body 클래스 추가 (iOS Safari 제외) - 디버깅 강화
     */
    static addPWABodyClass() {
        const device = PWAUtils.detectDevice();
        const isPWAEnv = PWAUtils.isPWAEnvironment();
        
        PWAUtils.log('=== PWA Body 클래스 추가 시도 ===');
        PWAUtils.log('디바이스:', device);
        PWAUtils.log('PWA 환경:', isPWAEnv);
        
        // iOS Safari에서는 실행하지 않음 (더 정확한 조건)
        if (device.isIOS && device.isSafari && !isPWAEnv) {
            PWAUtils.log('❌ iOS Safari 일반 브라우저 - PWA body 클래스 추가 안함');
            return;
        }
        
        // PWA 환경에서만 실행
        if (!isPWAEnv) {
            PWAUtils.log('❌ PWA 환경이 아님 - body 클래스 추가 안함');
            return;
        }
        
        const body = document.body;
        if (!body) {
            PWAUtils.log('❌ body 요소를 찾을 수 없음');
            return;
        }
        
        PWAUtils.log('✅ PWA 환경 확인됨 - body 클래스 추가 시작');
        
        // 기본 pwa 클래스 추가
        body.classList.add('pwa');
        PWAUtils.log('기본 pwa 클래스 추가됨');
        
        // 플랫폼별 클래스 추가
        if (device.isMobile) {
            body.classList.add('pwa-mobile');
            PWAUtils.log('pwa-mobile 클래스 추가됨');
            
            // iOS/Android 구분
            if (device.isIOS) {
                body.classList.add('pwa-ios');
                PWAUtils.log('pwa-ios 클래스 추가됨');
            } else if (device.isAndroid) {
                body.classList.add('pwa-android');
                PWAUtils.log('pwa-android 클래스 추가됨');
            }
            
            PWAUtils.log('✅ PWA 모바일 환경 - 모든 클래스 추가 완료');
        } else {
            body.classList.add('pwa-desktop');
            PWAUtils.log('✅ PWA 데스크탑 환경 - pwa, pwa-desktop 클래스 추가');
        }
        
        // 최종 body 클래스 확인
        const finalClasses = Array.from(body.classList);
        PWAUtils.log('최종 body 클래스:', finalClasses);
    }
    
    /**
     * PWA 환경 체크 및 body 클래스 제거
     */
    static removePWABodyClass() {
        const body = document.body;
        if (!body) return;
        
        const pwaClasses = [
            'pwa',
            'pwa-mobile',
            'pwa-desktop',
            'pwa-ios',
            'pwa-android'
        ];
        
        pwaClasses.forEach(className => {
            body.classList.remove(className);
        });
        
        PWAUtils.log('PWA body 클래스 제거됨');
    }
    
    /**
     * PWA 환경 변화 감지 및 클래스 업데이트
     */
    static updatePWABodyClass() {
        // 기존 클래스 제거
        PWAUtils.removePWABodyClass();
        
        // PWA 환경이면 클래스 추가
        if (PWAUtils.isPWAEnvironment()) {
            PWAUtils.addPWABodyClass();
        }
    }

    /**
     * 모바일 디버깅용 정보 표시 (임시)
     */
    static showMobileDebugInfo() {
        // 기존 디버그 정보 제거
        const existingDebug = document.getElementById('pwa-mobile-debug');
        if (existingDebug) {
            existingDebug.remove();
        }
        
        const device = PWAUtils.detectDevice();
        const isPWAEnv = PWAUtils.isPWAEnvironment();
        const bodyClasses = Array.from(document.body.classList);
        
        // 디버그 정보 HTML 생성
        const debugHTML = `
            <div id="pwa-mobile-debug" style="
                position: fixed;
                top: 10px;
                left: 10px;
                right: 10px;
                background: rgba(0,0,0,0.9);
                color: white;
                padding: 15px;
                border-radius: 8px;
                font-family: monospace;
                font-size: 12px;
                z-index: 999999;
                max-height: 50vh;
                overflow-y: auto;
                box-shadow: 0 4px 20px rgba(0,0,0,0.5);
            ">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                    <strong style="color: #4CAF50;">PWA 디버그 정보</strong>
                    <button onclick="document.getElementById('pwa-mobile-debug').remove()" style="
                        background: #f44336;
                        color: white;
                        border: none;
                        padding: 5px 10px;
                        border-radius: 4px;
                        cursor: pointer;
                        font-size: 12px;
                    ">닫기</button>
                </div>
                
                <div style="margin-bottom: 8px;">
                    <strong style="color: #2196F3;">디바이스:</strong> 
                    ${device.isIOS ? 'iOS' : device.isAndroid ? 'Android' : 'Desktop'}
                    ${device.isMobile ? ' (Mobile)' : ' (Desktop)'}
                </div>
                
                <div style="margin-bottom: 8px;">
                    <strong style="color: #2196F3;">브라우저:</strong> 
                    ${device.isSafari ? 'Safari' : device.isChrome ? 'Chrome' : device.isFirefox ? 'Firefox' : device.isEdge ? 'Edge' : '기타'}
                </div>
                
                <div style="margin-bottom: 8px;">
                    <strong style="color: ${isPWAEnv ? '#4CAF50' : '#FF9800'};">PWA 환경:</strong> 
                    ${isPWAEnv ? '✅ PWA로 실행 중' : '❌ 일반 브라우저'}
                </div>
                
                ${device.isIOS ? `
                <div style="margin-bottom: 8px;">
                    <strong style="color: #2196F3;">iOS 상세 정보:</strong><br>
                    • standalone: ${window.navigator.standalone}<br>
                    • display-mode: ${window.matchMedia('(display-mode: standalone)').matches}<br>
                    • 버전: iOS ${device.iOSVersion || '알 수 없음'}<br>
                    • viewport: ${window.innerHeight}px<br>
                    • screen: ${screen.height}px<br>
                    • 높이차: ${Math.abs(window.innerHeight - screen.height)}px<br>
                    • URL standalone: ${new URLSearchParams(window.location.search).get('standalone') || 'null'}
                </div>
                ` : ''}
                
                <div style="margin-bottom: 8px;">
                    <strong style="color: #2196F3;">Body 클래스:</strong><br>
                    ${bodyClasses.length > 0 ? bodyClasses.map(cls => 
                        cls.startsWith('pwa') ? `<span style="color: #4CAF50;">${cls}</span>` : cls
                    ).join(', ') : '없음'}
                </div>
                
                <div style="margin-bottom: 8px;">
                    <strong style="color: #2196F3;">URL:</strong><br>
                    <span style="font-size: 10px; word-break: break-all;">${window.location.href}</span>
                </div>
                
                <div style="margin-bottom: 8px;">
                    <strong style="color: #2196F3;">User Agent:</strong><br>
                    <span style="font-size: 10px; word-break: break-all;">${navigator.userAgent}</span>
                </div>
                
                <div style="margin-top: 10px; padding-top: 10px; border-top: 1px solid #444;">
                    <button onclick="window.updatePWABodyClass(); window.showMobileDebugInfo();" style="
                        background: #4CAF50;
                        color: white;
                        border: none;
                        padding: 8px 12px;
                        border-radius: 4px;
                        cursor: pointer;
                        font-size: 12px;
                        margin-right: 5px;
                    ">새로고침</button>
                    
                    <button onclick="window.addPWABodyClass(); window.showMobileDebugInfo();" style="
                        background: #2196F3;
                        color: white;
                        border: none;
                        padding: 8px 12px;
                        border-radius: 4px;
                        cursor: pointer;
                        font-size: 12px;
                        margin-right: 5px;
                    ">강제 추가</button>
                    
                    <button onclick="window.removePWABodyClass(); window.showMobileDebugInfo();" style="
                        background: #FF9800;
                        color: white;
                        border: none;
                        padding: 8px 12px;
                        border-radius: 4px;
                        cursor: pointer;
                        font-size: 12px;
                    ">클래스 제거</button>
                </div>
            </div>
        `;
        
        // DOM에 추가
        document.body.insertAdjacentHTML('beforeend', debugHTML);
        
        PWAUtils.log('모바일 디버그 정보 표시됨');
    }
    
    /**
     * 모바일 디버깅용 정보 숨기기
     */
    static hideMobileDebugInfo() {
        const debugElement = document.getElementById('pwa-mobile-debug');
        if (debugElement) {
            debugElement.remove();
            PWAUtils.log('모바일 디버그 정보 숨김');
        }
    }

    /**
     * 로그 출력
     */
    static log(...args) {
        if (PWAUtils.config.get('debug')) {
            console.log('[PWACore]', ...args);
        }
    }
}

// ========================================
// 스킨 로더 클래스
// ========================================

export class SkinLoader {
    /**
     * 스킨 로드
     */
    static async loadSkin(skinName, basePath = null) {
        const config = new PWAConfig();
        if (!basePath) basePath = config.get('skinBasePath');
        
        try {
            PWAUtils.log(`스킨 로딩 시작: ${skinName}`);
            
            const skinPath = `${basePath}${skinName}/`;
            const htmlPath = `${skinPath}popup.html`;
            const cssPath = `${skinPath}popup.css`;
            
            // HTML과 CSS 병렬 로드
            const [htmlResponse, cssResponse] = await Promise.all([
                fetch(htmlPath),
                fetch(cssPath)
            ]);
            
            if (!htmlResponse.ok || !cssResponse.ok) {
                throw new Error(`스킨 파일 로드 실패: HTML(${htmlResponse.status}), CSS(${cssResponse.status})`);
            }
            
            const htmlContent = await htmlResponse.text();
            const cssContent = await cssResponse.text();
            
            PWAUtils.log(`스킨 로딩 완료: ${skinName}`);
            
            return {
                name: skinName,
                html: htmlContent,
                css: cssContent,
                path: skinPath
            };
            
        } catch (error) {
            PWAUtils.log(`스킨 로딩 실패: ${skinName}`, error);
            throw error;
        }
    }
    
    /**
     * 폴백 스킨 로드 (default -> 하드코딩)
     */
    static async loadSkinWithFallback(skinName, basePath = null) {
        // 1차: 요청된 스킨 시도
        if (skinName !== 'default') {
            try {
                return await SkinLoader.loadSkin(skinName, basePath);
            } catch (error) {
                PWAUtils.log(`스킨 ${skinName} 로드 실패, default 스킨으로 폴백`);
            }
        }
        
        // 2차: default 스킨 시도
        try {
            return await SkinLoader.loadSkin('default', basePath);
        } catch (error) {
            PWAUtils.log('default 스킨 로드 실패, 하드코딩 스킨 사용');
            
            // 3차: 하드코딩된 기본 스킨
            return SkinLoader.getHardcodedSkin();
        }
    }
    
    /**
     * 하드코딩된 기본 스킨 (폴백용) - iOS 최적화
     */
    static getHardcodedSkin() {
        const html = `
            <div class="pwa-install-popup">
                <div class="pwa-popup-content">
                    <button class="pwa-close-btn">{{closeIcon}}</button>
                    
                    <div class="pwa-text-content">
                        <h2 class="pwa-popup-title">{{title}}</h2>
                        <p class="pwa-popup-description">{{description}}</p>
                    </div>
                    
                    <div class="pwa-button-group">
                        <button class="pwa-install-btn">{{installButtonText}}</button>
                        <button class="pwa-later-btn">{{laterButtonText}}</button>
                    </div>
                </div>
            </div>
        `;
        
        const css = `
            .pwa-install-popup {
                position: fixed !important;
                bottom: 0 !important;
                left: 0 !important;
                right: 0 !important;
                width: 100% !important;
                background: white !important;
                display: none !important;
                z-index: 999999 !important;
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif !important;
                box-shadow: 0 -4px 20px rgba(0,0,0,0.15) !important;
                border-top: 1px solid #e0e0e0 !important;
                transform: translateY(100%) !important;
                transition: transform 0.3s ease, opacity 0.3s ease, visibility 0.3s ease !important;
                opacity: 0 !important;
                visibility: hidden !important;
                padding-bottom: env(safe-area-inset-bottom, 0px) !important;
                -webkit-tap-highlight-color: transparent !important;
                -webkit-touch-callout: none !important;
                -webkit-user-select: none !important;
                user-select: none !important;
                -webkit-backface-visibility: hidden !important;
                backface-visibility: hidden !important;
            }
            
            .pwa-install-popup.show {
                display: flex !important;
                transform: translateY(0) !important;
                opacity: 1 !important;
                visibility: visible !important;
            }
            
            .pwa-install-popup:not(.show) {
                display: none !important;
                visibility: hidden !important;
                opacity: 0 !important;
                transform: translateY(100%) !important;
                box-shadow: none !important;
            }
            
            .pwa-popup-content {
                background: white !important;
                padding: 1rem 1.2rem !important;
                width: 100% !important;
                display: flex !important;
                align-items: center !important;
                justify-content: space-between !important;
                position: relative !important;
                padding-left: max(1.2rem, env(safe-area-inset-left)) !important;
                padding-right: max(1.2rem, env(safe-area-inset-right)) !important;
            }
            
            .pwa-close-btn {
                position: absolute !important;
                top: 8px !important;
                right: 12px !important;
                background: none !important;
                border: none !important;
                font-size: 20px !important;
                cursor: pointer !important;
                color: #999 !important;
                transition: color 0.3s !important;
                z-index: 1 !important;
                padding: 4px !important;
                -webkit-appearance: none !important;
                touch-action: manipulation !important;
                min-width: 44px !important;
                min-height: 44px !important;
            }
            
            .pwa-close-btn:hover {
                color: #333 !important;
            }
            
            .pwa-popup-title {
                font-size: 1rem !important;
                margin: 0 0 0.3rem 0 !important;
                color: #333 !important;
                font-weight: 600 !important;
            }
            
            .pwa-popup-description {
                color: #666 !important;
                margin: 0 !important;
                line-height: 1.4 !important;
                font-size: 0.85rem !important;
            }
            
            .pwa-text-content {
                flex: 1 !important;
                padding-right: 1rem !important;
            }
            
            .pwa-button-group {
                display: flex !important;
                gap: 8px !important;
                flex-shrink: 0 !important;
            }
            
            .pwa-install-btn {
                background: #007AFF !important;
                color: white !important;
                border: none !important;
                padding: 8px 16px !important;
                border-radius: 6px !important;
                font-size: 0.9rem !important;
                font-weight: 500 !important;
                cursor: pointer !important;
                transition: background-color 0.3s !important;
                white-space: nowrap !important;
                -webkit-appearance: none !important;
                touch-action: manipulation !important;
                min-height: 44px !important;
            }
            
            .pwa-install-btn:hover {
                background: #0056D6 !important;
            }
            
            .pwa-install-btn:disabled {
                background: #ccc !important;
                cursor: not-allowed !important;
            }
            
            .pwa-later-btn {
                background: transparent !important;
                color: #666 !important;
                border: 1px solid #ddd !important;
                padding: 8px 16px !important;
                border-radius: 6px !important;
                font-size: 0.9rem !important;
                cursor: pointer !important;
                transition: all 0.3s !important;
                white-space: nowrap !important;
                -webkit-appearance: none !important;
                touch-action: manipulation !important;
                min-height: 44px !important;
            }
            
            .pwa-later-btn:hover {
                background: #f5f5f5 !important;
            }
            
            @media (max-width: 480px) {
                .pwa-popup-content {
                    flex-direction: column !important;
                    align-items: flex-start !important;
                    padding: 1rem !important;
                }
                
                .pwa-text-content {
                    padding-right: 0 !important;
                    margin-bottom: 1rem !important;
                }
                
                .pwa-button-group {
                    width: 100% !important;
                    justify-content: space-between !important;
                }
                
                .pwa-install-btn, .pwa-later-btn {
                    flex: 1 !important;
                }
            }
        `;
        
        return {
            name: 'hardcoded',
            html: html,
            css: css,
            path: 'hardcoded'
        };
    }
    
    /**
     * 템플릿 변수 치환
     */
    static processTemplate(template, variables) {
        let processed = template;
        
        for (const [key, value] of Object.entries(variables)) {
            const regex = new RegExp(`{{${key}}}`, 'g');
            processed = processed.replace(regex, value);
        }
        
        return processed;
    }
    
    /**
     * CSS를 DOM에 주입
     */
    static injectCSS(css, skinName) {
        const styleId = `pwa-install-popup-styles-${skinName}`;
        
        // 기존 스타일 제거
        const existingStyle = document.getElementById(styleId);
        if (existingStyle) {
            existingStyle.remove();
        }
        
        // 새 스타일 추가
        const style = document.createElement('style');
        style.id = styleId;
        style.textContent = css;
        document.head.appendChild(style);
        
        PWAUtils.log(`CSS 주입 완료: ${skinName}`);
    }
    
    /**
     * 스킨 유효성 검증
     */
    static validateSkin(skin) {
        const requiredClasses = [
            'pwa-install-popup',
            'pwa-close-btn',
            'pwa-install-btn',
            'pwa-later-btn'
        ];
        
        // HTML에 필수 클래스가 있는지 확인
        for (const className of requiredClasses) {
            if (!skin.html.includes(className)) {
                PWAUtils.log(`스킨 검증 실패: 필수 클래스 ${className} 누락`);
                return false;
            }
        }
        
        PWAUtils.log(`스킨 검증 통과: ${skin.name}`);
        return true;
    }
}

// ========================================
// iOS 최적화 유틸리티
// ========================================

export class IOSOptimizer {
    /**
     * iOS 최적화 CSS 주입
     */
    static injectOptimizedCSS() {
        const device = PWAUtils.detectDevice();
        if (!device.isIOS) return;
        
        const styleId = 'pwa-ios-optimized-styles';
        
        // 기존 스타일 제거
        const existingStyle = document.getElementById(styleId);
        if (existingStyle) {
            existingStyle.remove();
        }
        
        // iOS 전용 CSS
        const iOSOptimizedCSS = `
            /* iOS Safari 전용 최적화 */
            .pwa-install-popup {
                -webkit-transform: translateY(100%) !important;
                -webkit-transition: -webkit-transform 0.3s ease, opacity 0.3s ease, visibility 0.3s ease !important;
                -webkit-backface-visibility: hidden !important;
                backface-visibility: hidden !important;
            }
            
            .pwa-install-popup.show {
                -webkit-transform: translateY(0) !important;
            }
            
            .pwa-install-popup:not(.show) {
                display: none !important;
                visibility: hidden !important;
                opacity: 0 !important;
                -webkit-transform: translateY(100%) !important;
                transform: translateY(100%) !important;
                box-shadow: none !important;
                -webkit-box-shadow: none !important;
            }
            
            .pwa-install-popup * {
                -webkit-tap-highlight-color: transparent !important;
                -webkit-touch-callout: none !important;
                -webkit-user-select: none !important;
                user-select: none !important;
            }
            
            .pwa-install-btn, .pwa-later-btn, .pwa-close-btn {
                cursor: pointer !important;
                -webkit-appearance: none !important;
                touch-action: manipulation !important;
                min-width: 44px !important;
                min-height: 44px !important;
            }
            
            @supports (padding: max(0px)) {
                .pwa-popup-content {
                    padding-left: max(1.2rem, env(safe-area-inset-left)) !important;
                    padding-right: max(1.2rem, env(safe-area-inset-right)) !important;
                    padding-bottom: max(1rem, env(safe-area-inset-bottom)) !important;
                }
            }
            
            @media screen and (max-height: 680px) {
                .pwa-install-popup {
                    bottom: env(safe-area-inset-bottom, 0px) !important;
                }
            }
        `;
        
        // 새 스타일 추가
        const style = document.createElement('style');
        style.id = styleId;
        style.textContent = iOSOptimizedCSS;
        document.head.appendChild(style);
        
        PWAUtils.log('iOS 최적화 CSS 주입 완료');
    }
    
    /**
     * iOS 이벤트 리스너 설정
     */
    static setupEventListeners() {
        const device = PWAUtils.detectDevice();
        if (!device.isIOS) return;
        
        // 화면 회전 시 팝업 위치 재조정
        window.addEventListener('orientationchange', function() {
            setTimeout(() => {
                const popup = document.querySelector('.pwa-install-popup.show');
                if (popup) {
                    // 강제 리플로우
                    popup.style.display = 'none';
                    popup.offsetHeight; // 강제 리플로우 트리거
                    popup.style.display = 'flex';
                    PWAUtils.log('iOS 화면 회전 후 팝업 위치 재조정');
                }
            }, 500);
        });
        
        // Safari 주소창 숨김/표시 감지
        if (device.isSafari) {
            let lastHeight = window.innerHeight;
            
            window.addEventListener('resize', function() {
                const currentHeight = window.innerHeight;
                const heightDiff = Math.abs(currentHeight - lastHeight);
                
                // 주소창 숨김/표시로 인한 높이 변화 감지 (50px 이상)
                if (heightDiff > 50) {
                    const popup = document.querySelector('.pwa-install-popup.show');
                    if (popup) {
                        // 팝업 위치 재조정
                        popup.style.bottom = '0px';
                        PWAUtils.log('iOS Safari 주소창 변화 감지, 팝업 위치 재조정');
                    }
                }
                
                lastHeight = currentHeight;
            });
        }
        
        PWAUtils.log('iOS 이벤트 리스너 설정 완료');
    }
    
    /**
     * iOS 디버깅 정보
     */
    static getDebugInfo() {
        const device = PWAUtils.detectDevice();
        if (!device.isIOS) return null;
        
        return {
            device: device,
            isPWAEnvironment: PWAUtils.isPWAEnvironment(),
            standalone: window.navigator.standalone,
            displayMode: window.matchMedia('(display-mode: standalone)').matches,
            viewportHeight: window.innerHeight,
            screenHeight: screen.height,
            userAgent: navigator.userAgent,
            dismissInfo: PWAStorage.getDismissedInfo()
        };
    }
}