/**
 * PWA Installer - 메인 통합 파일 (모듈형)
 * 전역 API 제공, 자동 초기화, 하위 호환성 유지
 */

import { PWAUtils, PWAStorage, SkinLoader, IOSOptimizer, PWAConfig } from './pwa-core.js';
import { PWAInstallPage, PWAInstallPopup, getPWAServerSettings, getDeferredPrompt } from './pwa-components.js';

// ========================================
// 메인 PWA Installer 클래스 (iOS 최적화)
// ========================================

class PWAInstaller {
    constructor() {
        this.installPage = null;
        this.installPopup = null;
        this.isPageMode = false;
        this.isPopupMode = false;
        this.device = PWAUtils.detectDevice();
        this.config = new PWAConfig();
    }

    /**
     * 설치 페이지 모드로 초기화
     */
    async initInstallPage() {
        if (this.isPageMode) return this.installPage;

        PWAUtils.log('PWA 설치 페이지 모드 초기화');
        
        this.installPage = new PWAInstallPage();
        await this.installPage.init();
        
        this.isPageMode = true;
        return this.installPage;
    }

    /**
     * 설치 유도 팝업 모드로 초기화 (iOS 최적화)
     */
    async initPopup(options = {}) {
        if (this.isPopupMode) return this.installPopup;

        PWAUtils.log(`PWA 설치 팝업 모드 초기화 (iOS: ${this.device.isIOS})`);
        
        this.installPopup = new PWAInstallPopup(options);
        await this.installPopup.init();
        
        this.isPopupMode = true;
        return this.installPopup;
    }

    /**
     * 팝업 수동 표시 (서버 설정 새로고침 포함)
     */
    async showPopup(options = {}) {
        if (!this.isPopupMode) {
            await this.initPopup(options);
        }
        
        if (this.installPopup) {
            // 서버 설정 새로고침
            if (this.installPopup.options.useServerConfig) {
                await this.installPopup.loadServerConfig();
                this.installPopup.createPopupHTML();
                this.installPopup.bindEvents();
            }
            this.installPopup.showPopup();
        }
    }

    /**
     * 팝업 숨기기
     */
    hidePopup() {
        if (this.installPopup) {
            this.installPopup.hidePopup();
        }
    }

    /**
     * 현재 디바이스 정보 반환
     */
    getDeviceInfo() {
        return {
            device: this.device,
            isPWASupported: PWAUtils.isPWASupported(),
            isPWAEnvironment: PWAUtils.isPWAEnvironment(),
            canInstallNatively: !!getDeferredPrompt()
        };
    }

    /**
     * PWA 설치 상태 확인
     */
    checkInstallStatus() {
        return {
            isInstalled: PWAUtils.isPWAEnvironment(),
            canInstall: PWAUtils.isPWASupported() && !PWAUtils.isPWAEnvironment(),
            hasPrompt: !!getDeferredPrompt()
        };
    }

    /**
     * 정리 작업
     */
    destroy() {
        if (this.installPopup) {
            this.installPopup.destroy();
            this.installPopup = null;
            this.isPopupMode = false;
        }
        
        this.isPageMode = false;
        PWAUtils.log('PWA Installer 정리 완료');
    }
}

// ========================================
// 전역 인스턴스 및 자동 초기화 (iOS 최적화)
// ========================================

// 전역 인스턴스 생성
const pwaInstaller = new PWAInstaller();

// 자동 초기화 (iOS 최적화)
document.addEventListener('DOMContentLoaded', async function() {
    PWAUtils.log(`DOM 로드 완료 - PWA Installer 자동 감지 (iOS: ${PWAUtils.detectDevice().isIOS})`);
    
    const device = PWAUtils.detectDevice();
    
    // PWA 환경 체크 및 body 클래스 추가 (iOS Safari 제외)
    PWAUtils.addPWABodyClass();
    
    // iOS 최적화 적용
    if (device.isIOS) {
        PWAUtils.log('iOS 디바이스 감지 - 최적화 적용');
        IOSOptimizer.injectOptimizedCSS();
        IOSOptimizer.setupEventListeners();
        
        // iOS 전용 디버깅
        PWAUtils.log('iOS 환경 정보:', IOSOptimizer.getDebugInfo());
    }
    
    // 설치 페이지인지 확인
    if (document.getElementById('pwa-install-section') || 
        document.getElementById('install-button') ||
        document.querySelector('.install-button')) {
        
        PWAUtils.log('PWA 설치 페이지 감지됨');
        await pwaInstaller.initInstallPage();
    } else {
        // 일반 페이지에서는 팝업 모드로 초기화
        PWAUtils.log(`일반 페이지 - 팝업 모드 초기화 (iOS: ${device.isIOS})`);
        await pwaInstaller.initPopup({
            skin: 'default',
            useServerConfig: true,
            autoShow: true
        });
    }
});

// PWA 환경 변화 감지 (페이지 가시성 변경 시)
document.addEventListener('visibilitychange', function() {
    if (!document.hidden) {
        // 페이지가 다시 보일 때 PWA 환경 재체크
        setTimeout(() => {
            PWAUtils.updatePWABodyClass();
        }, 100);
    }
});

// 페이지 로드 완료 후에도 한 번 더 체크 (늦은 PWA 감지 대응)
window.addEventListener('load', function() {
    setTimeout(() => {
        PWAUtils.updatePWABodyClass();
    }, 500);
});

// ========================================
// 전역 함수 및 호환성 제공
// ========================================

/**
 * 간편 초기화 함수들
 */
window.initPWAInstallPage = function() {
    return pwaInstaller.initInstallPage();
};

window.initPWAInstallPopup = async function(options = {}) {
    return await pwaInstaller.initPopup(options);
};

window.showPWAInstallPopup = async function(options = {}) {
    await pwaInstaller.showPopup(options);
};

window.hidePWAInstallPopup = function() {
    pwaInstaller.hidePopup();
};

/**
 * 유틸리티 함수들
 */
window.detectDevice = function() {
    return PWAUtils.detectDevice();
};

window.checkPWAInstallability = function() {
    return PWAUtils.isPWASupported();
};

window.isPWAEnvironment = function() {
    return PWAUtils.isPWAEnvironment();
};

/**
 * 모바일 디버깅 함수들
 */
window.showMobileDebugInfo = function() {
    PWAUtils.showMobileDebugInfo();
};

window.hideMobileDebugInfo = function() {
    PWAUtils.hideMobileDebugInfo();
};

/**
 * PWA 환경 관련 함수들
 */
window.addPWABodyClass = function() {
    PWAUtils.addPWABodyClass();
};

window.removePWABodyClass = function() {
    PWAUtils.removePWABodyClass();
};

window.updatePWABodyClass = function() {
    PWAUtils.updatePWABodyClass();
};

/**
 * 서버 설정 관련 함수들
 */
window.loadPWAServerSettings = async function() {
    return await PWAUtils.loadPWASettings();
};

window.getPWAServerSettings = function() {
    return getPWAServerSettings();
};

// ========================================
// iOS 전용 디버깅 및 개발 도구
// ========================================

// iOS 감지 및 디버깅 함수
window.debugiOSPWA = function() {
    const device = PWAUtils.detectDevice();
    if (!device.isIOS) {
        console.log('iOS 디바이스가 아닙니다.');
        return null;
    }
    
    const info = IOSOptimizer.getDebugInfo();
    const popupConditions = {
        shouldShow: !PWAUtils.isPWAEnvironment() && device.isIOS && device.isSafari,
        isPWAEnv: PWAUtils.isPWAEnvironment(),
        isIOS: device.isIOS,
        isSafari: device.isSafari
    };
    
    console.group('[iOS PWA Debug]');
    console.log('환경 정보:', info);
    console.log('팝업 표시 조건:', popupConditions);
    console.groupEnd();
    
    return { ...info, popupConditions };
};

// 강제 팝업 표시 (iOS 테스트용)
window.forceShowiOSPopup = async function() {
    if (typeof pwaInstaller !== 'undefined') {
        // 무시 데이터 임시 삭제
        const dismissInfo = PWAStorage.getDismissedInfo();
        PWAStorage.clearDismissed();
        
        await pwaInstaller.showPopup({
            autoShow: false,
            showDelay: 0,
            useServerConfig: false
        });
        
        console.log('iOS 팝업 강제 표시됨');
        
        // 무시 데이터 복원 (3초 후)
        setTimeout(() => {
            if (dismissInfo) {
                PWAStorage.setDismissed(24);
            }
        }, 3000);
    } else {
        console.error('pwaInstaller가 정의되지 않음');
    }
};

// iOS PWA 무시 데이터 초기화
window.resetiOSPWADismiss = function() {
    PWAStorage.clearDismissed();
    console.log('iOS PWA 무시 데이터 초기화됨');
};

// ========================================
// 전역 객체 등록
// ========================================

// 메인 API
window.PWAInstaller = pwaInstaller;

// 유틸리티 클래스들
window.PWAUtils = PWAUtils;
window.PWAStorage = PWAStorage;
window.SkinLoader = SkinLoader;
window.IOSOptimizer = IOSOptimizer;

// 개별 클래스들 (고급 사용자용)
window.PWAInstallPage = PWAInstallPage;
window.PWAInstallPopup = PWAInstallPopup;

// 기존 호환성 (pwa-install.js 파일용)
window.PWAInstall = {
    detectDevice: PWAUtils.detectDevice,
    checkPWAInstallability: PWAUtils.isPWASupported,
    showInstallInstructions: () => {
        if (pwaInstaller.installPage) {
            pwaInstaller.installPage.showInstallInstructions();
        }
    },
    hideAllInstallSections: () => {
        if (pwaInstaller.installPage) {
            pwaInstaller.installPage.hideAllInstallSections();
        }
    },
    updateInstallStatus: (status) => {
        if (pwaInstaller.installPage) {
            pwaInstaller.installPage.updateInstallStatus(status);
        }
    },
    updatePageContent: (title, description) => {
        if (pwaInstaller.installPage) {
            pwaInstaller.installPage.updatePageContent(title, description);
        }
    }
};

// ========================================
// 디버깅 및 개발 도구 (iOS 특화)
// ========================================

const config = new PWAConfig();

if (config.get('debug')) {
    const device = PWAUtils.detectDevice();
    
    // 디버깅 함수
    window.debugPWAInstaller = function() {
        const deviceInfo = pwaInstaller.getDeviceInfo();
        const installStatus = pwaInstaller.checkInstallStatus();
        
        console.group('[PWA Installer Debug - iOS 최적화]');
        console.log('버전:', config.get('version'));
        console.log('디바이스 정보:', deviceInfo);
        console.log('설치 상태:', installStatus);
        console.log('서버 설정:', getPWAServerSettings());
        console.log('beforeinstallprompt:', !!getDeferredPrompt());
        console.log('설치 페이지 모드:', pwaInstaller.isPageMode);
        console.log('팝업 모드:', pwaInstaller.isPopupMode);
        
        if (device.isIOS) {
            console.log('=== iOS 전용 정보 ===');
            const iOSInfo = IOSOptimizer.getDebugInfo();
            if (iOSInfo) {
                console.log('iOS 디버그 정보:', iOSInfo);
            }
        }
        
        console.groupEnd();
        
        return {
            device: deviceInfo,
            install: installStatus,
            server: getPWAServerSettings(),
            hasPrompt: !!getDeferredPrompt(),
            modes: {
                page: pwaInstaller.isPageMode,
                popup: pwaInstaller.isPopupMode
            }
        };
    };
    
    // 개발용 단축키 (iOS 특화)
    window.addEventListener('keydown', async (event) => {
        // Ctrl+Alt+P: 팝업 강제 표시
        if (event.ctrlKey && event.altKey && event.key === 'p') {
            event.preventDefault();
            PWAUtils.log('개발용 단축키: 팝업 강제 표시');
            if (device.isIOS) {
                await window.forceShowiOSPopup();
            } else {
                await pwaInstaller.showPopup();
            }
        }
        
        // Ctrl+Alt+R: 서버 설정 새로고침
        if (event.ctrlKey && event.altKey && event.key === 'r') {
            event.preventDefault();
            PWAUtils.log('개발용 단축키: 서버 설정 새로고침');
            await PWAUtils.loadPWASettings();
        }
        
        // Ctrl+Alt+I: 디버그 정보 표시
        if (event.ctrlKey && event.altKey && event.key === 'i') {
            event.preventDefault();
            if (device.isIOS) {
                window.debugiOSPWA();
            } else {
                window.debugPWAInstaller();
            }
        }
        
        // Ctrl+Alt+M: 모바일 디버그 정보 표시/숨기기
        if (event.ctrlKey && event.altKey && event.key === 'm') {
            event.preventDefault();
            const debugElement = document.getElementById('pwa-mobile-debug');
            if (debugElement) {
                window.hideMobileDebugInfo();
            } else {
                window.showMobileDebugInfo();
            }
        }
        
        // Ctrl+Alt+D: iOS PWA 무시 데이터 초기화
        if (event.ctrlKey && event.altKey && event.key === 'd') {
            event.preventDefault();
            if (device.isIOS) {
                window.resetiOSPWADismiss();
            }
        }
    });
    
    PWAUtils.log(`PWA Installer 디버그 모드 활성화 (iOS 최적화: ${device.isIOS})`);
    PWAUtils.log('공통 단축키: Ctrl+Alt+M (모바일 디버그), Ctrl+Alt+I (디버그), Ctrl+Alt+P (팝업)');
    if (device.isIOS) {
        PWAUtils.log('iOS 단축키: Ctrl+Alt+D (무시 초기화)');
    } else {
        PWAUtils.log('기타 단축키: Ctrl+Alt+R (설정 새로고침)');
    }
}

// ========================================
// Service Worker 업데이트 알림
// ========================================

if ('serviceWorker' in navigator) {
    navigator.serviceWorker.addEventListener('controllerchange', () => {
        PWAUtils.log('Service Worker 업데이트됨 - 페이지 새로고침 권장');
        if (window.PushManager && window.PushManager.showToast) {
            window.PushManager.showToast('앱이 업데이트되었습니다. 페이지를 새로고침해주세요.', 'info');
        }
    });
}

// ========================================
// 최종 로그 (iOS 최적화)
// ========================================
PWAUtils.log(`PWA Installer v${config.get('version')} 로드 완료 (모듈형 - iOS 최적화: ${PWAUtils.detectDevice().isIOS})`);
PWAUtils.log('사용 가능한 API:');
PWAUtils.log('- window.PWAInstaller (메인 API)');
PWAUtils.log('- window.initPWAInstallPage() (설치 페이지 초기화)');
PWAUtils.log('- window.initPWAInstallPopup() (팝업 초기화 - 비동기)');
PWAUtils.log('- window.showPWAInstallPopup() (팝업 표시 - 비동기)');
PWAUtils.log('- window.loadPWAServerSettings() (서버 설정 로드)');
PWAUtils.log('- window.debugPWAInstaller() (디버깅)');

if (PWAUtils.detectDevice().isIOS) {
    PWAUtils.log('=== iOS 전용 API ===');
    PWAUtils.log('- window.debugiOSPWA() (iOS 디버깅)');
    PWAUtils.log('- window.forceShowiOSPopup() (iOS 팝업 강제 표시)');
    PWAUtils.log('- window.resetiOSPWADismiss() (iOS 무시 데이터 초기화)');
}

// 모듈 export (ES6 모듈 사용 시)
export { 
    pwaInstaller as default,
    PWAInstaller,
    PWAInstallPage,
    PWAInstallPopup,
    PWAUtils,
    PWAStorage,
    SkinLoader,
    IOSOptimizer
};