<?php
// 파일 맨 위에 추가
include_once('./_common.php');
include_once(G5_LIB_PATH.'/pushmanager.lib.php');

// CORS 헤더 설정 (필요한 경우)
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// OPTIONS 요청 처리
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

header('Content-Type: application/json; charset=utf-8');

$action = $_GET['action'] ?? $_POST['action'] ?? '';

switch ($action) {
    // 클라이언트용 액션들
    case 'get_vapid_key':
        get_vapid_key_from_db();
        break;
        
    case 'get_pwa_config':
        get_pwa_config_from_db();
        break;
        
    case 'subscribe':
        handle_push_subscription();
        break;
        
    case 'unsubscribe':
        handle_push_unsubscription();
        break;
        
    case 'check_status':
        check_subscription_status();
        break;

    case 'get_user_devices':
        handle_get_user_devices();
        break;
    
    // 알림 설정 관련 액션들 (토큰 체크 필요)
    case 'get_notification_settings':
        check_token_and_execute('handle_get_notification_settings');
        break;
        
    case 'save_notification_settings':
        check_token_and_execute('handle_save_notification_settings');
        break;
        
    case 'toggle_device_notification':
        check_token_and_execute('handle_toggle_device_notification');
        break;
        
    // 관리자용 액션들 (관리자 권한 확인)
    case 'check_member':
    case 'get_level_count':
    case 'get_gender_count':
    case 'search_members':
        if (!$is_admin) {
            http_response_code(403);
            echo json_encode(array('success' => false, 'message' => '관리자 권한이 필요합니다.'));
            exit;
        }
        
        switch ($action) {
            case 'check_member':
                check_member();
                break;
            case 'get_level_count':
                get_level_count();
                break;
            case 'get_gender_count':
                get_gender_count();
                break;
            case 'search_members':
                search_members();
                break;
        }
        break;
        
    default:
        http_response_code(400);
        echo json_encode(array('success' => false, 'message' => '지원하지 않는 액션입니다.'));
        exit;
}

/**
 * 토큰 체크 후 함수 실행
 * @param string $function_name
 */
function check_token_and_execute($function_name) {
    // 그누보드5 토큰 체크 (버전 호환성 고려)
    $token_valid = false;
    
    if (function_exists('check_token')) {
        $token_valid = check_token();
    } elseif (function_exists('verify_token')) {
        $token_valid = verify_token();
    } else {
        // 토큰 체크 함수가 없으면 기본적인 체크
        if (defined('G5_TOKEN_NAME') && isset($_GET[G5_TOKEN_NAME]) && isset($_SESSION[G5_TOKEN_NAME])) {
            $token_valid = ($_GET[G5_TOKEN_NAME] === $_SESSION[G5_TOKEN_NAME]);
        } else {
            // 토큰 시스템이 없으면 통과 (보안 위험 있음)
            $token_valid = true;
        }
    }
    
    if (!$token_valid) {
        http_response_code(403);
        echo json_encode(array('success' => false, 'message' => '잘못된 접근입니다.'));
        exit;
    }
    
    if (function_exists($function_name)) {
        $function_name();
    } else {
        http_response_code(500);
        echo json_encode(array('success' => false, 'message' => '함수를 찾을 수 없습니다.'));
        exit;
    }
}

// ========================================
// PWA 설정 관련 함수 (새로 추가)
// ========================================

/**
 * DB에서 PWA 설정 조회
 */
function get_pwa_config_from_db() {
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        http_response_code(405);
        echo json_encode(array('success' => false, 'message' => 'GET 메서드만 허용됩니다.'));
        exit;
    }
    
    try {
        // 푸시매니저 설정 조회
        $config = get_pushmanager_config();
        
        // 기본 응답 구조
        $response = array(
            'success' => true,
            'pwa_config' => array(
                'title' => '앱 설치 안내',
                'description' => '홈 화면에 추가하여 앱처럼 편리하게 이용하세요!',
                'installButtonText' => '설치하기',
                'laterButtonText' => '나중에',
                'showDelay' => 3000,
                'dismissHours' => 24,
                'autoShow' => true,
                'installUrl' => G5_URL . '/plugin/pushmanager/pwa_install.php',
                'skin' => 'default'
            ),
            'site_info' => array(
                'name' => '우리 사이트',
                'url' => G5_URL
            ),
            'icons' => array(
                'push_icon' => null,
                'badge_icon' => null
            )
        );
        
        if ($config && is_array($config)) {

                        // 👈 디버깅 추가
            error_log("Skin value from DB: " . ($config['pwa_popup_skin'] ?? 'NULL'));
            
            if (isset($config['pwa_popup_skin']) && !empty($config['pwa_popup_skin'])) {
                $response['pwa_config']['skin'] = $config['pwa_popup_skin'];
                error_log("Skin set to: " . $config['pwa_popup_skin']); // 👈 디버깅
            } else {
                error_log("Skin not found or empty in config"); // 👈 디버깅
            }
            
            // PWA 설정 업데이트
            if (isset($config['pwa_popup_title']) && !empty($config['pwa_popup_title'])) {
                $response['pwa_config']['title'] = $config['pwa_popup_title'];
            }
            
            if (isset($config['pwa_popup_description']) && !empty($config['pwa_popup_description'])) {
                $response['pwa_config']['description'] = $config['pwa_popup_description'];
            }

            if (isset($config['pwa_popup_skin']) && !empty($config['pwa_popup_skin'])) {
                $response['pwa_config']['skin'] = $config['pwa_popup_skin'];
            }
            
            // 사이트 정보 업데이트
            if (isset($config['site_name']) && !empty($config['site_name'])) {
                $response['site_info']['name'] = $config['site_name'];
            } elseif (function_exists('get_config_vars')) {
                $cf_title = get_config_vars('cf_title');
                if ($cf_title) {
                    $response['site_info']['name'] = $cf_title;
                }
            }
            
            if (isset($config['site_url']) && !empty($config['site_url'])) {
                $response['site_info']['url'] = $config['site_url'];
            }
            
            // 아이콘 정보 업데이트
            if (isset($config['push_icon_url']) && !empty($config['push_icon_url'])) {
                $response['icons']['push_icon'] = $config['push_icon_url'];
            }
            
            if (isset($config['badge_icon_url']) && !empty($config['badge_icon_url'])) {
                $response['icons']['badge_icon'] = $config['badge_icon_url'];
            }
        }
        
        // 캐시 헤더 설정 (5분 캐시)
        header('Cache-Control: public, max-age=300');
        header('Expires: ' . gmdate('D, d M Y H:i:s', time() + 300) . ' GMT');
        
        echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        
        // 로그 기록
        if (function_exists('log_pushmanager_activity')) {
            log_pushmanager_activity('api', 'PWA 설정 조회', 'PWA 설정 API 호출 완료');
        }
        
    } catch (Exception $e) {
        // 오류 응답
        http_response_code(500);
        
        $error_response = array(
            'success' => false,
            'error' => '설정을 불러오는 중 오류가 발생했습니다.',
            'debug' => $e->getMessage(),
            'pwa_config' => array(
                'title' => '앱 설치 안내',
                'description' => '홈 화면에 추가하여 앱처럼 편리하게 이용하세요!',
                'installButtonText' => '설치하기',
                'laterButtonText' => '나중에',
                'showDelay' => 3000,
                'dismissHours' => 24,
                'autoShow' => true,
                'installUrl' => G5_URL . '/plugin/pushmanager/pwa_install.php',
                'skin' => 'default'
            )
        );
        
        echo json_encode($error_response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    }
    
    exit;
}

// ========================================
// 기존 함수들
// ========================================

/**
 * 회원 존재 및 구독 여부 확인
 */
function check_member() {
    global $g5;
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(array('success' => false, 'message' => 'POST 메서드만 허용됩니다.'));
        exit;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    $member_id = clean_xss_tags($input['member_id']);
    
    if (empty($member_id)) {
        echo json_encode(array('success' => false, 'message' => '회원 아이디가 필요합니다.'));
        exit;
    }
    
    // 회원 존재 확인
    $sql = "SELECT mb_id, mb_name FROM {$g5['member_table']} WHERE mb_id = '{$member_id}'";
    $member = sql_fetch($sql);
    
    if (!$member) {
        echo json_encode(array('success' => false, 'message' => '존재하지 않는 회원입니다.'));
        exit;
    }
    
    // 구독 여부 확인
    $sql = "SELECT COUNT(*) as cnt FROM pushmanager_subscriptions WHERE member_id = '{$member_id}' AND is_active = 1";
    $subscription_result = sql_fetch($sql);
    $has_subscription = $subscription_result['cnt'] > 0;
    
    echo json_encode(array(
        'success' => true,
        'data' => array(
            'member_id' => $member['mb_id'],
            'member_name' => $member['mb_name'],
            'has_subscription' => $has_subscription
        )
    ));
    exit;
}

/**
 * 레벨별 회원 수 조회
 */
function get_level_count() {
    global $g5;
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(array('success' => false, 'message' => 'POST 메서드만 허용됩니다.'));
        exit;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    $level = intval($input['level']);
    
    if ($level < 1) {
        echo json_encode(array('success' => false, 'message' => '올바른 레벨이 필요합니다.'));
        exit;
    }
    
    // 해당 레벨 총 회원 수
    $sql = "SELECT COUNT(*) as cnt FROM {$g5['member_table']} WHERE mb_level = {$level}";
    $total_result = sql_fetch($sql);
    $total_members = $total_result['cnt'];
    
    // 해당 레벨 구독자 수
    $sql = "SELECT COUNT(DISTINCT ps.member_id) as cnt 
            FROM pushmanager_subscriptions ps 
            INNER JOIN {$g5['member_table']} m ON ps.member_id = m.mb_id 
            WHERE m.mb_level = {$level} AND ps.is_active = 1";
    $subscription_result = sql_fetch($sql);
    $subscription_count = $subscription_result['cnt'];
    
    echo json_encode(array(
        'success' => true,
        'data' => array(
            'level' => $level,
            'total_members' => $total_members,
            'subscription_count' => $subscription_count
        )
    ));
    exit;
}

/**
 * 성별 회원 수 조회
 */
function get_gender_count() {
    global $g5;
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(array('success' => false, 'message' => 'POST 메서드만 허용됩니다.'));
        exit;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    $gender = clean_xss_tags($input['gender']);
    
    if (!in_array($gender, array('M', 'F'))) {
        echo json_encode(array('success' => false, 'message' => '올바른 성별이 필요합니다.'));
        exit;
    }
    
    // 해당 성별 총 회원 수
    $sql = "SELECT COUNT(*) as cnt FROM {$g5['member_table']} WHERE mb_sex = '{$gender}'";
    $total_result = sql_fetch($sql);
    $total_members = $total_result['cnt'];
    
    // 해당 성별 구독자 수
    $sql = "SELECT COUNT(DISTINCT ps.member_id) as cnt 
            FROM pushmanager_subscriptions ps 
            INNER JOIN {$g5['member_table']} m ON ps.member_id = m.mb_id 
            WHERE m.mb_sex = '{$gender}' AND ps.is_active = 1";
    $subscription_result = sql_fetch($sql);
    $subscription_count = $subscription_result['cnt'];
    
    echo json_encode(array(
        'success' => true,
        'data' => array(
            'gender' => $gender,
            'total_members' => $total_members,
            'subscription_count' => $subscription_count
        )
    ));
    exit;
}

/**
 * 회원 검색
 */
function search_members() {
    global $g5;
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(array('success' => false, 'message' => 'POST 메서드만 허용됩니다.'));
        exit;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    $search = clean_xss_tags($input['search']);
    
    if (empty($search)) {
        echo json_encode(array('success' => false, 'message' => '검색어가 필요합니다.'));
        exit;
    }
    
    // 회원 검색 (아이디, 이름, 이메일)
    $search_sql = "SELECT m.mb_id, m.mb_name, m.mb_email,
                          (SELECT COUNT(*) FROM pushmanager_subscriptions ps 
                           WHERE ps.member_id = m.mb_id AND ps.is_active = 1) as has_subscription
                   FROM {$g5['member_table']} m 
                   WHERE (m.mb_id LIKE '%{$search}%' 
                          OR m.mb_name LIKE '%{$search}%' 
                          OR m.mb_email LIKE '%{$search}%')
                   ORDER BY m.mb_datetime DESC 
                   LIMIT 20";
    
    $result = sql_query($search_sql);
    $members = array();
    
    while ($row = sql_fetch_array($result)) {
        $members[] = array(
            'mb_id' => $row['mb_id'],
            'mb_name' => $row['mb_name'],
            'mb_email' => $row['mb_email'],
            'has_subscription' => $row['has_subscription'] > 0
        );
    }
    
    echo json_encode(array(
        'success' => true,
        'data' => array(
            'search' => $search,
            'members' => $members,
            'count' => count($members)
        )
    ));
    exit;
}

/**
 * DB에서 VAPID 공개키 조회
 */
function get_vapid_key_from_db() {
    header('Content-Type: application/json; charset=utf-8');
    
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        http_response_code(405);
        echo json_encode(array('success' => false, 'message' => 'GET 메서드만 허용됩니다.'));
        exit;
    }
    
    try {
        $config = get_pushmanager_config();
        
        if (!$config) {
            http_response_code(404);
            echo json_encode(array('success' => false, 'message' => 'Push Manager 설정이 없습니다.'));
            exit;
        }
        
        if (empty($config['vapid_public_key'])) {
            http_response_code(404);
            echo json_encode(array('success' => false, 'message' => 'VAPID 공개키가 설정되지 않았습니다.'));
            exit;
        }
        
        echo json_encode(array(
            'success' => true,
            'data' => array(
                'vapidPublicKey' => $config['vapid_public_key'],
                'siteName' => $config['site_name']
            )
        ));
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(array('success' => false, 'message' => '서버 오류: ' . $e->getMessage()));
    }
    
    exit;
}

/**
 * 구독 상태 확인
 */
function check_subscription_status() {
    global $member;
    
    try {
        $user = get_current_user_identifier();
        $devices = get_user_devices();
        $settings = get_user_notification_settings();
        
        echo json_encode(array(
            'success' => true,
            'data' => array(
                'user_type' => $user['type'],
                'user_id' => $user['id'],
                'devices_count' => count($devices),
                'settings' => $settings,
                'is_member' => $user['type'] === 'member'
            )
        ));
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(array('success' => false, 'message' => '상태 확인 중 오류가 발생했습니다: ' . $e->getMessage()));
    }
    
    exit;
}
?>