<?php
// 🚀 간소화된 test_push.php (통합 함수 사용)

include_once('./_common.php');

$pushmanager_lib = G5_LIB_PATH . '/pushmanager.lib.php';
if (file_exists($pushmanager_lib)) {
    include_once($pushmanager_lib);
} else {
    echo json_encode(['success' => false, 'message' => 'Push Manager 라이브러리를 찾을 수 없습니다.']);
    exit;
}

// 사용자 식별
$user = get_current_user_identifier();
if (!$user['id']) {
    echo json_encode(['success' => false, 'message' => '사용자 식별에 실패했습니다.']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => '잘못된 요청입니다.']);
    exit;
}

$action = $_POST['action'] ?? '';
$device_id = $_POST['device_id'] ?? '';

// 테스트 메시지 템플릿
$test_templates = [
    [
        'title' => '🧪 테스트 알림',
        'body' => '푸시 알림이 정상적으로 작동하고 있습니다!',
        'tag' => 'test_' . time()
    ],
    [
        'title' => '✅ 연결 확인',
        'body' => '기기와 서버 간 푸시 연결이 활성화되어 있습니다.',
        'tag' => 'test_' . time()
    ],
    [
        'title' => '📱 기기 테스트',
        'body' => '이 메시지가 보인다면 푸시 알림 설정이 완료되었습니다.',
        'tag' => 'test_' . time()
    ],
    [
        'title' => '🎉 알림 성공!',
        'body' => date('H:i') . ' - 테스트 푸시가 성공적으로 전달되었습니다.',
        'tag' => 'test_' . time()
    ]
];

$test_message = $test_templates[array_rand($test_templates)];

try {
    switch ($action) {
        case 'test_current_device':
            // 🎯 현재 브라우저 테스트 (대폭 간소화)
            $current_ip = get_real_ip();
            $current_user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
            
            // 현재 브라우저의 구독 정보 조회
            $where_conditions = array();
            $where_conditions[] = "is_active = 1";
            $where_conditions[] = "ip_address = '" . addslashes($current_ip) . "'";
            
            if ($user['member_id']) {
                $where_conditions[] = "member_id = '" . addslashes($user['member_id']) . "'";
            } else {
                $where_conditions[] = "member_id IS NULL";
                if ($current_user_agent) {
                    $where_conditions[] = "user_agent = '" . addslashes($current_user_agent) . "'";
                }
            }
            
            $subscription_sql = "SELECT id, endpoint, p256dh_key, auth_key, member_id, user_agent
                               FROM pushmanager_subscriptions 
                               WHERE " . implode(' AND ', $where_conditions) . "
                               ORDER BY created_at DESC LIMIT 1";
            
            $subscription_data = sql_fetch($subscription_sql);
            
            if (!$subscription_data) {
                throw new Exception('이 브라우저의 구독 정보를 찾을 수 없습니다. 먼저 푸시 알림을 구독해주세요.');
            }
            
            // 🚀 통합 함수로 발송 (구독 정보 직접 전달)
            $subscriptions = array(
                array(
                    'endpoint' => $subscription_data['endpoint'],
                    'keys' => array(
                        'p256dh' => $subscription_data['p256dh_key'],
                        'auth' => $subscription_data['auth_key']
                    ),
                    'member_id' => $subscription_data['member_id']
                )
            );
            
            $options = array(
                'notification_type' => 'system',
                'require_interaction' => false
            );
            
            $result = send_push_unified($subscriptions, $test_message['title'], $test_message['body'], $options);
            
            if ($result['success']) {
                echo json_encode([
                    'success' => true,
                    'message' => '현재 브라우저로 테스트 푸시가 성공적으로 발송되었습니다.',
                    'subscription_id' => $subscription_data['id'],
                    'sent_count' => $result['data']['sent'],
                    'user_agent' => substr($subscription_data['user_agent'], 0, 50) . '...'
                ]);
            } else {
                throw new Exception('푸시 발송에 실패했습니다: ' . $result['message']);
            }
            break;
            
        case 'test_single':
            // 🎯 개별 기기 테스트 (간소화)
            if (empty($device_id) || !is_numeric($device_id)) {
                throw new Exception('올바르지 않은 기기 ID입니다.');
            }
            
            // 해당 기기의 구독 정보 조회 (권한 확인 포함)
            $where_conditions = array();
            $where_conditions[] = "id = " . intval($device_id);
            $where_conditions[] = "is_active = 1";
            
            if ($user['member_id']) {
                $where_conditions[] = "member_id = '" . addslashes($user['member_id']) . "'";
            } else {
                $where_conditions[] = "member_id IS NULL";
                $where_conditions[] = "ip_address = '" . addslashes(get_real_ip()) . "'";
            }
            
            $subscription_sql = "SELECT endpoint, p256dh_key, auth_key, member_id 
                               FROM pushmanager_subscriptions 
                               WHERE " . implode(' AND ', $where_conditions);
            
            $subscription_data = sql_fetch($subscription_sql);
            
            if (!$subscription_data) {
                throw new Exception('기기를 찾을 수 없거나 권한이 없습니다.');
            }
            
            // 🚀 통합 함수로 발송
            $subscriptions = array(
                array(
                    'endpoint' => $subscription_data['endpoint'],
                    'keys' => array(
                        'p256dh' => $subscription_data['p256dh_key'],
                        'auth' => $subscription_data['auth_key']
                    ),
                    'member_id' => $subscription_data['member_id']
                )
            );
            
            $options = array('notification_type' => 'system');
            $result = send_push_unified($subscriptions, $test_message['title'], $test_message['body'], $options);
            
            if ($result['success']) {
                echo json_encode([
                    'success' => true,
                    'message' => '테스트 푸시가 성공적으로 발송되었습니다.',
                    'device_id' => $device_id,
                    'sent_count' => $result['data']['sent']
                ]);
            } else {
                throw new Exception('푸시 발송에 실패했습니다: ' . $result['message']);
            }
            break;
            
        case 'test_all':
            // 🎯 모든 기기 테스트 (대폭 간소화)
            $recipients = array();
            $options = array('notification_type' => 'system');
            
            // 사용자별 필터 설정
            if ($user['member_id']) {
                $recipients = array($user['member_id']);
            } else {
                // 비회원인 경우 IP 기반 필터링은 통합 함수에서 처리하기 어려우므로
                // 직접 구독 정보 조회
                $ip_address = get_real_ip();
                $sql = "SELECT endpoint, p256dh_key, auth_key, member_id 
                       FROM pushmanager_subscriptions 
                       WHERE member_id IS NULL 
                       AND ip_address = '" . addslashes($ip_address) . "' 
                       AND is_active = 1";
                
                $result_query = sql_query($sql);
                $subscriptions = array();
                
                while ($row = sql_fetch_array($result_query)) {
                    $subscriptions[] = array(
                        'endpoint' => $row['endpoint'],
                        'keys' => array(
                            'p256dh' => $row['p256dh_key'],
                            'auth' => $row['auth_key']
                        ),
                        'member_id' => $row['member_id']
                    );
                }
                
                if (empty($subscriptions)) {
                    throw new Exception('활성화된 기기가 없습니다.');
                }
                
                // 비회원인 경우 구독 정보 직접 전달
                $result = send_push_unified($subscriptions, $test_message['title'], $test_message['body'], $options);
                
                if ($result['success']) {
                    echo json_encode([
                        'success' => true,
                        'message' => "총 {$result['data']['sent']}개 기기에 테스트 푸시가 발송되었습니다.",
                        'count' => $result['data']['sent'],
                        'total' => $result['data']['total'],
                        'failed' => $result['data']['failed']
                    ]);
                } else {
                    throw new Exception('테스트 푸시 발송에 실패했습니다: ' . $result['message']);
                }
                break;
            }
            
            // 회원인 경우 통합 함수 사용
            $result = send_push_unified($recipients, $test_message['title'], $test_message['body'], $options);
            
            if ($result['success']) {
                $message = "총 {$result['data']['sent']}개 기기에 테스트 푸시가 발송되었습니다.";
                if ($result['data']['failed'] > 0) {
                    $message .= " (실패: {$result['data']['failed']}개)";
                }
                
                echo json_encode([
                    'success' => true,
                    'message' => $message,
                    'count' => $result['data']['sent'],
                    'total' => $result['data']['total'],
                    'failed' => $result['data']['failed']
                ]);
            } else {
                throw new Exception('테스트 푸시 발송에 실패했습니다: ' . $result['message']);
            }
            break;
            
        default:
            throw new Exception('올바르지 않은 액션입니다.');
    }
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>