<?php
if (!defined('_GNUBOARD_')) exit; // 개별 페이지 접근 불가

/**
 * 현재 사이트의 Origin URL 가져오기
 * @return string
 */
function get_site_origin() {
    // DB에서 설정된 사이트 URL 우선 사용
    $push_config = get_pushmanager_config();
    if ($push_config && !empty($push_config['site_url'])) {
        return rtrim($push_config['site_url'], '/');
    }
    
    // 현재 요청을 기반으로 Origin 생성
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    
    return $protocol . $host;
}

/**
 * API 요청용 공통 헤더 생성
 * @param string $api_key
 * @return array
 */
function get_api_headers($api_key = '') {
    $headers = array(
        'Content-Type: application/json',
        'User-Agent: GnuBoard5-PushManager/1.0',
        'Origin: ' . get_site_origin()
    );
    
    if (!empty($api_key)) {
        $headers[] = 'X-API-Key: ' . $api_key;
    }
    
    return $headers;
}

/**
 * Push Manager 설정 조회 (캐싱 적용)
 * @return array|false
 */
function get_pushmanager_config() {
    static $config_cache = null;
    
    if ($config_cache === null) {
        $sql = "SELECT * FROM pushmanager_config ORDER BY id DESC LIMIT 1";
        $result = sql_query($sql, false);
        
        if ($result && sql_num_rows($result) > 0) {
            $config_cache = sql_fetch_array($result);
        } else {
            $config_cache = false;
        }
    }
    
    return $config_cache;
}

/**
 * Push 아이콘 이미지 경로 가져오기
 * @return string
 */
function get_push_icon_url() {
    $config = get_pushmanager_config();
    
    // DB에서 설정된 아이콘 URL이 있으면 사용
    if ($config && !empty($config['push_icon_url'])) {
        // 절대 경로인지 확인 (http:// 또는 https://로 시작)
        if (strpos($config['push_icon_url'], 'http://') === 0 || strpos($config['push_icon_url'], 'https://') === 0) {
            return $config['push_icon_url'];
        } else {
            // 상대 경로인 경우 사이트 Origin과 결합
            return G5_URL . '/' . ltrim($config['push_icon_url'], '/');
        }
    }
    
    // 기본 아이콘 경로 반환
    return get_site_origin() . '/img/push-icon.png';
}

/**
 * Push 배지 이미지 경로 가져오기
 * @return string
 */
function get_push_badge_url() {
    $config = get_pushmanager_config();
    
    // DB에서 설정된 배지 URL이 있으면 사용
    if ($config && !empty($config['badge_icon_url'])) {
        // 절대 경로인지 확인 (http:// 또는 https://로 시작)
        if (strpos($config['badge_icon_url'], 'http://') === 0 || strpos($config['badge_icon_url'], 'https://') === 0) {
            return $config['badge_icon_url'];
        } else {
            // 상대 경로인 경우 사이트 Origin과 결합
            return G5_URL . '/' . ltrim($config['badge_icon_url'], '/');
        }
    }
    
    // 기본 배지 경로 반환
    return get_site_origin() . '/img/push-badge.png';
}

/**
 * 공통 HTTP 응답 처리
 * @param string $response
 * @param int $http_code
 * @param string $error
 * @return array
 */
function parse_api_response($response, $http_code, $error) {
    if ($error) {
        return array(
            'success' => false,
            'message' => 'cURL 오류: ' . $error
        );
    }
    
    $decoded_response = json_decode($response, true);
    
    if ($http_code >= 200 && $http_code < 300) {
        return array(
            'success' => true,
            'data' => $decoded_response['data'] ?? $decoded_response ?? array(),
            'message' => $decoded_response['message'] ?? '성공'
        );
    } else {
        return array(
            'success' => false,
            'message' => $decoded_response['message'] ?? 'HTTP 오류: ' . $http_code,
            'code' => $decoded_response['code'] ?? 'HTTP_ERROR'
        );
    }
}

/**
 * 통합된 API 요청 함수
 * @param string $url
 * @param array $data
 * @param string $method
 * @param string $api_key
 * @return array
 */
function send_api_request($url, $data = array(), $method = 'POST', $api_key = '') {
    $ch = curl_init();
    
    curl_setopt_array($ch, array(
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_HTTPHEADER => get_api_headers($api_key)
    ));
    
    if ($method === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    } elseif ($method === 'GET' && !empty($data)) {
        $url .= '?' . http_build_query($data);
        curl_setopt($ch, CURLOPT_URL, $url);
    }
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    return parse_api_response($response, $http_code, $error);
}

/**
 * API 서버에 사이트 등록 및 키 발급 요청
 * @param array $config
 * @return array
 */
function request_api_registration($config) {
    $api_url = rtrim($config['api_server_url'], '/') . '/api/register';
    
    $post_data = array(
        'site_name' => $config['site_name'],
        'site_url' => $config['site_url'],
        'admin_email' => $config['admin_email']
    );
    
    return send_api_request($api_url, $post_data);
}

/**
 * API 서버 연결 테스트
 * @param array $config
 * @return array
 */
function test_api_connection($config) {
    $api_url = rtrim($config['api_server_url'], '/') . '/health';

    
    return send_api_request($api_url, array(), 'GET');
}

/**
 * 실제 IP 주소 가져오기
 * @return string
 */
function get_real_ip() {
    $ip_keys = array(
        'HTTP_CLIENT_IP',
        'HTTP_X_FORWARDED_FOR',
        'HTTP_X_FORWARDED',
        'HTTP_X_CLUSTER_CLIENT_IP',
        'HTTP_FORWARDED_FOR',
        'HTTP_FORWARDED',
        'REMOTE_ADDR'
    );
    
    foreach ($ip_keys as $key) {
        if (array_key_exists($key, $_SERVER) === true) {
            foreach (explode(',', $_SERVER[$key]) as $ip) {
                $ip = trim($ip);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                    return $ip;
                }
            }
        }
    }
    
    return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
}

/**
 * 구독 정보 저장
 * @param array $subscription_data
 * @return bool
 */
function save_push_subscription($subscription_data) {
    global $member;
    
    $endpoint = addslashes($subscription_data['endpoint']);
    $p256dh_key = addslashes($subscription_data['keys']['p256dh']);
    $auth_key = addslashes($subscription_data['keys']['auth']);
    $member_id = $member['mb_id'] ?? null;
    $user_agent = addslashes($_SERVER['HTTP_USER_AGENT'] ?? '');
    $ip_address = get_real_ip();
    
    // 기존 구독 정보 확인
    $sql = "SELECT id FROM pushmanager_subscriptions WHERE endpoint = '{$endpoint}'";
    $existing = sql_fetch($sql);
    
    if ($existing) {
        // 업데이트
        $sql = "UPDATE pushmanager_subscriptions SET 
                member_id = " . ($member_id ? "'{$member_id}'" : 'NULL') . ",
                p256dh_key = '{$p256dh_key}',
                auth_key = '{$auth_key}',
                user_agent = '{$user_agent}',
                ip_address = '{$ip_address}',
                is_active = 1,
                updated_at = NOW()
                WHERE id = {$existing['id']}";
    } else {
        // 신규 삽입
        $sql = "INSERT INTO pushmanager_subscriptions 
                (member_id, endpoint, p256dh_key, auth_key, user_agent, ip_address, is_active, created_at) 
                VALUES 
                (" . ($member_id ? "'{$member_id}'" : 'NULL') . ", '{$endpoint}', '{$p256dh_key}', '{$auth_key}', '{$user_agent}', '{$ip_address}', 1, NOW())";
    }
    
    return sql_query($sql, false) ? true : false;
}

/**
 * 구독 해제
 * @param string $endpoint
 * @return bool
 */
function unsubscribe_push($endpoint) {
    $endpoint = addslashes($endpoint);
    
    $sql = "UPDATE pushmanager_subscriptions SET 
            is_active = 0,
            updated_at = NOW()
            WHERE endpoint = '{$endpoint}'";
    
    return sql_query($sql, false) ? true : false;
}

/**
 * 구독 쿼리 실행 및 결과 포맷팅 (공통 함수)
 * @param string $sql
 * @return array
 */
function execute_subscription_query($sql) {
    $result = sql_query($sql);
    $subscriptions = array();
    
    while ($row = sql_fetch_array($result)) {
        $subscriptions[] = array(
            'endpoint' => $row['endpoint'],
            'keys' => array(
                'p256dh' => $row['p256dh_key'],
                'auth' => $row['auth_key']
            ),
            'member_id' => $row['member_id']
        );
    }
    
    return $subscriptions;
}

/**
 * 활성 구독자 조회 (통합 필터 지원) - 버그 수정
 * @param array $filters
 * @return array
 */
function get_active_subscriptions($filters = array()) {
    global $g5;
    
    $where_conditions = array();
    $joins = array();
    $use_alias = false;
    
    // 회원 ID 필터
    if (isset($filters['member_ids']) && is_array($filters['member_ids'])) {
        $member_ids = array_map('addslashes', $filters['member_ids']);
        $member_ids_condition = "member_id IN ('" . implode("','", $member_ids) . "')";
        $where_conditions[] = $member_ids_condition;
    }
    
    // 날짜 필터
    if (isset($filters['created_after'])) {
        $where_conditions[] = "created_at >= '" . addslashes($filters['created_after']) . "'";
    }
    
    // 레벨 필터
    if (isset($filters['level'])) {
        $level = intval($filters['level']);
        $joins[] = "INNER JOIN {$g5['member_table']} m ON ps.member_id = m.mb_id";
        $where_conditions[] = "m.mb_level = {$level}";
        $use_alias = true;
    }
    
    // 성별 필터
    if (isset($filters['gender'])) {
        $gender = addslashes($filters['gender']);
        if (empty($joins)) {
            $joins[] = "INNER JOIN {$g5['member_table']} m ON ps.member_id = m.mb_id";
        }
        $where_conditions[] = "m.mb_sex = '{$gender}'";
        $use_alias = true;
    }
    
    // 활성 구독자 조건 추가 (테이블 별칭 고려)
    if ($use_alias) {
        $where_conditions[] = "ps.is_active = 1";
        $select_fields = "ps.endpoint, ps.p256dh_key, ps.auth_key, ps.member_id";
        $table_name = "pushmanager_subscriptions ps";
        $order_by = "ps.created_at DESC";
    } else {
        $where_conditions[] = "is_active = 1";
        $select_fields = "endpoint, p256dh_key, auth_key, member_id";
        $table_name = "pushmanager_subscriptions";
        $order_by = "created_at DESC";
    }
    
    $sql = "SELECT {$select_fields}
            FROM {$table_name} 
            " . implode(' ', $joins) . "
            WHERE " . implode(' AND ', $where_conditions) . " 
            ORDER BY {$order_by}";
    
    return execute_subscription_query($sql);
}

/**
 * 구독자 통계 조회 (한 번에 모든 통계 수집)
 * @return array
 */
function get_subscription_stats() {
    $stats = array(
        'total' => 0,
        'active' => 0,
        'inactive' => 0,
        'today' => 0,
        'this_week' => 0,
        'member' => 0,
        'guest' => 0
    );
    
    $sql = "SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active,
                SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as inactive,
                SUM(CASE WHEN DATE(created_at) = CURDATE() THEN 1 ELSE 0 END) as today,
                SUM(CASE WHEN WEEK(created_at) = WEEK(NOW()) AND YEAR(created_at) = YEAR(NOW()) THEN 1 ELSE 0 END) as this_week,
                SUM(CASE WHEN member_id IS NOT NULL THEN 1 ELSE 0 END) as member
            FROM pushmanager_subscriptions";
    
    $result = sql_fetch($sql);
    if ($result) {
        $stats['total'] = intval($result['total']);
        $stats['active'] = intval($result['active']);
        $stats['inactive'] = intval($result['inactive']);
        $stats['today'] = intval($result['today']);
        $stats['this_week'] = intval($result['this_week']);
        $stats['member'] = intval($result['member']);
        $stats['guest'] = intval($result['total'] - $result['member']);
    }
    
    return $stats;
}

/**
 * 활동 로그 기록 (간소화된 파라미터 처리)
 * @param string $type
 * @param string $title
 * @param string $message
 * @param int $target_count
 * @param int $success_count
 * @param int $error_count
 * @param string $error_message
 * @return bool
 */
function log_pushmanager_activity($type, $title = null, $message = null, $target_count = null, $success_count = null, $error_count = null, $error_message = null) {
    // 값들을 안전하게 처리
    $values = array(
        'type' => "'" . addslashes($type) . "'",
        'title' => $title ? "'" . addslashes($title) . "'" : 'NULL',
        'message' => $message ? "'" . addslashes($message) . "'" : 'NULL',
        'target_count' => $target_count !== null ? intval($target_count) : 'NULL',
        'success_count' => $success_count !== null ? intval($success_count) : 'NULL',
        'error_count' => $error_count !== null ? intval($error_count) : 'NULL',
        'error_message' => $error_message ? "'" . addslashes($error_message) . "'" : 'NULL'
    );
    
    $sql = "INSERT INTO pushmanager_logs 
            (type, title, message, target_count, success_count, error_count, error_message, created_at) 
            VALUES 
            ({$values['type']}, {$values['title']}, {$values['message']}, {$values['target_count']}, {$values['success_count']}, {$values['error_count']}, {$values['error_message']}, NOW())";
    
    return sql_query($sql, false) ? true : false;
}

/**
 * 푸시 메시지 전송 (개선된 로깅)
 * @param string $title
 * @param string $message
 * @param array $options
 * @return array
 */
function send_push_notification($title, $message, $options = array()) {
    $config = get_pushmanager_config();
    
    if (!$config || !$config['api_key']) {
        return array(
            'success' => false,
            'message' => 'API 키가 설정되지 않았습니다.'
        );
    }
    
    // 활성 구독자 조회
    $subscriptions = get_active_subscriptions($options);
    
    if (empty($subscriptions)) {
        return array(
            'success' => false,
            'message' => '발송할 구독자가 없습니다.'
        );
    }
    
    $api_url = rtrim($config['api_server_url'], '/') . '/api/push/multiple';
    
    // API 서버에서 요구하는 payload 형식으로 구성
    $payload = array(
        'title' => $title,
        'body' => $message,  // API 서버에서 'body'를 사용할 수도 있음
        'message' => $message  // 호환성을 위해 둘 다 포함
    );
    
    // 옵션에서 푸시 관련 설정 추출
    $push_options = array();
    if (isset($options['icon'])) $push_options['icon'] = $options['icon'];
    if (isset($options['badge'])) $push_options['badge'] = $options['badge'];
    if (isset($options['click_action'])) $push_options['click_action'] = $options['click_action'];
    if (isset($options['require_interaction'])) $push_options['require_interaction'] = $options['require_interaction'];
    if (isset($options['silent'])) $push_options['silent'] = $options['silent'];
    
    if (!empty($push_options)) {
        $payload = array_merge($payload, $push_options);
    }
    
    $post_data = array(
        'subscriptions' => $subscriptions,
        'payload' => $payload
    );
    
    $result = send_api_request($api_url, $post_data, 'POST', $config['api_key']);
    
    // 로그 기록
    if ($result['success']) {
        log_pushmanager_activity('send', $title, $message, count($subscriptions), 
                                $result['data']['success_count'] ?? 0, 
                                $result['data']['error_count'] ?? 0);
    } else {
        log_pushmanager_activity('error', $title, $message, count($subscriptions), 0, 0, $result['message']);
    }
    
    return $result;
}

/**
 * 관리자용 푸시 발송 함수 (기본 옵션 적용)
 * @param string $title
 * @param string $message
 * @param array $options
 * @return array
 */
function admin_send_push($title, $message, $options = array()) {
    // 기본 옵션 설정
    $default_options = array(
        'icon' => '/img/push-icon.png',
        'badge' => '/img/push-badge.png',
        'click_action' => '/',
        'require_interaction' => false,
        'silent' => false
    );
    
    $options = array_merge($default_options, $options);
    
    return send_push_notification($title, $message, $options);
}

/**
 * 특정 회원에게 푸시 발송
 * @param array $member_ids
 * @param string $title
 * @param string $message
 * @param array $options
 * @return array
 */
function send_push_to_members($member_ids, $title, $message, $options = array()) {
    $options['member_ids'] = $member_ids;
    return admin_send_push($title, $message, $options);
}

/**
 * 최근 가입한 회원들에게 푸시 발송
 * @param int $days
 * @param string $title
 * @param string $message
 * @param array $options
 * @return array
 */
function send_push_to_recent_members($days, $title, $message, $options = array()) {
    $date_limit = date('Y-m-d H:i:s', strtotime("-{$days} days"));
    $options['created_after'] = $date_limit;
    return admin_send_push($title, $message, $options);
}

/**
 * 레벨별 구독 정보 조회
 * @param int $level
 * @return array
 */
function get_subscriptions_by_level($level) {
    return get_active_subscriptions(array('level' => $level));
}

/**
 * 성별 구독 정보 조회
 * @param string $gender
 * @return array
 */
function get_subscriptions_by_gender($gender) {
    return get_active_subscriptions(array('gender' => $gender));
}

/**
 * 회원 ID 목록으로 구독 정보 조회
 * @param array $member_ids
 * @return array
 */
function get_subscriptions_by_member_ids($member_ids) {
    return get_active_subscriptions(array('member_ids' => $member_ids));
}

/**
 * 전체 활성 구독자 조회
 * @return array
 */
function get_all_active_subscriptions() {
    return get_active_subscriptions();
}

/**
 * 공통 JSON 응답 처리
 * @param bool $success
 * @param string $message
 * @param mixed $data
 * @param int $http_code
 */
function send_json_response($success, $message, $data = null, $http_code = 200) {
    header('Content-Type: application/json; charset=utf-8');
    
    if ($http_code !== 200) {
        http_response_code($http_code);
    }
    
    $response = array(
        'success' => $success,
        'message' => $message
    );
    
    if ($data !== null) {
        $response['data'] = $data;
    }
    
    echo json_encode($response);
    exit;
}

/**
 * 푸시 구독 등록 (AJAX 용)
 * @return void
 */
function handle_push_subscription() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        send_json_response(false, 'POST 메서드만 허용됩니다.', null, 405);
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['subscription'])) {
        send_json_response(false, '구독 정보가 필요합니다.', null, 400);
    }
    
    $subscription = $input['subscription'];
    
    // 필수 필드 검증
    if (!isset($subscription['endpoint']) || !isset($subscription['keys']['p256dh']) || !isset($subscription['keys']['auth'])) {
        send_json_response(false, '구독 정보가 올바르지 않습니다.', null, 400);
    }
    
    if (save_push_subscription($subscription)) {
        send_json_response(true, '구독이 등록되었습니다.');
    } else {
        send_json_response(false, '구독 등록에 실패했습니다.', null, 500);
    }
}

/**
 * 푸시 구독 해제 (AJAX 용)
 * @return void
 */
function handle_push_unsubscription() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        send_json_response(false, 'POST 메서드만 허용됩니다.', null, 405);
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['endpoint'])) {
        send_json_response(false, '엔드포인트가 필요합니다.', null, 400);
    }
    
    if (unsubscribe_push($input['endpoint'])) {
        send_json_response(true, '구독이 해제되었습니다.');
    } else {
        send_json_response(false, '구독 해제에 실패했습니다.', null, 500);
    }
}

/**
 * VAPID 공개키 가져오기 (AJAX 용)
 * @return void
 */
function get_vapid_public_key() {
    $config = get_pushmanager_config();
    
    if (!$config || !$config['vapid_public_key']) {
        send_json_response(false, 'VAPID 공개키가 설정되지 않았습니다.', null, 404);
    }
    
    send_json_response(true, '성공', array('vapid_public_key' => $config['vapid_public_key']));
}

/**
 * 최근 푸시 로그 조회
 * @param int $limit
 * @return array
 */
function get_recent_push_logs($limit = 10) {
    $limit = intval($limit);
    
    $sql = "SELECT * FROM pushmanager_logs 
            ORDER BY created_at DESC 
            LIMIT {$limit}";
    
    $result = sql_query($sql);
    $logs = array();
    
    while ($row = sql_fetch_array($result)) {
        $logs[] = $row;
    }
    
    return $logs;
}

/**
 * 푸시 설정 상태 확인
 * @return array
 */
function check_pushmanager_status() {
    $status = array(
        'installed' => false,
        'configured' => false,
        'api_key_issued' => false,
        'active_subscriptions' => 0
    );
    
    // 테이블 존재 확인
    $sql = "SHOW TABLES LIKE 'pushmanager_config'";
    $result = sql_query($sql, false);
    if (sql_num_rows($result) > 0) {
        $status['installed'] = true;
        
        // 설정 확인
        $config = get_pushmanager_config();
        if ($config && $config['site_name'] && $config['api_server_url']) {
            $status['configured'] = true;
            
            // API 키 확인
            if ($config['api_key']) {
                $status['api_key_issued'] = true;
                
                // 활성 구독자 수 확인
                $stats = get_subscription_stats();
                $status['active_subscriptions'] = $stats['active'];
            }
        }
    }
    
    return $status;
}

/**
 * 푸시 매니저 초기화 (테스트용)
 * @return bool
 */
function reset_pushmanager() {
    $tables = array('pushmanager_logs', 'pushmanager_subscriptions', 'pushmanager_config');
    
    foreach ($tables as $table) {
        $sql = "DROP TABLE IF EXISTS {$table}";
        if (!sql_query($sql, false)) {
            return false;
        }
    }
    
    return true;
}

// 하위 호환성을 위해 제거된 함수들을 별칭으로 유지
function send_api_request_with_auth($url, $data = array(), $api_key = '', $method = 'POST') {
    return send_api_request($url, $data, $method, $api_key);
}

function send_multiple_push_notifications($subscriptions, $payload, $options = array()) {
    // API 서버에서 요구하는 형식으로 직접 전송
    $config = get_pushmanager_config();
    
    if (!$config || !$config['api_key']) {
        return array(
            'success' => false,
            'message' => 'API 키가 설정되지 않았습니다.'
        );
    }
    
    $api_url = rtrim($config['api_server_url'], '/') . '/api/push/multiple';
    
    $post_data = array(
        'subscriptions' => $subscriptions,
        'payload' => $payload,
        'options' => (object)$options
    );
    
    return send_api_request($api_url, $post_data, 'POST', $config['api_key']);
}

// ========================================
// 1. 마이그레이션 SQL (install.php에 추가할 코드)
// ========================================

// pushmanager_user_settings 테이블 생성 쿼리
$sql_queries[] = "
    CREATE TABLE IF NOT EXISTS `pushmanager_user_settings` (
        `id` int(11) NOT NULL AUTO_INCREMENT,
        `member_id` varchar(20) DEFAULT NULL COMMENT '회원 ID (NULL = 비회원)',
        `session_id` varchar(64) DEFAULT NULL COMMENT '비회원 세션 ID',
        `message_notification` tinyint(1) NOT NULL DEFAULT 1 COMMENT '쪽지 알림 설정',
        `comment_notification` tinyint(1) NOT NULL DEFAULT 1 COMMENT '댓글 알림 설정',
        `inquiry_notification` tinyint(1) NOT NULL DEFAULT 1 COMMENT '1:1문의 답변 알림 설정',
        `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
        `updated_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (`id`),
        UNIQUE KEY `unique_member` (`member_id`),
        UNIQUE KEY `unique_session` (`session_id`),
        KEY `idx_member_id` (`member_id`),
        KEY `idx_session_id` (`session_id`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COMMENT='사용자별 알림 설정';
";

// ========================================
// 2. 알림 설정 관리 함수들 (pushmanager.lib.php에 추가할 코드)
// ========================================

/**
 * 현재 사용자 식별자 가져오기 (회원 ID 또는 세션 ID)
 * @return array
 */
function get_current_user_identifier() {
    global $member;
    
    if ($member['mb_id']) {
        return array(
            'type' => 'member',
            'id' => $member['mb_id'],
            'member_id' => $member['mb_id'],
            'session_id' => null
        );
    } else {
        $session_id = session_id();
        if (empty($session_id)) {
            session_start();
            $session_id = session_id();
        }
        
        return array(
            'type' => 'guest',
            'id' => $session_id,
            'member_id' => null,
            'session_id' => $session_id
        );
    }
}

/**
 * 사용자 알림 설정 조회
 * @param string $member_id
 * @param string $session_id
 * @return array
 */
function get_user_notification_settings($member_id = null, $session_id = null) {
    if (!$member_id && !$session_id) {
        $user = get_current_user_identifier();
        $member_id = $user['member_id'];
        $session_id = $user['session_id'];
    }
    
    $where_conditions = array();
    if ($member_id) {
        $where_conditions[] = "member_id = '" . addslashes($member_id) . "'";
    } elseif ($session_id) {
        $where_conditions[] = "session_id = '" . addslashes($session_id) . "'";
    } else {
        // 식별자가 없으면 기본값 반환
        return array(
            'message_notification' => 1,
            'comment_notification' => 1,
            'inquiry_notification' => 1
        );
    }
    
    $sql = "SELECT message_notification, comment_notification, inquiry_notification 
            FROM pushmanager_user_settings 
            WHERE " . implode(' AND ', $where_conditions) . " 
            LIMIT 1";
    
    $result = sql_fetch($sql);
    
    if ($result) {
        return array(
            'message_notification' => intval($result['message_notification']),
            'comment_notification' => intval($result['comment_notification']),
            'inquiry_notification' => intval($result['inquiry_notification'])
        );
    } else {
        // 설정이 없으면 기본값 반환 (모든 알림 켜짐)
        return array(
            'message_notification' => 1,
            'comment_notification' => 1,
            'inquiry_notification' => 1
        );
    }
}

/**
 * 사용자 알림 설정 저장
 * @param array $settings
 * @param string $member_id
 * @param string $session_id
 * @return bool
 */
function save_user_notification_settings($settings, $member_id = null, $session_id = null) {
    if (!$member_id && !$session_id) {
        $user = get_current_user_identifier();
        $member_id = $user['member_id'];
        $session_id = $user['session_id'];
    }
    
    // 설정값 검증 및 기본값 처리
    $message_notification = isset($settings['message_notification']) ? intval($settings['message_notification']) : 1;
    $comment_notification = isset($settings['comment_notification']) ? intval($settings['comment_notification']) : 1;
    $inquiry_notification = isset($settings['inquiry_notification']) ? intval($settings['inquiry_notification']) : 1;
    
    // 기존 설정 확인
    $where_conditions = array();
    if ($member_id) {
        $where_conditions[] = "member_id = '" . addslashes($member_id) . "'";
    } elseif ($session_id) {
        $where_conditions[] = "session_id = '" . addslashes($session_id) . "'";
    } else {
        return false; // 식별자가 없으면 실패
    }
    
    $existing_sql = "SELECT id FROM pushmanager_user_settings WHERE " . implode(' AND ', $where_conditions);
    $existing = sql_fetch($existing_sql);
    
    if ($existing) {
        // 업데이트
        $sql = "UPDATE pushmanager_user_settings SET 
                message_notification = {$message_notification},
                comment_notification = {$comment_notification},
                inquiry_notification = {$inquiry_notification},
                updated_at = NOW()
                WHERE id = {$existing['id']}";
    } else {
        // 신규 삽입
        $member_id_value = $member_id ? "'" . addslashes($member_id) . "'" : 'NULL';
        $session_id_value = $session_id ? "'" . addslashes($session_id) . "'" : 'NULL';
        
        $sql = "INSERT INTO pushmanager_user_settings 
                (member_id, session_id, message_notification, comment_notification, inquiry_notification, created_at) 
                VALUES 
                ({$member_id_value}, {$session_id_value}, {$message_notification}, {$comment_notification}, {$inquiry_notification}, NOW())";
    }
    
    return sql_query($sql, false) ? true : false;
}

/**
 * 사용자의 활성 구독 기기 목록 조회
 * @param string $member_id
 * @param string $session_id
 * @return array
 */
function get_user_devices($member_id = null, $session_id = null) {
    if (!$member_id && !$session_id) {
        $user = get_current_user_identifier();
        $member_id = $user['member_id'];
        $session_id = $user['session_id'];
    }
    
    $where_conditions = array();
    if ($member_id) {
        $where_conditions[] = "member_id = '" . addslashes($member_id) . "'";
    } else {
        // 비회원은 IP와 세션으로 추정 (정확하지 않을 수 있음)
        $ip_address = get_real_ip();
        $where_conditions[] = "member_id IS NULL";
        $where_conditions[] = "ip_address = '" . addslashes($ip_address) . "'";
    }
    
    $sql = "SELECT id, endpoint, user_agent, ip_address, is_active, created_at, updated_at
            FROM pushmanager_subscriptions 
            WHERE " . implode(' AND ', $where_conditions) . "
            ORDER BY created_at DESC";
    
    $result = sql_query($sql);
    $devices = array();
    
    while ($row = sql_fetch_array($result)) {
        // User Agent 파싱해서 기기 정보 추출
        $device_info = parse_user_agent($row['user_agent']);
        
        $devices[] = array(
            'id' => $row['id'],
            'endpoint' => $row['endpoint'],
            'user_agent' => $row['user_agent'],
            'device_name' => $device_info['device_name'],
            'device_icon' => $device_info['device_icon'],
            'device_description' => $device_info['device_description'],
            'ip_address' => $row['ip_address'],
            'is_active' => intval($row['is_active']),
            'status' => $device_info['status'],
            'created_at' => $row['created_at'],
            'updated_at' => $row['updated_at']
        );
    }
    
    return $devices;
}

/**
 * User Agent 파싱하여 기기 정보 추출
 * @param string $user_agent
 * @return array
 */
function parse_user_agent($user_agent) {
    $device_name = 'Unknown Device';
    $device_icon = '🖥️';
    $device_description = $user_agent;
    $status = '● 온라인'; // 실제로는 마지막 활동 시간으로 판단해야 함
    
    if (empty($user_agent)) {
        return array(
            'device_name' => $device_name,
            'device_icon' => $device_icon,
            'device_description' => 'User Agent 정보 없음',
            'status' => '● 오프라인'
        );
    }
    
    // Windows
    if (strpos($user_agent, 'Windows') !== false) {
        if (strpos($user_agent, 'Chrome') !== false) {
            $device_name = 'Chrome on Windows';
            $device_icon = '💻';
        } elseif (strpos($user_agent, 'Firefox') !== false) {
            $device_name = 'Firefox on Windows';
            $device_icon = '💻';
        } elseif (strpos($user_agent, 'Edge') !== false) {
            $device_name = 'Edge on Windows';
            $device_icon = '💻';
        } else {
            $device_name = 'Windows Device';
            $device_icon = '💻';
        }
    }
    // macOS
    elseif (strpos($user_agent, 'Macintosh') !== false || strpos($user_agent, 'Mac OS X') !== false) {
        if (strpos($user_agent, 'Chrome') !== false) {
            $device_name = 'Chrome on macOS';
            $device_icon = '🖥️';
        } elseif (strpos($user_agent, 'Safari') !== false && strpos($user_agent, 'Chrome') === false) {
            $device_name = 'Safari on macOS';
            $device_icon = '🖥️';
        } elseif (strpos($user_agent, 'Firefox') !== false) {
            $device_name = 'Firefox on macOS';
            $device_icon = '🖥️';
        } else {
            $device_name = 'macOS Device';
            $device_icon = '🖥️';
        }
    }
    // iPhone
    elseif (strpos($user_agent, 'iPhone') !== false) {
        if (strpos($user_agent, 'Safari') !== false) {
            $device_name = 'Safari on iPhone';
            $device_icon = '📱';
        } elseif (strpos($user_agent, 'Chrome') !== false) {
            $device_name = 'Chrome on iPhone';
            $device_icon = '📱';
        } else {
            $device_name = 'iPhone';
            $device_icon = '📱';
        }
    }
    // Android
    elseif (strpos($user_agent, 'Android') !== false) {
        if (strpos($user_agent, 'Chrome') !== false) {
            $device_name = 'Chrome on Android';
            $device_icon = '🤖';
        } elseif (strpos($user_agent, 'Firefox') !== false) {
            $device_name = 'Firefox on Android';
            $device_icon = '🤖';
        } else {
            $device_name = 'Android Device';
            $device_icon = '🤖';
        }
    }
    // iPad
    elseif (strpos($user_agent, 'iPad') !== false) {
        $device_name = 'Safari on iPad';
        $device_icon = '📱';
    }
    // Linux
    elseif (strpos($user_agent, 'Linux') !== false) {
        $device_name = 'Linux Device';
        $device_icon = '🐧';
    }
    
    // User Agent 길이 제한 (화면 표시용)
    if (strlen($user_agent) > 80) {
        $device_description = substr($user_agent, 0, 77) . '...';
    }
    
    return array(
        'device_name' => $device_name,
        'device_icon' => $device_icon,
        'device_description' => $device_description,
        'status' => $status
    );
}

/**
 * 특정 기기의 알림 활성화/비활성화
 * @param int $subscription_id
 * @param bool $is_active
 * @return bool
 */
function toggle_device_notification($subscription_id, $is_active) {
    $subscription_id = intval($subscription_id);
    $is_active = $is_active ? 1 : 0;
    
    $sql = "UPDATE pushmanager_subscriptions SET 
            is_active = {$is_active},
            updated_at = NOW()
            WHERE id = {$subscription_id}";
    
    return sql_query($sql, false) ? true : false;
}

/**
 * 통합 알림 발송 함수 (사용자 설정 확인)
 * @param string $notification_type 알림 타입 ('comment', 'message', 'inquiry', 'system')
 * @param array $member_ids 수신자 회원 ID 배열
 * @param string $title 푸시 제목
 * @param string $message 푸시 메시지
 * @param array $options 추가 옵션
 * @return array
 */
function send_notification($notification_type, $member_ids, $title, $message, $options = array()) {
    // 지원하는 알림 타입 정의
    $valid_types = array('comment', 'message', 'inquiry', 'system');
    
    if (!in_array($notification_type, $valid_types)) {
        return array(
            'success' => false,
            'message' => '지원하지 않는 알림 타입입니다: ' . $notification_type
        );
    }
    
    // member_ids가 배열이 아니면 배열로 변환
    if (!is_array($member_ids)) {
        $member_ids = array($member_ids);
    }
    
    // 빈 배열 체크
    if (empty($member_ids)) {
        return array(
            'success' => false,
            'message' => '수신자가 지정되지 않았습니다.'
        );
    }
    
    // system 타입은 설정 무시하고 무조건 발송
    if ($notification_type === 'system') {
        return send_push_notification($title, $message, array_merge($options, array('member_ids' => $member_ids)));
    }
    
    // 각 회원별로 설정 확인 후 필터링
    $valid_subscriptions = array();
    $checked_members = array();
    $skipped_members = array();
    
    foreach ($member_ids as $member_id) {
        if (empty($member_id)) continue;
        
        // 사용자 알림 설정 확인
        $user_settings = get_user_notification_settings($member_id);
        $setting_key = $notification_type . '_notification';
        
        // 해당 알림 타입이 활성화되어 있는지 확인
        if (isset($user_settings[$setting_key]) && $user_settings[$setting_key]) {
            // 활성화되어 있으면 해당 회원의 구독 정보 가져오기
            $member_subscriptions = get_active_subscriptions(array('member_ids' => array($member_id)));
            $valid_subscriptions = array_merge($valid_subscriptions, $member_subscriptions);
            $checked_members[] = $member_id;
        } else {
            // 비활성화되어 있으면 건너뜀
            $skipped_members[] = $member_id;
        }
    }
    
    // 결과 로깅
    if (!empty($skipped_members)) {
        $skipped_list = implode(', ', $skipped_members);
        log_pushmanager_activity('skip', $title, "알림 설정으로 건너뜀 ({$notification_type}): {$skipped_list}");
    }
    
    // 발송할 구독자가 없으면 종료
    if (empty($valid_subscriptions)) {
        return array(
            'success' => false,
            'message' => '알림 설정으로 인해 발송할 구독자가 없습니다.',
            'details' => array(
                'total_members' => count($member_ids),
                'checked_members' => $checked_members,
                'skipped_members' => $skipped_members,
                'notification_type' => $notification_type
            )
        );
    }
    
    // 기본 옵션 설정
    $default_options = array(
        'icon' => get_push_icon_url(),
        'badge' => get_push_badge_url(),
        'require_interaction' => false,
        'silent' => false
    );
    
    // 알림 타입별 기본 옵션 추가
    switch ($notification_type) {
        case 'message':
            $default_options['require_interaction'] = true;
            $default_options['tag'] = 'message_' . time();
            break;
            
        case 'comment':
            $default_options['require_interaction'] = false;
            $default_options['tag'] = 'comment_' . time();
            break;
            
        case 'inquiry':
            $default_options['require_interaction'] = true;
            $default_options['tag'] = 'inquiry_' . time();
            break;
    }
    
    // 사용자 옵션과 병합
    $final_options = array_merge($default_options, $options);
    
    // 실제 푸시 발송
    $result = send_push_notification($title, $message, $final_options);
    
    // 결과에 추가 정보 포함
    if ($result['success']) {
        $result['details'] = array(
            'notification_type' => $notification_type,
            'total_members' => count($member_ids),
            'valid_members' => $checked_members,
            'skipped_members' => $skipped_members,
            'sent_count' => count($valid_subscriptions)
        );
    }
    
    return $result;
}

/**
 * 시스템 알림 발송 (설정 무시)
 * @param array $member_ids
 * @param string $title
 * @param string $message
 * @param array $options
 * @return array
 */
function send_system_notification($member_ids, $title, $message, $options = array()) {
    return send_notification('system', $member_ids, $title, $message, $options);
}

/**
 * 알림 타입 유효성 검사
 * @param string $notification_type
 * @return bool
 */
function is_valid_notification_type($notification_type) {
    $valid_types = array('comment', 'message', 'inquiry', 'system');
    return in_array($notification_type, $valid_types);
}

/**
 * 사용자의 특정 알림 타입 설정 확인
 * @param string $member_id
 * @param string $notification_type
 * @return bool
 */
function is_notification_enabled($member_id, $notification_type) {
    if (!is_valid_notification_type($notification_type)) {
        return false;
    }
    
    // system 타입은 항상 활성화
    if ($notification_type === 'system') {
        return true;
    }
    
    $user_settings = get_user_notification_settings($member_id);
    $setting_key = $notification_type . '_notification';
    
    return isset($user_settings[$setting_key]) && $user_settings[$setting_key];
}

/**
 * 여러 회원의 알림 설정 일괄 확인
 * @param array $member_ids
 * @param string $notification_type
 * @return array ['enabled' => [...], 'disabled' => [...]]
 */
function check_notification_settings($member_ids, $notification_type) {
    $result = array(
        'enabled' => array(),
        'disabled' => array(),
        'notification_type' => $notification_type
    );
    
    foreach ($member_ids as $member_id) {
        if (is_notification_enabled($member_id, $notification_type)) {
            $result['enabled'][] = $member_id;
        } else {
            $result['disabled'][] = $member_id;
        }
    }
    
    return $result;
}

/**
 * 알림 발송 통계
 * @param string $notification_type (선택사항)
 * @return array
 */
function get_notification_stats($notification_type = null) {
    $where_condition = '';
    if ($notification_type && is_valid_notification_type($notification_type)) {
        $where_condition = "WHERE type = '" . addslashes($notification_type) . "'";
    }
    
    $sql = "SELECT 
                type,
                COUNT(*) as total_count,
                SUM(target_count) as total_targets,
                SUM(success_count) as total_success,
                SUM(error_count) as total_errors,
                DATE(created_at) as date
            FROM pushmanager_logs 
            {$where_condition}
            GROUP BY type, DATE(created_at)
            ORDER BY created_at DESC
            LIMIT 30";
    
    $result = sql_query($sql);
    $stats = array();
    
    while ($row = sql_fetch_array($result)) {
        $stats[] = array(
            'type' => $row['type'],
            'date' => $row['date'],
            'total_count' => intval($row['total_count']),
            'total_targets' => intval($row['total_targets']),
            'total_success' => intval($row['total_success']),
            'total_errors' => intval($row['total_errors']),
            'success_rate' => $row['total_targets'] > 0 ? round(($row['total_success'] / $row['total_targets']) * 100, 2) : 0
        );
    }
    
    return $stats;
}

// ========================================
// 3. AJAX 처리 함수들
// ========================================

/**
 * 알림 설정 저장 AJAX 처리
 */
function handle_save_notification_settings() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        send_json_response(false, 'POST 메서드만 허용됩니다.', null, 405);
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        // Form 데이터로 전송된 경우
        $input = $_POST;
    }
    
    $settings = array(
        'message_notification' => isset($input['message_notification']) ? intval($input['message_notification']) : 0,
        'comment_notification' => isset($input['comment_notification']) ? intval($input['comment_notification']) : 0,
        'inquiry_notification' => isset($input['inquiry_notification']) ? intval($input['inquiry_notification']) : 0
    );
    
    // 기기별 설정 처리
    $device_settings = array();
    if (isset($input['devices']) && is_array($input['devices'])) {
        foreach ($input['devices'] as $device) {
            if (isset($device['id']) && isset($device['enabled'])) {
                $device_settings[] = array(
                    'id' => intval($device['id']),
                    'enabled' => intval($device['enabled'])
                );
            }
        }
    }
    
    // 알림 설정 저장
    $settings_saved = save_user_notification_settings($settings);
    
    // 기기별 설정 저장
    $devices_saved = true;
    foreach ($device_settings as $device) {
        if (!toggle_device_notification($device['id'], $device['enabled'])) {
            $devices_saved = false;
        }
    }
    
    if ($settings_saved && $devices_saved) {
        send_json_response(true, '알림 설정이 저장되었습니다.');
    } else {
        send_json_response(false, '설정 저장 중 오류가 발생했습니다.', null, 500);
    }
}

/**
 * 현재 알림 설정 조회 AJAX 처리
 */
function handle_get_notification_settings() {
    $settings = get_user_notification_settings();
    $devices = get_user_devices();
    
    send_json_response(true, '성공', array(
        'settings' => $settings,
        'devices' => $devices
    ));
}

/**
 * 기기별 알림 토글 AJAX 처리
 */
function handle_toggle_device_notification() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        send_json_response(false, 'POST 메서드만 허용됩니다.', null, 405);
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['device_id']) || !isset($input['is_active'])) {
        send_json_response(false, '기기 ID와 활성화 상태가 필요합니다.', null, 400);
    }
    
    $device_id = intval($input['device_id']);
    $is_active = intval($input['is_active']);
    
    if (toggle_device_notification($device_id, $is_active)) {
        send_json_response(true, '기기 알림 설정이 변경되었습니다.');
    } else {
        send_json_response(false, '기기 알림 설정 변경에 실패했습니다.', null, 500);
    }
}

/**
 * 회원이 탈퇴할 때 알림 설정 정리
 * @param string $member_id
 * @return bool
 */
function cleanup_member_notification_data($member_id) {
    $member_id = addslashes($member_id);
    
    // 알림 설정 삭제
    $sql1 = "DELETE FROM pushmanager_user_settings WHERE member_id = '{$member_id}'";
    
    // 구독 정보 비활성화 (완전 삭제 대신 비활성화)
    $sql2 = "UPDATE pushmanager_subscriptions SET is_active = 0, member_id = NULL WHERE member_id = '{$member_id}'";
    
    $result1 = sql_query($sql1, false);
    $result2 = sql_query($sql2, false);
    
    return $result1 && $result2;
}
?>