// PWA 설치 유도 팝업 생성 및 관리 클래스
class PWAInstallPopup {
    constructor(options = {}) {
        this.options = {
            title: options.title || '앱으로 설치하세요!',
            description: options.description || '더 빠르고 편리한 앱 경험을 위해 홈 화면에 설치해보세요.',
            installButtonText: options.installButtonText || '설치하기',
            laterButtonText: options.laterButtonText || '나중에',
            showDelay: options.showDelay || 3000,
            dismissHours: options.dismissHours || 24,
            installUrl: options.installUrl || 'https://pushmanager.kr/plugin/pushmanager/pwa_install.php',
            ...options
        };
        
        this.deferredPrompt = null;
        this.popup = null;
        this.isInitialized = false;
        this.canInstallNatively = false;
        
        this.init();
    }
    
    // 초기화
    init() {
        if (this.isInitialized) return;
        
        this.createStyles();
        this.createPopupHTML();
        this.bindEvents();
        this.checkAndShowPopup();
        
        this.isInitialized = true;
    }
    
    // CSS 스타일 생성
    createStyles() {
        const styleId = 'pwa-install-popup-styles';
        if (document.getElementById(styleId)) return;
        
        const style = document.createElement('style');
        style.id = styleId;
        style.textContent = `
            .pwa-install-popup {
                position: fixed;
                bottom: 0;
                left: 0;
                width: 100%;
                background: white;
                display: none;
                z-index: 99999;
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                animation: pwa-slideUpFromBottom 0.3s ease;
                box-shadow: 0 -4px 20px rgba(0,0,0,0.15);
                border-top: 1px solid #e0e0e0;
            }
            
            .pwa-install-popup.show {
                display: flex !important;
            }
            
            .pwa-popup-content {
                background: white;
                padding: 1rem 1.2rem;
                width: 100%;
                display: flex;
                align-items: center;
                justify-content: space-between;
                position: relative;
                animation: pwa-slideUpFromBottom 0.4s ease;
                max-height: none;
                overflow: visible;
                text-align: left;
            }
            
            .pwa-close-btn {
                position: absolute;
                top: 8px;
                right: 12px;
                background: none;
                border: none;
                font-size: 20px;
                cursor: pointer;
                color: #999;
                transition: color 0.3s;
                z-index: 1;
                padding: 4px;
            }
            
            .pwa-close-btn:hover {
                color: #333;
            }
            
            .pwa-popup-icon {
                font-size: 2.5rem;
                margin-bottom: 0.8rem;
            }
            
            .pwa-popup-title {
                font-size: 1rem;
                margin: 0 0 0.3rem 0;
                color: #333;
                font-weight: 600;
            }
            
            .pwa-popup-description {
                color: #666;
                margin: 0;
                line-height: 1.4;
                font-size: 0.85rem;
            }
            
            .pwa-text-content {
                flex: 1;
                padding-right: 1rem;
            }
            
            .pwa-button-group {
                display: flex;
                gap: 8px;
                flex-shrink: 0;
            }
            .pwa-install-btn {
                background: #007AFF;
                color: white;
                border: none;
                padding: 8px 16px;
                border-radius: 6px;
                font-size: 0.9rem;
                font-weight: 500;
                cursor: pointer;
                transition: background-color 0.3s;
                white-space: nowrap;
            }
            
            .pwa-install-btn:hover {
                background: #0056D6;
            }
            
            .pwa-install-btn:disabled {
                background: #ccc;
                cursor: not-allowed;
            }
            
            .pwa-later-btn {
                background: transparent;
                color: #666;
                border: 1px solid #ddd;
                padding: 8px 16px;
                border-radius: 6px;
                font-size: 0.9rem;
                cursor: pointer;
                transition: all 0.3s;
                white-space: nowrap;
            }
            
            .pwa-debug-btn {
                background: #FF9500;
                color: white;
                border: none;
                padding: 6px 12px;
                border-radius: 4px;
                font-size: 0.8rem;
                cursor: pointer;
                transition: background-color 0.3s;
                white-space: nowrap;
                margin-left: 8px;
            }
            
            .pwa-debug-btn:hover {
                background: #E68400;
            }
            
            .pwa-features {
                display: flex;
                justify-content: space-around;
                margin: 1.5rem 0;
                text-align: center;
                gap: 1rem;
            }
            
            .pwa-feature {
                display: flex;
                flex-direction: column;
                align-items: center;
                flex: 1;
            }
            
            .pwa-feature-icon {
                font-size: 2rem;
                margin-bottom: 0.5rem;
                color: #667eea;
            }
            
            .pwa-feature-text {
                font-size: 0.9rem;
                color: #666;
            }
            
            @keyframes pwa-slideUpFromBottom {
                from { 
                    transform: translateY(100%);
                }
                to { 
                    transform: translateY(0);
                }
            }
            
            @media (max-width: 480px) {
                .pwa-popup-content {
                    flex-direction: column;
                    align-items: flex-start;
                    padding: 1rem;
                }
                
                .pwa-text-content {
                    padding-right: 0;
                    margin-bottom: 1rem;
                }
                
                .pwa-button-group {
                    width: 100%;
                    justify-content: space-between;
                }
                
                .pwa-install-btn, .pwa-later-btn {
                    flex: 1;
                }
            }
        `;
        
        document.head.appendChild(style);
    }
    
    // 팝업 HTML 생성
    createPopupHTML() {
        const popupId = 'pwa-install-popup';
        if (document.getElementById(popupId)) return;
        
        const popup = document.createElement('div');
        popup.id = popupId;
        popup.className = 'pwa-install-popup';
        
        popup.innerHTML = `
            <div class="pwa-popup-content">
                <button class="pwa-close-btn">&times;</button>
                
                <div class="pwa-text-content">
                    <h2 class="pwa-popup-title">${this.options.title}</h2>
                    <p class="pwa-popup-description">${this.options.description}</p>
                </div>
                
                <div class="pwa-button-group">
                    <button class="pwa-install-btn">${this.options.installButtonText}</button>
                    <button class="pwa-later-btn">${this.options.laterButtonText}</button>
                </div>
            </div>
        `;
        
        document.body.appendChild(popup);
        this.popup = popup;
    }
    
    // 이벤트 바인딩
    bindEvents() {
        if (!this.popup) return;
        
        // 닫기 버튼
        this.popup.querySelector('.pwa-close-btn').addEventListener('click', () => {
            this.hidePopup();
        });
        
        // 설치 버튼
        this.popup.querySelector('.pwa-install-btn').addEventListener('click', () => {
            this.handleInstallClick();
        });
        
        // 나중에 버튼
        this.popup.querySelector('.pwa-later-btn').addEventListener('click', () => {
            this.dismissPopup();
        });
        
        // beforeinstallprompt 이벤트
        window.addEventListener('beforeinstallprompt', (e) => {
            e.preventDefault();
            this.deferredPrompt = e;
            this.canInstallNatively = true;
            this.updateInstallButton();
        });
        
        // PWA 설치 완료 이벤트
        window.addEventListener('appinstalled', () => {
            this.clearDismissData();
            this.hidePopup();
        });
    }
    
    // 설치 버튼 텍스트 업데이트
    updateInstallButton() {
        const installBtn = this.popup?.querySelector('.pwa-install-btn');
        if (!installBtn) return;
        
        // 강제로 Samsung Internet 처리
        if (this.isSamsungBrowser()) {
            installBtn.textContent = this.options.installButtonText;
            installBtn.disabled = false;
            return;
        }
        
        // iOS는 항상 "자세히보기"
        if (this.isIOSDevice()) {
            installBtn.textContent = '자세히보기';
            installBtn.disabled = false;
            return;
        }
        
        // 네이티브 설치 가능한 경우 "설치하기"
        if (this.canInstallNatively || this.deferredPrompt) {
            installBtn.textContent = this.options.installButtonText;
            installBtn.disabled = false;
        } else {
            installBtn.textContent = '자세히보기';
            installBtn.disabled = false;
        }
    }
    
    // 설치 버튼 클릭 처리
    handleInstallClick() {
        // iOS는 항상 안내 페이지로
        if (this.isIOSDevice()) {
            this.goToInstallPage();
            return;
        }
        
        // Samsung Internet 처리
        if (this.isSamsungBrowser()) {
            this.installPWA(); // Samsung Internet 전용 처리
            return;
        }
        
        // 네이티브 설치 가능한 경우
        if ((this.canInstallNatively && this.deferredPrompt) || this.deferredPrompt) {
            this.installPWA();
        } else {
            // 기타 경우 안내 페이지로 이동
            this.goToInstallPage();
        }
    }
    
    // PWA 설치 페이지로 이동
    goToInstallPage() {
        window.location.href = this.options.installUrl;
    }
    
    // 모바일 디바이스 감지 (User Agent 기반)
    isMobileDevice() {
        const userAgent = navigator.userAgent.toLowerCase();
        
        // iOS 디바이스
        if (/iphone|ipad|ipod/.test(userAgent)) {
            return true;
        }
        
        // Android 디바이스
        if (/android/.test(userAgent)) {
            return true;
        }
        
        // 기타 모바일 디바이스
        if (/webos|blackberry|iemobile|opera mini|mobile|phone|tablet/.test(userAgent)) {
            return true;
        }
        
        // Windows Phone
        if (/windows phone/.test(userAgent)) {
            return true;
        }
        
        return false;
    }
    
    // PWA 설치 지원 플랫폼 감지
    isPWASupportedPlatform() {
        const userAgent = navigator.userAgent.toLowerCase();
        
        // iOS (Safari 11.3+ PWA 지원)
        if (/iphone|ipad|ipod/.test(userAgent)) {
            return true;
        }
        
        // Android (Chrome PWA 지원)
        if (/android/.test(userAgent)) {
            return true;
        }
        
        // 데스크톱 Chrome, Edge, Firefox 등도 PWA 지원
        if (/chrome|edge|firefox/.test(userAgent)) {
            return true;
        }
        
        return false;
    }
    
    // PWA 환경 감지
    isPWAEnvironment() {
        // iOS Safari PWA 감지
        if (window.navigator.standalone === true) {
            return true;
        }
        
        // Android Chrome PWA 감지
        if (window.matchMedia('(display-mode: standalone)').matches) {
            return true;
        }
        
        // 기타 PWA 환경 감지
        if (document.referrer.includes('android-app://')) {
            return true;
        }
        
        return false;
    }
    
    // iOS 디바이스 감지
    isIOSDevice() {
        return /iPad|iPhone|iPod/.test(navigator.userAgent) && !window.MSStream;
    }
    
    // 갤럭시 브라우저(Samsung Internet) 감지
    isSamsungBrowser() {
        const userAgent = navigator.userAgent.toLowerCase();
        return /samsung/.test(userAgent) || 
               /samsungbrowser/.test(userAgent) || 
               userAgent.includes('samsung') ||
               userAgent.includes('samsungbrowser');
    }
    
    // 디버깅용 정보 표시 (모바일에서 쉽게 확인)
    showDebugInfo() {
        const userAgent = navigator.userAgent;
        const isSamsung = this.isSamsungBrowser();
        const isIOS = this.isIOSDevice();
        const canInstall = this.canInstallNatively;
        const hasPrompt = !!this.deferredPrompt;
        const isPWAEnv = this.isPWAEnvironment();
        const isMobile = this.isMobileDevice();
        const isPWASupported = this.isPWASupportedPlatform();
        
        // 현재 버튼 텍스트도 확인
        const currentButtonText = this.popup?.querySelector('.pwa-install-btn')?.textContent || '버튼 없음';
        
        const debugInfo = [
            `🔍 PWA 디버깅 정보`,
            ``,
            `📱 디바이스 감지:`,
            `• 모바일 디바이스: ${isMobile}`,
            `• Samsung Internet: ${isSamsung}`,
            `• iOS 디바이스: ${isIOS}`,
            `• PWA 지원 플랫폼: ${isPWASupported}`,
            `• PWA 환경에서 실행: ${isPWAEnv}`,
            ``,
            `⚙️ 설치 상태:`,
            `• 네이티브 설치 가능: ${canInstall}`,
            `• beforeinstallprompt: ${hasPrompt}`,
            ``,
            `🔘 버튼 상태:`,
            `• 현재 버튼 텍스트: "${currentButtonText}"`,
            `• 예상 버튼: ${this.getExpectedButtonText()}`,
            ``,
            `🌐 User Agent:`,
            `${userAgent}`,
            ``,
            `🛠️ 문제 해결: Samsung이면서 "자세히보기"라면`,
            `updateInstallButton() 함수에 문제가 있습니다.`
        ].join('\n');
        
        alert(debugInfo);
        
        // 추가 디버깅: updateInstallButton 강제 실행
        console.log('🔧 updateInstallButton 강제 실행...');
        this.updateInstallButton();
    }
    
    // 예상 버튼 텍스트 반환
    getExpectedButtonText() {
        if (this.isIOSDevice()) return '자세히보기 (iOS)';
        if (this.isSamsungBrowser()) return '설치하기 (Samsung)';
        if (this.canInstallNatively || this.deferredPrompt) return '설치하기 (Native)';
        return '자세히보기 (Fallback)';
    }
    
    // 네이티브 설치 가능 여부 확인
    checkNativeInstallSupport() {
        const userAgent = navigator.userAgent.toLowerCase();
        
        // 디버깅을 위해 User Agent 로깅
        console.log('User Agent:', navigator.userAgent);
        
        // iOS는 네이티브 설치 불가 (정책상 제한) - 수동 설치 안내 필요
        if (this.isIOSDevice()) {
            this.canInstallNatively = false;
            console.log('iOS 디바이스: 수동 설치 안내가 필요합니다.');
            return;
        }
        
        // Samsung Internet (갤럭시 브라우저) 우선 처리
        if (this.isSamsungBrowser()) {
            this.canInstallNatively = true;
            console.log('Samsung Internet 감지: PWA 네이티브 설치 지원');
            
            // 모바일에서 쉽게 확인할 수 있도록 alert 추가 (개발 시에만 사용)
            // this.showDebugInfo();
            return;
        }
        
        // Android 기반 브라우저들
        if (/android/.test(userAgent)) {
            // Chrome (Samsung이 아닌 경우)
            if (/chrome/.test(userAgent)) {
                console.log('Android Chrome: beforeinstallprompt 이벤트 대기 중...');
                return;
            }
            
            // 기타 Android 브라우저 (Edge, Firefox 등)
            if (/edge|firefox/.test(userAgent)) {
                console.log('Android 기타 브라우저: beforeinstallprompt 이벤트 대기 중...');
                return;
            }
            
            // 기타 Android 브라우저들도 시도해볼 수 있도록 설정
            console.log('Android 브라우저: beforeinstallprompt 이벤트 대기 중...');
            return;
        }
        
        // 데스크톱 Chrome, Edge 등
        if (/chrome|edge/.test(userAgent) && !/mobile/.test(userAgent)) {
            console.log('데스크톱 브라우저: beforeinstallprompt 이벤트 대기 중...');
            return;
        }
        
        // 기타 브라우저는 기본적으로 안내 페이지로 이동
        this.canInstallNatively = false;
        console.log('기타 브라우저: 안내 페이지로 이동합니다.');
    }
    
    // 팝업 표시 여부 확인 및 표시
    checkAndShowPopup() {
        // PWA를 지원하지 않는 플랫폼에서는 팝업 표시하지 않음
        if (!this.isPWASupportedPlatform()) {
            return;
        }
        
        // PWA 환경에서 이미 실행 중인 경우 팝업 표시하지 않음
        if (this.isPWAEnvironment()) {
            return;
        }
        
        // 네이티브 설치 지원 여부 확인
        this.checkNativeInstallSupport();
        
        // 이전에 팝업을 닫았는지 확인
        const dismissed = localStorage.getItem('pwa-install-dismissed');
        const dismissedTime = localStorage.getItem('pwa-install-dismissed-time');
        
        if (dismissed && dismissedTime) {
            const timeDiff = Date.now() - parseInt(dismissedTime);
            const hoursDiff = timeDiff / (1000 * 60 * 60);
            
            if (hoursDiff < this.options.dismissHours) {
                return;
            }
        }
        
        // 지연 후 팝업 표시
        setTimeout(() => {
            this.showPopup();
        }, this.options.showDelay);
    }
    
    // 팝업 표시
    showPopup() {
        if (this.popup) {
            // 먼저 팝업을 표시하고
            this.popup.classList.add('show');
            
            // 그 다음에 버튼 업데이트 (DOM이 완전히 로드된 후)
            setTimeout(() => {
                this.updateInstallButton();
            }, 100);
        }
    }
    
    // 팝업 숨기기
    hidePopup() {
        if (this.popup) {
            this.popup.classList.remove('show');
        }
    }
    
    // PWA 네이티브 설치
    installPWA() {
        if (this.deferredPrompt) {
            const installBtn = this.popup?.querySelector('.pwa-install-btn');
            if (installBtn) {
                installBtn.disabled = true;
                installBtn.textContent = '설치 중...';
            }
            
            this.deferredPrompt.prompt();
            this.deferredPrompt.userChoice.then((choiceResult) => {
                if (choiceResult.outcome === 'accepted') {
                    this.clearDismissData();
                } else {
                    if (installBtn) {
                        installBtn.disabled = false;
                        installBtn.textContent = this.options.installButtonText;
                    }
                }
                this.deferredPrompt = null;
                this.canInstallNatively = false;
            });
            
            this.hidePopup();
        } else {
            // beforeinstallprompt가 없는 경우
            if (this.isSamsungBrowser()) {
                // Samsung Internet은 자체 설치 UI를 제공하므로 안내 메시지 표시
                alert('메뉴(⋮) → "앱으로 추가" 또는 "홈 화면에 추가"를 선택하여 앱을 설치할 수 있습니다.');
                this.hidePopup();
                return;
            }
            
            this.goToInstallPage();
        }
    }
    
    // 팝업 무시 (나중에)
    dismissPopup() {
        localStorage.setItem('pwa-install-dismissed', 'true');
        localStorage.setItem('pwa-install-dismissed-time', Date.now().toString());
        this.hidePopup();
    }
    
    // 무시 데이터 삭제
    clearDismissData() {
        localStorage.removeItem('pwa-install-dismissed');
        localStorage.removeItem('pwa-install-dismissed-time');
    }
    
    // 팝업 제거
    destroy() {
        if (this.popup) {
            this.popup.remove();
            this.popup = null;
        }
        
        const style = document.getElementById('pwa-install-popup-styles');
        if (style) {
            style.remove();
        }
        
        this.isInitialized = false;
    }
}

// 전역 함수로 간편하게 사용할 수 있도록 제공
function initPWAInstallPopup(options = {}) {
    // 이미 초기화된 경우 중복 방지
    if (window.pwaInstallPopup) {
        console.log('PWA 설치 팝업이 이미 초기화되었습니다.');
        return window.pwaInstallPopup;
    }
    
    // DOM이 로드된 후 실행
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            window.pwaInstallPopup = new PWAInstallPopup(options);
        });
    } else {
        window.pwaInstallPopup = new PWAInstallPopup(options);
    }
    
    return window.pwaInstallPopup;
}

// 자동 초기화 (원한다면 이 부분을 주석 처리)
initPWAInstallPopup();

// 디버깅용 - 개발 시에만 사용
// 갤럭시 브라우저에서 쉽게 정보 확인하려면 주석을 해제하세요
// window.showPWADebug = () => window.pwaInstallPopup?.showDebugInfo();
/*
// 기본 설정으로 초기화
initPWAInstallPopup();

// 커스텀 설정으로 초기화
initPWAInstallPopup({
    title: '우리 앱을 설치하세요!',
    description: '더 나은 사용자 경험을 위해 앱을 설치해보세요.',
    installButtonText: '지금 설치',
    laterButtonText: '다음에',
    showDelay: 5000,        // 5초 후 표시
    dismissHours: 48        // 48시간 동안 숨김
});

// 수동으로 팝업 표시
window.pwaInstallPopup.showPopup();

// 팝업 제거
window.pwaInstallPopup.destroy();
*/