<?php
// notification_settings.php (표시 전용)
include_once('./_common.php');

if(!$is_member) {
    alert("로그인 후 이용해주세요.");
}

// pushmanager.lib.php 포함
$pushmanager_lib = G5_LIB_PATH . '/pushmanager.lib.php';
if (file_exists($pushmanager_lib)) {
    include_once($pushmanager_lib);
} else {
    alert('Push Manager 라이브러리를 찾을 수 없습니다.', G5_URL);
}

// 현재 설정 불러오기
$current_settings = array(
    'message_notification' => 1,
    'comment_notification' => 1,  
    'inquiry_notification' => 1
);

$user_devices = array();

if (function_exists('get_user_notification_settings')) {
    $current_settings = get_user_notification_settings();
}

if (function_exists('get_user_devices')) {
    $user_devices = get_user_devices();
}

$g5['title'] = '알림 설정';
include_once(G5_PATH.'/head.php');
?>

<!-- Push Manager JS 로드 -->
<script src="./pushmanager.js"></script>

<style>
.notification-container {
    width: 100%;
    padding: 40px 20px;
    background: #fff;
}

.page-header {
    text-align: center;
    margin-bottom: 50px;
    padding: 50px 0;
    background: #f8f9fa;
    border-radius: 0;
    border-bottom: 1px solid #e9ecef;
}

.page-header-content {
    max-width: 600px;
    margin: 0 auto;
}

.page-title {
    font-size: 36px;
    margin: 0 0 16px 0;
    color: #212529;
    border: none;
    padding: 0;
    font-weight: 300;
    letter-spacing: -1px;
}

.page-subtitle {
    color: #6c757d;
    margin: 0;
    font-size: 16px;
    line-height: 1.6;
    font-weight: 400;
}

.section {
    margin: 50px 0;
}

.section-title {
    font-size: 22px;
    margin-bottom: 25px;
    color: #333;
    display: flex;
    align-items: center;
    font-weight: 600;
}

.section-title .emoji {
    margin-right: 12px;
    font-size: 24px;
}

/* 푸시 구독 섹션 스타일 */
.push-subscription-section {
    background: #fff;
    border: 1px solid #e2e8f0;
    border-radius: 12px;
    padding: 24px;
    margin-bottom: 30px;
}

.push-subscription-header {
    display: flex;
    align-items: center;
    margin-bottom: 16px;
}

.push-subscription-icon {
    font-size: 20px;
    margin-right: 8px;
    color: #3b82f6;
}

.push-subscription-title {
    font-size: 18px;
    margin: 0;
    font-weight: 600;
    color: #1e293b;
}

.push-subscription-description {
    font-size: 14px;
    line-height: 1.5;
    margin-bottom: 20px;
    color: #64748b;
}

.push-status-container {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 16px;
    background: #f8fafc;
    border-radius: 8px;
    border: 1px solid #e2e8f0;
}

.push-status-info {
    flex: 1;
}

.push-status-text {
    font-size: 15px;
    font-weight: 600;
    margin-bottom: 4px;
    display: flex;
    align-items: center;
    color: #1e293b;
}

.push-status-detail {
    font-size: 13px;
    color: #64748b;
    line-height: 1.4;
}

.push-status-indicator {
    width: 8px;
    height: 8px;
    border-radius: 50%;
    margin-right: 8px;
}

.status-checking {
    background: #fbbf24;
    animation: pulse 2s infinite;
}

.status-success {
    background: #10b981;
}

.status-warning {
    background: #f59e0b;
}

.status-error {
    background: #ef4444;
}

@keyframes pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.5; }
}

.push-action-buttons {
    display: flex;
    gap: 8px;
    margin-left: 16px;
}

.btn-push {
    background: #3b82f6;
    color: white;
    border: none;
    padding: 8px 16px;
    border-radius: 6px;
    font-size: 13px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    gap: 6px;
    min-width: 100px;
    justify-content: center;
}

.btn-push:hover {
    background: #2563eb;
    transform: translateY(-1px);
}

.btn-push:active {
    transform: translateY(0);
}

.btn-push:disabled {
    background: #9ca3af;
    cursor: not-allowed;
    transform: none;
}

.btn-push.secondary {
    background: #6b7280;
}

.btn-push.secondary:hover {
    background: #4b5563;
}

.loading-spinner {
    width: 12px;
    height: 12px;
    border: 2px solid rgba(255, 255, 255, 0.3);
    border-top: 2px solid white;
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.setting-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
    gap: 16px;
    margin-bottom: 24px;
}

.setting-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px;
    background: #fff;
    border-radius: 8px;
    border: 1px solid #e2e8f0;
    transition: all 0.2s ease;
}

.setting-item:hover {
    border-color: #3b82f6;
    box-shadow: 0 2px 8px rgba(59, 130, 246, 0.1);
}

.setting-info h4 {
    margin: 0 0 4px 0;
    font-size: 15px;
    color: #1e293b;
    font-weight: 600;
}

.setting-info p {
    margin: 0;
    font-size: 13px;
    color: #64748b;
    line-height: 1.4;
}

.switch {
    position: relative;
    display: inline-block;
    width: 44px;
    height: 24px;
    flex-shrink: 0;
}

.switch input {
    opacity: 0;
    width: 0;
    height: 0;
}

.slider {
    position: absolute;
    cursor: pointer;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: #cbd5e1;
    transition: all 0.2s ease;
    border-radius: 24px;
}

.slider:before {
    position: absolute;
    content: "";
    height: 20px;
    width: 20px;
    left: 2px;
    bottom: 2px;
    background: white;
    transition: all 0.2s ease;
    border-radius: 50%;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.2);
}

input:checked + .slider {
    background: #3b82f6;
}

input:checked + .slider:before {
    transform: translateX(20px);
}

.device-grid {
    display: grid;
    grid-template-columns: 1fr;
    gap: 12px;
}

.device-item {
    padding: 20px;
    background: #f8fafc;
    border-radius: 12px;
    border: 1px solid #e2e8f0;
    transition: all 0.2s ease;
}

.device-item:hover {
    background: #f1f5f9;
    border-color: #cbd5e1;
}

.device-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.device-info {
    display: flex;
    align-items: center;
    flex: 1;
    min-width: 0;
}

.device-icon {
    font-size: 18px;
    margin-right: 12px;
    width: 24px;
    flex-shrink: 0;
}

.device-details {
    flex: 1;
    min-width: 0;
}

.device-details h5 {
    margin: 0 0 4px 0;
    font-size: 15px;
    color: #1e293b;
    font-weight: 600;
}

.device-details small {
    color: #64748b;
    font-size: 13px;
    display: block;
    line-height: 1.3;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.device-status {
    font-size: 12px;
    margin-top: 4px;
    font-weight: 500;
}

.status-active {
    color: #059669;
}

.status-inactive {
    color: #dc2626;
}

.device-controls {
    display: flex;
    gap: 12px;
    align-items: center;
}

.btn-test {
    background: #10b981;
    color: white;
    border: none;
    padding: 8px 16px;
    border-radius: 8px;
    font-size: 13px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
    min-width: 80px;
    box-shadow: 0 1px 2px rgba(0, 0, 0, 0.1);
}

.btn-test:hover {
    background: #059669;
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.btn-test:active {
    transform: translateY(0);
}

.btn-test:disabled {
    background: #9ca3af;
    cursor: not-allowed;
    transform: none;
    box-shadow: none;
}

.btn-test.sending {
    background: #f59e0b;
}

.btn-test.success {
    background: #10b981;
}

.btn-test.error {
    background: #ef4444;
}

.btn-save {
    background: #3b82f6;
    color: white;
    border: none;
    padding: 12px 32px;
    border-radius: 12px;
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.btn-save:hover {
    background: #2563eb;
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(0, 0, 0, 0.15);
}

.btn-save:active {
    transform: translateY(0);
}

.btn-save:disabled {
    background: #9ca3af;
    cursor: not-allowed;
    transform: none;
    box-shadow: none;
}

.empty-state {
    text-align: center;
    padding: 48px 24px;
    color: #64748b;
    background: #f8fafc;
    border-radius: 12px;
    border: 2px dashed #cbd5e1;
}

.empty-state .emoji {
    font-size: 48px;
    margin-bottom: 16px;
    display: block;
    opacity: 0.7;
}

.empty-state h3 {
    font-size: 18px;
    margin-bottom: 8px;
    color: #374151;
    font-weight: 600;
}

.empty-state p {
    font-size: 14px;
    line-height: 1.5;
}

.test-section {
    background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
    padding: 20px;
    border-radius: 12px;
    margin-bottom: 20px;
    border: 1px solid #fcd34d;
}

.test-section h3 {
    margin: 0 0 8px 0;
    font-size: 16px;
    color: #92400e;
    font-weight: 600;
}

.test-section p {
    margin: 0 0 16px 0;
    font-size: 14px;
    color: #92400e;
    line-height: 1.5;
}

.btn-test-all {
    background: #0ea5e9;
    color: white;
    border: none;
    padding: 10px 20px;
    border-radius: 10px;
    font-size: 14px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.btn-test-all:hover {
    background: #0284c7;
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(0, 0, 0, 0.15);
}

.btn-test-all:disabled {
    background: #9ca3af;
    cursor: not-allowed;
    transform: none;
    box-shadow: none;
}

.save-section {
    text-align: center;
    margin-top: 32px;
    padding-top: 24px;
    border-top: 1px solid #e2e8f0;
}

/* 반응형 처리 */
@media (max-width: 1200px) {
    .notification-container {
        padding: 30px 15px;
    }
    
    .setting-grid {
        grid-template-columns: 1fr;
    }
    
    .device-grid {
        grid-template-columns: 1fr;
        gap: 12px;
    }
    
    .push-status-container {
        flex-direction: column;
        gap: 16px;
        align-items: stretch;
    }
    
    .push-action-buttons {
        margin-left: 0;
        justify-content: center;
    }
}

@media (max-width: 768px) {
    .notification-container {
        padding: 20px 10px;
    }
    
    .page-header {
        margin-bottom: 30px;
        padding: 40px 20px;
    }
    
    .page-title {
        font-size: 30px;
    }
    
    .page-subtitle {
        font-size: 15px;
    }
    
    .section-title {
        font-size: 20px;
    }
    
    .setting-grid {
        grid-template-columns: 1fr;
        gap: 15px;
    }
    
    .device-grid {
        grid-template-columns: 1fr;
        gap: 12px;
    }
    
    .setting-item {
        flex-direction: column;
        align-items: flex-start;
        padding: 15px;
    }
    
    .switch {
        margin-top: 15px;
        align-self: flex-end;
    }
    
    .device-item {
        padding: 16px;
    }
    
    .device-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 16px;
    }
    
    .device-info {
        width: 100%;
        align-items: flex-start;
    }
    
    .device-controls {
        width: 100%;
        justify-content: flex-end;
        align-items: center;
        margin-top: 0;
    }
    
    .device-details small {
        white-space: normal;
        overflow: visible;
        text-overflow: unset;
    }
    
    .btn-save {
        padding: 12px 30px;
        font-size: 16px;
        width: 100%;
        max-width: 300px;
    }
    
    .push-subscription-section {
        padding: 24px;
    }
    
    .push-subscription-title {
        font-size: 20px;
    }
    
    .push-subscription-description {
        font-size: 14px;
    }
}

@media (max-width: 480px) {
    .notification-container {
        padding: 15px 8px;
    }
    
    .page-header {
        margin-bottom: 25px;
        padding: 35px 15px;
    }
    
    .page-title {
        font-size: 28px;
    }
    
    .page-subtitle {
        font-size: 14px;
    }
    
    .section-title {
        font-size: 18px;
        flex-direction: column;
        text-align: center;
    }
    
    .section-title .emoji {
        margin-right: 0;
        margin-bottom: 5px;
    }
    
    .setting-item, .device-item {
        padding: 12px;
    }
    
    .setting-info h4 {
        font-size: 16px;
    }
    
    .device-details h5 {
        font-size: 14px;
    }
    
    .device-controls {
        gap: 8px;
    }
    
    .btn-test {
        padding: 6px 12px;
        font-size: 12px;
        min-width: 70px;
    }
    
    .push-subscription-section {
        padding: 20px;
    }
    
    .push-subscription-icon {
        font-size: 28px;
    }
    
    .push-subscription-title {
        font-size: 18px;
    }
    
    .push-status-container {
        padding: 16px;
    }
    
    .push-action-buttons {
        flex-direction: column;
        gap: 8px;
    }
    
    .btn-push {
        width: 100%;
    }
}
</style>

<div class="notification-container">
    <div class="page-header">
        <div class="page-header-content">
            <h1 class="page-title">알림 설정</h1>
            <p class="page-subtitle">
                받고 싶은 푸시 알림을 선택하고 기기별로 설정을 관리할 수 있습니다.
            </p>
        </div>
    </div>
    
    <!-- 푸시 구독 섹션 -->
    <div class="push-subscription-section">
        <p class="push-subscription-description">
            푸시 알림을 받으려면 브라우저에서 알림 권한을 허용하고 구독해야 합니다.
        </p>
        
        <div class="push-status-container">
            <div class="push-status-info">
                <div class="push-status-text" id="push-status">
                    <div class="push-status-indicator status-checking" id="push-indicator"></div>
                    브라우저 지원 확인 중...
                </div>
                <div class="push-status-detail" id="push-status-detail">
                    잠시만 기다려주세요...
                </div>
            </div>
            
            <div class="push-action-buttons">
                <button type="button" class="btn-push" id="push-subscribe-btn" style="display: none;">
                    구독하기
                </button>
                <button type="button" class="btn-push secondary" id="push-unsubscribe-btn" style="display: none;">
                    구독 해제
                </button>
            </div>
        </div>
    </div>

    <form method="post" action="./notification_setting_update.php">
        <!-- 알림 종류 설정 -->
        <div class="section">
            <h2 class="section-title">
                알림 종류
            </h2>
            <div class="setting-grid">
                <div class="setting-item">
                    <div class="setting-info">
                        <h4>쪽지 알림</h4>
                        <p>새로운 쪽지가 도착했을 때 알림을 받습니다</p>
                    </div>
                    <label class="switch">
                        <input type="checkbox" name="message_notification" value="1" 
                               <?php echo $current_settings['message_notification'] ? 'checked' : ''; ?>>
                        <span class="slider"></span>
                    </label>
                </div>

                <div class="setting-item">
                    <div class="setting-info">
                        <h4>댓글 알림</h4>
                        <p>내 게시물에 새로운 댓글이 달렸을 때 알림을 받습니다</p>
                    </div>
                    <label class="switch">
                        <input type="checkbox" name="comment_notification" value="1"
                               <?php echo $current_settings['comment_notification'] ? 'checked' : ''; ?>>
                        <span class="slider"></span>
                    </label>
                </div>

                <div class="setting-item">
                    <div class="setting-info">
                        <h4>1:1 문의 답변</h4>
                        <p>1:1 문의에 대한 답변이 등록되었을 때 알림을 받습니다</p>
                    </div>
                    <label class="switch">
                        <input type="checkbox" name="inquiry_notification" value="1"
                               <?php echo $current_settings['inquiry_notification'] ? 'checked' : ''; ?>>
                        <span class="slider"></span>
                    </label>
                </div>
            </div>
        </div>

        <!-- 기기별 설정 -->
        <div class="section">
            <h2 class="section-title">
                등록된 기기 (<?php echo count($user_devices); ?>개)
            </h2>
            
            <?php if (empty($user_devices)): ?>
                <div class="empty-state">
                    <span class="emoji">📱</span>
                    <h3>등록된 기기가 없습니다</h3>
                    <p>위의 "알림 구독하기" 버튼을 클릭하여 푸시 알림을 구독하면 기기가 자동으로 등록됩니다.</p>
                </div>
            <?php else: ?>
                <div class="test-section">
                    <h3>테스트 알림</h3>
                    <p>등록된 기기에 테스트 푸시를 발송하여 알림이 정상적으로 작동하는지 확인할 수 있습니다.</p>
                    <button type="button" class="btn-test-all" onclick="sendTestToAll()">
                        모든 활성 기기에 테스트 발송
                    </button>
                </div>
                
                <div class="device-grid">
                    <?php foreach ($user_devices as $device): ?>
                        <div class="device-item">
                            <div class="device-header">
                                <div class="device-info">
                                    <span class="device-icon"><?php echo $device['device_icon'] ?? '🖥️'; ?></span>
                                    <div class="device-details">
                                        <h5><?php echo htmlspecialchars($device['device_name'] ?? 'Unknown Device'); ?></h5>
                                        <small><?php echo htmlspecialchars(substr($device['device_description'] ?? '', 0, 80)); ?></small>
                                        <small class="device-status <?php echo $device['is_active'] ? 'status-active' : 'status-inactive'; ?>">
                                            <?php echo $device['is_active'] ? '● 활성' : '● 비활성'; ?> | 
                                            등록일: <?php echo date('m/d H:i', strtotime($device['created_at'])); ?>
                                        </small>
                                    </div>
                                </div>
                                <div class="device-controls">
                                    <?php if ($device['is_active']): ?>
                                        <button type="button" class="btn-test" onclick="sendTestPush(<?php echo $device['id']; ?>, this)">
                                            테스트
                                        </button>
                                    <?php endif; ?>
                                    <label class="switch">
                                        <input type="checkbox" name="device_settings[<?php echo $device['id']; ?>]" value="1"
                                               <?php echo $device['is_active'] ? 'checked' : ''; ?>>
                                        <span class="slider"></span>
                                    </label>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>

        <!-- 저장 버튼 -->
        <div class="save-section">
            <!-- 확실한 전송을 위한 hidden input -->
            <input type="hidden" name="save_settings" value="1">
            <button type="submit" class="btn-save" id="saveBtn">
                설정 저장
            </button>
        </div>
    </form>
</div>

<script>
// 전역 변수
let notificationPushManager = null;
let statusCheckInterval = null;

// 페이지 로드 완료 후 초기화
document.addEventListener('DOMContentLoaded', function() {
    console.log('[알림 설정] 페이지 로드 완료');
    
    // 기본 폼 처리
    setupFormHandling();
    
    // 스위치 애니메이션
    setupSwitchAnimations();
    
    // 푸시 매니저 초기화 (지연 실행으로 pushmanager.js 로드 대기)
    setTimeout(initializePushManager, 100);
});

// 폼 처리 설정
function setupFormHandling() {
    const form = document.querySelector('form');
    const saveBtn = document.getElementById('saveBtn');
    
    if (form && saveBtn) {
        form.addEventListener('submit', function(e) {
            saveBtn.textContent = '저장 중...';
            saveBtn.disabled = true;
        });
    }
}

// 스위치 애니메이션 설정
function setupSwitchAnimations() {
    const switches = document.querySelectorAll('.switch input');
    switches.forEach(function(switchInput) {
        switchInput.addEventListener('change', function() {
            this.parentElement.style.transform = 'scale(0.95)';
            setTimeout(() => {
                this.parentElement.style.transform = 'scale(1)';
            }, 100);
        });
    });
}

// 푸시 매니저 초기화
function initializePushManager() {
    console.log('[알림 설정] 푸시 매니저 초기화 시작');
    
    // 즉시 기본 상태 체크
    checkBasicSupport();
    
    // 푸시 매니저 인스턴스 확인 (여러 방법으로 시도)
    let attempts = 0;
    const maxAttempts = 100; // 10초 대기
    
    const findPushManager = () => {
        attempts++;
        
        // 다양한 방법으로 푸시 매니저 찾기
        const possibleManagers = [
            window.gnuBoard5Push,
            window.GnuBoard5PushManager,
            window.pushManager
        ];
        
        let foundManager = null;
        
        // 인스턴스 확인
        for (let manager of possibleManagers) {
            if (manager && typeof manager === 'object' && manager.getStatus) {
                foundManager = manager;
                break;
            }
        }
        
        // 클래스 확인해서 인스턴스 생성
        if (!foundManager && window.GnuBoard5PushManager) {
            try {
                foundManager = new window.GnuBoard5PushManager({
                    debug: true,
                    autoInit: true,
                    showToast: true
                });
                console.log('[알림 설정] 새 푸시 매니저 인스턴스 생성');
            } catch (error) {
                console.error('[알림 설정] 푸시 매니저 생성 실패:', error);
            }
        }
        
        if (foundManager) {
            notificationPushManager = foundManager;
            console.log('[알림 설정] 푸시 매니저 발견:', foundManager);
            
            // 초기화 완료 대기
            if (foundManager.init && typeof foundManager.init === 'function') {
                foundManager.init().then(() => {
                    console.log('[알림 설정] 푸시 매니저 초기화 완료');
                    startStatusUpdates();
                    setupPushButtons();
                }).catch(error => {
                    console.error('[알림 설정] 푸시 매니저 초기화 실패:', error);
                    createFallbackPushManager();
                });
            } else {
                // 이미 초기화된 경우
                startStatusUpdates();
                setupPushButtons();
            }
            
        } else if (attempts < maxAttempts) {
            setTimeout(findPushManager, 100);
        } else {
            console.warn('[알림 설정] 푸시 매니저를 찾을 수 없음, 기본 구현 사용');
            createFallbackPushManager();
        }
    };
    
    findPushManager();
}

// 기본 지원 확인
function checkBasicSupport() {
    if (!('serviceWorker' in navigator) || !('PushManager' in window) || !('Notification' in window)) {
        updatePushStatus('❌ 이 브라우저는 푸시 알림을 지원하지 않습니다', 
                        '최신 버전의 Chrome, Firefox, Edge를 사용해주세요.', 'error');
        hideAllPushButtons();
        return false;
    }
    
    if (Notification.permission === 'denied') {
        updatePushStatus('🚫 푸시 알림 권한이 거부되었습니다', 
                        '브라우저 설정에서 알림 권한을 허용해주세요.', 'error');
        hideAllPushButtons();
        return false;
    }
    
    updatePushStatus('⏳ 푸시 구독 상태 확인 중...', 
                    '잠시만 기다려주세요...', 'checking');
    return true;
}

// 대체 푸시 매니저 생성
function createFallbackPushManager() {
    console.log('[알림 설정] 대체 푸시 매니저 생성');
    
    notificationPushManager = {
        isSupported: 'serviceWorker' in navigator && 'PushManager' in window && 'Notification' in window,
        subscription: null,
        
        async getStatus() {
            const isSupported = this.isSupported;
            const permission = isSupported ? Notification.permission : 'unsupported';
            
            if (isSupported && permission === 'granted') {
                try {
                    const registration = await navigator.serviceWorker.ready;
                    this.subscription = await registration.pushManager.getSubscription();
                } catch (error) {
                    console.error('[알림 설정] 구독 상태 확인 실패:', error);
                }
            }
            
            return {
                isSupported: isSupported,
                permission: permission,
                isSubscribed: !!this.subscription
            };
        },
        
        async subscribe() {
            try {
                // 권한 요청
                const permission = await Notification.requestPermission();
                if (permission !== 'granted') {
                    throw new Error('알림 권한이 거부되었습니다');
                }
                
                // Service Worker 등록
                const registration = await navigator.serviceWorker.register('./service-worker.js');
                await navigator.serviceWorker.ready;
                
                // 푸시 구독
                this.subscription = await registration.pushManager.subscribe({
                    userVisibleOnly: true
                });
                
                // 서버에 구독 정보 전송
                const subscriptionData = {
                    endpoint: this.subscription.endpoint,
                    keys: {
                        p256dh: arrayBufferToBase64(this.subscription.getKey('p256dh')),
                        auth: arrayBufferToBase64(this.subscription.getKey('auth'))
                    }
                };
                
                const response = await fetch('./pushmanager_ajax.php?action=subscribe', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ subscription: subscriptionData })
                });
                
                const result = await response.json();
                if (!result.success) {
                    throw new Error(result.message || '서버 등록 실패');
                }
                
                return this.subscription;
                
            } catch (error) {
                console.error('[알림 설정] 구독 실패:', error);
                throw error;
            }
        },
        
        async unsubscribe() {
            try {
                if (this.subscription) {
                    await this.subscription.unsubscribe();
                    
                    // 서버에 구독 해제 알림
                    await fetch('./pushmanager_ajax.php?action=unsubscribe', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({ endpoint: this.subscription.endpoint })
                    });
                    
                    this.subscription = null;
                }
                return true;
            } catch (error) {
                console.error('[알림 설정] 구독 해제 실패:', error);
                throw error;
            }
        }
    };
    
    startStatusUpdates();
    setupPushButtons();
}

// 상태 업데이트 시작
function startStatusUpdates() {
    // 즉시 업데이트
    updatePushUI();
    
    // 인터벌 설정 (5초마다)
    if (statusCheckInterval) {
        clearInterval(statusCheckInterval);
    }
    statusCheckInterval = setInterval(updatePushUI, 5000);
}

// 푸시 UI 업데이트
async function updatePushUI() {
    if (!notificationPushManager) return;
    
    try {
        const status = await notificationPushManager.getStatus();
        console.log('[알림 설정] 푸시 상태:', status);
        
        if (!status.isSupported) {
            updatePushStatus('❌ 이 브라우저는 푸시 알림을 지원하지 않습니다', 
                            '최신 버전의 Chrome, Firefox, Edge를 사용해주세요.', 'error');
            hideAllPushButtons();
            return;
        }
        
        if (status.permission === 'denied') {
            updatePushStatus('🚫 푸시 알림 권한이 거부되었습니다', 
                            '브라우저 설정에서 알림 권한을 허용해주세요.', 'error');
            hideAllPushButtons();
            return;
        }
        
        if (status.isSubscribed) {
            updatePushStatus('✅ 푸시 알림이 활성화되었습니다', 
                            '모든 알림을 정상적으로 받을 수 있습니다.', 'success');
            showPushButton('unsubscribe');
        } else {
            updatePushStatus('⚠️ 푸시 알림이 구독되지 않았습니다', 
                            '구독하기 버튼을 클릭하여 알림을 활성화하세요.', 'warning');
            showPushButton('subscribe');
        }
        
    } catch (error) {
        console.error('[알림 설정] 상태 업데이트 실패:', error);
        updatePushStatus('❌ 상태 확인 중 오류가 발생했습니다', 
                        '페이지를 새로고침해주세요.', 'error');
    }
}

// 푸시 상태 업데이트
function updatePushStatus(statusText, detailText, type) {
    const statusElement = document.getElementById('push-status');
    const detailElement = document.getElementById('push-status-detail');
    
    if (statusElement) {
        statusElement.innerHTML = `<div class="push-status-indicator status-${type}"></div>${statusText}`;
    }
    
    if (detailElement) {
        detailElement.textContent = detailText;
    }
}

// 푸시 버튼 표시/숨김
function showPushButton(type) {
    const subscribeBtn = document.getElementById('push-subscribe-btn');
    const unsubscribeBtn = document.getElementById('push-unsubscribe-btn');
    
    if (type === 'subscribe') {
        if (subscribeBtn) {
            subscribeBtn.style.display = 'flex';
            subscribeBtn.disabled = false;
        }
        if (unsubscribeBtn) {
            unsubscribeBtn.style.display = 'none';
        }
    } else if (type === 'unsubscribe') {
        if (subscribeBtn) {
            subscribeBtn.style.display = 'none';
        }
        if (unsubscribeBtn) {
            unsubscribeBtn.style.display = 'flex';
            unsubscribeBtn.disabled = false;
        }
    }
}

function hideAllPushButtons() {
    const subscribeBtn = document.getElementById('push-subscribe-btn');
    const unsubscribeBtn = document.getElementById('push-unsubscribe-btn');
    
    if (subscribeBtn) subscribeBtn.style.display = 'none';
    if (unsubscribeBtn) unsubscribeBtn.style.display = 'none';
}

// 푸시 버튼 이벤트 설정
function setupPushButtons() {
    const subscribeBtn = document.getElementById('push-subscribe-btn');
    const unsubscribeBtn = document.getElementById('push-unsubscribe-btn');
    
    // 구독 버튼
    if (subscribeBtn) {
        // 기존 이벤트 제거
        const newSubscribeBtn = subscribeBtn.cloneNode(true);
        subscribeBtn.parentNode.replaceChild(newSubscribeBtn, subscribeBtn);
        
        newSubscribeBtn.addEventListener('click', async function() {
            await handleSubscribe(this);
        });
    }
    
    // 구독 해제 버튼
    if (unsubscribeBtn) {
        // 기존 이벤트 제거
        const newUnsubscribeBtn = unsubscribeBtn.cloneNode(true);
        unsubscribeBtn.parentNode.replaceChild(newUnsubscribeBtn, unsubscribeBtn);
        
        newUnsubscribeBtn.addEventListener('click', async function() {
            if (confirm('정말 푸시 알림 구독을 해제하시겠습니까?')) {
                await handleUnsubscribe(this);
            }
        });
    }
}

// 구독 처리
async function handleSubscribe(button) {
    const originalText = button.innerHTML;
    
    try {
        button.innerHTML = '<div class="loading-spinner"></div> 구독 중...';
        button.disabled = true;
        
        await notificationPushManager.subscribe();
        
        button.innerHTML = '✅ 완료';
        showToast('푸시 알림 구독이 완료되었습니다!', 'success');
        
        // 1초 후 페이지 새로고침 (기기 목록 업데이트)
        setTimeout(() => {
            window.location.reload();
        }, 1000);
        
    } catch (error) {
        console.error('[알림 설정] 구독 실패:', error);
        
        button.innerHTML = '❌ 실패';
        
        let errorMessage = '푸시 알림 구독에 실패했습니다.';
        if (error.message.includes('denied') || error.message.includes('거부')) {
            errorMessage = '알림 권한이 거부되었습니다. 브라우저 설정에서 허용해주세요.';
        }
        
        showToast(errorMessage, 'error');
        
        setTimeout(() => {
            button.innerHTML = originalText;
            button.disabled = false;
        }, 3000);
    }
}

// 구독 해제 처리
async function handleUnsubscribe(button) {
    const originalText = button.innerHTML;
    
    try {
        button.innerHTML = '<div class="loading-spinner"></div> 해제 중...';
        button.disabled = true;
        
        await notificationPushManager.unsubscribe();
        
        button.innerHTML = '✅ 완료';
        showToast('푸시 알림 구독이 해제되었습니다.', 'info');
        
        // 1초 후 페이지 새로고침
        setTimeout(() => {
            window.location.reload();
        }, 1000);
        
    } catch (error) {
        console.error('[알림 설정] 구독 해제 실패:', error);
        
        button.innerHTML = '❌ 실패';
        showToast('구독 해제에 실패했습니다: ' + error.message, 'error');
        
        setTimeout(() => {
            button.innerHTML = originalText;
            button.disabled = false;
        }, 3000);
    }
}

// ArrayBuffer를 Base64로 변환
function arrayBufferToBase64(buffer) {
    const bytes = new Uint8Array(buffer);
    let binary = '';
    for (let i = 0; i < bytes.byteLength; i++) {
        binary += String.fromCharCode(bytes[i]);
    }
    return window.btoa(binary);
}

// 토스트 메시지 표시
function showToast(message, type = 'info') {
    // 기존 토스트 제거
    const existingToast = document.getElementById('custom-toast');
    if (existingToast) {
        existingToast.remove();
    }
    
    // 새 토스트 생성
    const toast = document.createElement('div');
    toast.id = 'custom-toast';
    toast.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 16px 24px;
        border-radius: 12px;
        color: white;
        font-weight: 600;
        z-index: 10000;
        max-width: 400px;
        word-wrap: break-word;
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.2);
        backdrop-filter: blur(10px);
        ${getToastStyles(type)}
        transform: translateX(100%);
        transition: transform 0.3s ease;
    `;
    
    toast.textContent = message;
    document.body.appendChild(toast);
    
    // 애니메이션으로 표시
    setTimeout(() => {
        toast.style.transform = 'translateX(0)';
    }, 100);
    
    // 5초 후 제거
    setTimeout(() => {
        toast.style.transform = 'translateX(100%)';
        setTimeout(() => {
            if (toast.parentNode) {
                toast.remove();
            }
        }, 300);
    }, 5000);
}

// 토스트 스타일 가져오기
function getToastStyles(type) {
    const styles = {
        success: 'background: linear-gradient(135deg, #10b981, #059669);',
        error: 'background: linear-gradient(135deg, #ef4444, #dc2626);',
        warning: 'background: linear-gradient(135deg, #f59e0b, #d97706);',
        info: 'background: linear-gradient(135deg, #3b82f6, #2563eb);'
    };
    
    return styles[type] || styles.info;
}

// 개별 기기 테스트 푸시 발송
function sendTestPush(deviceId, button) {
    if (button.disabled) return;
    
    const originalText = button.textContent;
    button.textContent = '발송 중...';
    button.disabled = true;
    button.classList.add('sending');
    
    fetch('./test_push.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'device_id=' + deviceId + '&action=test_single'
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            button.textContent = '✅ 발송됨';
            button.classList.remove('sending');
            button.classList.add('success');
            
            setTimeout(() => {
                button.textContent = originalText;
                button.disabled = false;
                button.classList.remove('success');
            }, 3000);
        } else {
            button.textContent = '❌ 실패';
            button.classList.remove('sending');
            button.classList.add('error');
            
            setTimeout(() => {
                button.textContent = originalText;
                button.disabled = false;
                button.classList.remove('error');
            }, 3000);
            
            showToast('테스트 푸시 발송에 실패했습니다: ' + (data.message || '알 수 없는 오류'), 'error');
        }
    })
    .catch(error => {
        button.textContent = '❌ 오류';
        button.classList.remove('sending');
        button.classList.add('error');
        
        setTimeout(() => {
            button.textContent = originalText;
            button.disabled = false;
            button.classList.remove('error');
        }, 3000);
        
        console.error('[알림 설정] 테스트 푸시 오류:', error);
        showToast('테스트 푸시 발송 중 오류가 발생했습니다.', 'error');
    });
}

// 모든 활성 기기에 테스트 푸시 발송
function sendTestToAll() {
    const button = document.querySelector('.btn-test-all');
    if (!button || button.disabled) return;
    
    const originalText = button.textContent;
    button.textContent = '📤 발송 중...';
    button.disabled = true;
    
    fetch('./test_push.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'action=test_all'
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            button.textContent = '✅ 모든 기기에 발송됨 (' + data.count + '개)';
            showToast(`${data.count}개 기기에 테스트 알림을 발송했습니다.`, 'success');
            
            setTimeout(() => {
                button.textContent = originalText;
                button.disabled = false;
            }, 5000);
        } else {
            button.textContent = '❌ 발송 실패';
            
            setTimeout(() => {
                button.textContent = originalText;
                button.disabled = false;
            }, 3000);
            
            showToast('테스트 푸시 발송에 실패했습니다: ' + (data.message || '알 수 없는 오류'), 'error');
        }
    })
    .catch(error => {
        button.textContent = '❌ 오류 발생';
        
        setTimeout(() => {
            button.textContent = originalText;
            button.disabled = false;
        }, 3000);
        
        console.error('[알림 설정] 전체 테스트 푸시 오류:', error);
        showToast('테스트 푸시 발송 중 오류가 발생했습니다.', 'error');
    });
}

// 페이지 언로드 시 정리
window.addEventListener('beforeunload', function() {
    if (statusCheckInterval) {
        clearInterval(statusCheckInterval);
    }
});

// 개발용 디버그 함수
window.debugPushManager = function() {
    console.log('Push Manager:', notificationPushManager);
    if (notificationPushManager && notificationPushManager.getStatus) {
        notificationPushManager.getStatus().then(status => {
            console.log('Push Status:', status);
        });
    }
};
</script>

<?php
include_once(G5_PATH.'/tail.php');
?>