<?php
if (!defined('_GNUBOARD_')) exit; // 개별 페이지 접근 불가

class G5_PUSH_HOOK_DEBUG {
    
    private $debug_mode = true; // 디버깅 모드
    private $log_file = '';
    
    public static function getInstance()
    {
        static $instance = null;
        if (null === $instance) {
            $instance = new self();
        }
        return $instance;
    }

    public function __construct() {
        $this->log_file = G5_DATA_PATH . '/push_debug.log';
        $this->debug_log("=== Push Hook 초기화 시작 ===");
        $this->add_hooks();
        $this->debug_log("=== Push Hook 초기화 완료 ===");
    }

    public function add_hooks(){
        $this->debug_log("훅 등록 시작");
        
        // 댓글 작성 후
        add_event('comment_update_after', array($this, 'comment_push'), 1, 5);
        $this->debug_log("comment_update_after 훅 등록됨");
        
        // 글 답변 작성 후  
        add_event('write_update_after', array($this, 'reply_push'), 1, 5);
        $this->debug_log("write_update_after 훅 등록됨");
        
        // 쪽지 보내기 후
        add_event('memo_form_update_after', array($this, 'memo_push'), 1, 3);
        $this->debug_log("memo_form_update_after 훅 등록됨");
        
        $this->debug_log("모든 훅 등록 완료");
    }

    private function debug_log($message) {
        if (!$this->debug_mode) return;
        
        $timestamp = date('Y-m-d H:i:s');
        $log_message = "[{$timestamp}] {$message}\n";
        
        // 파일에 로그 기록
        file_put_contents($this->log_file, $log_message, FILE_APPEND | LOCK_EX);
        
        // 화면에도 출력 (개발 환경에서만)
        if (defined('G5_DEBUG') && G5_DEBUG) {
            echo "<!-- PUSH DEBUG: {$message} -->\n";
        }
    }

    private function check_push_requirements() {
        $this->debug_log("푸시 요구사항 체크 시작");
        
        $requirements = array();
        
        // pushmanager 라이브러리 존재 확인
        $push_lib_path = G5_LIB_PATH.'/pushmanager.lib.php';
        $requirements['library_exists'] = file_exists($push_lib_path);
        $this->debug_log("라이브러리 파일 존재: " . ($requirements['library_exists'] ? 'YES' : 'NO') . " ({$push_lib_path})");
        
        // 통합 알림 함수 존재 확인
        $requirements['unified_function_exists'] = function_exists('send_notification');
        $this->debug_log("send_notification 통합 함수 존재: " . ($requirements['unified_function_exists'] ? 'YES' : 'NO'));
        
        // 기존 함수 존재 확인 (fallback용)
        $requirements['legacy_function_exists'] = function_exists('send_push_to_members');
        $this->debug_log("send_push_to_members (레거시) 함수 존재: " . ($requirements['legacy_function_exists'] ? 'YES' : 'NO'));
        
        // 설정 확인
        if (function_exists('get_pushmanager_config')) {
            $config = get_pushmanager_config();
            $requirements['config_exists'] = !empty($config);
            $requirements['api_key_exists'] = !empty($config['api_key']);
            $this->debug_log("푸시 설정 존재: " . ($requirements['config_exists'] ? 'YES' : 'NO'));
            $this->debug_log("API 키 존재: " . ($requirements['api_key_exists'] ? 'YES' : 'NO'));
        } else {
            $requirements['config_exists'] = false;
            $requirements['api_key_exists'] = false;
            $this->debug_log("get_pushmanager_config 함수가 존재하지 않음");
        }
        
        // 구독자 확인
        if (function_exists('get_subscription_stats')) {
            $stats = get_subscription_stats();
            $requirements['has_subscribers'] = $stats['active'] > 0;
            $this->debug_log("활성 구독자 수: " . $stats['active']);
        } else {
            $requirements['has_subscribers'] = false;
            $this->debug_log("get_subscription_stats 함수가 존재하지 않음");
        }
        
        return $requirements;
    }

    private function load_pushmanager_library() {
        $this->debug_log("pushmanager 라이브러리 로딩 시도");
        
        if (function_exists('send_notification')) {
            $this->debug_log("통합 알림 함수가 이미 존재함");
            return true;
        }
        
        $push_lib_path = G5_LIB_PATH.'/pushmanager.lib.php';
        
        if (file_exists($push_lib_path)) {
            include_once($push_lib_path);
            $this->debug_log("라이브러리 파일 로드됨: {$push_lib_path}");
            
            if (function_exists('send_notification')) {
                $this->debug_log("통합 알림 함수들 로드 성공");
                return true;
            } else {
                $this->debug_log("라이브러리 로드했지만 통합 함수가 존재하지 않음");
                return false;
            }
        } else {
            $this->debug_log("라이브러리 파일이 존재하지 않음: {$push_lib_path}");
            return false;
        }
    }

    /**
     * 쪽지 발송시 웹푸시 전송 (통합 함수 사용)
     */
    public function memo_push($member_list, $str_nick_list, $redirect_url, $memo=''){
        global $g5, $is_member, $member;

        $this->debug_log("=== 쪽지 푸시 시작 ===");
        $this->debug_log("is_member: " . ($is_member ? 'YES' : 'NO'));
        
        if( ! $is_member ) {
            $this->debug_log("비회원이므로 쪽지 푸시 종료");
            return;
        }

        $this->debug_log("발송자: " . $member['mb_id'] . " (" . $member['mb_nick'] . ")");

        // 라이브러리 로드
        if (!$this->load_pushmanager_library()) {
            $this->debug_log("pushmanager 라이브러리 로드 실패 - 쪽지 푸시 종료");
            return;
        }

        // 요구사항 체크
        $requirements = $this->check_push_requirements();

        if( ! $memo && isset($_POST['me_memo']) ){
            $memo = $_POST['me_memo'];
        }
        
        $this->debug_log("쪽지 내용: " . substr($memo, 0, 100) . "...");

        if( isset($member_list['id']) && $member_list['id'] ){
            $this->debug_log("수신자 수: " . count($member_list['id']));
            
            for($i=0;$i<count($member_list['id']);$i++){
                
                $mb_id = $member_list['id'][$i];
                $me_id = isset($member_list['me_id'][$i]) ? (int) $member_list['me_id'][$i] : 0;

                $this->debug_log("처리 중인 수신자: {$mb_id}, 쪽지 ID: {$me_id}");

                if( ($mb_id === $member['mb_id']) || ! $me_id ) {
                    $this->debug_log("자신에게 보내거나 쪽지 ID가 없음 - 건너뜀");
                    continue;
                }

                // 수신자 정보 확인
                $recipient = get_member($mb_id, '*', true);
                
                if( $recipient && $recipient['mb_id'] ){
                    $this->debug_log("수신자 정보 확인됨: " . $recipient['mb_nick']);
                    
                    $sender_name = $member['mb_nick'] ? $member['mb_nick'] : $member['mb_name'];
                    
                    // 푸시 메시지 구성
                    $push_title = "새 쪽지가 도착했습니다";
                    $push_message = "{$sender_name}님으로부터 쪽지가 도착했습니다.";
                    
                    // 메모 내용이 있으면 일부 포함 (최대 50자)
                    if ($memo) {
                        $memo_preview = cut_str(strip_tags($memo), 50);
                        $push_message = "{$sender_name}: {$memo_preview}";
                    }
                    
                    $this->debug_log("푸시 제목: {$push_title}");
                    $this->debug_log("푸시 메시지: {$push_message}");
                    
                    // 🎯 통합 알림 함수 사용
                    if (function_exists('send_notification')) {
                        $this->debug_log("통합 알림 함수로 쪽지 알림 발송 시작...");
                        
                        // 쪽지 전용 옵션
                        $options = array(
                            'click_action' => G5_URL."/".G5_BBS_DIR."/memo_view.php?me_id=".$me_id."&kind=recv",
                            'require_interaction' => true,
                            'tag' => 'memo_' . $me_id
                        );
                        
                        $result = send_notification(
                            'message',
                            array($mb_id), 
                            $push_title, 
                            $push_message,
                            $options
                        );
                        
                        $this->debug_log("통합 알림 함수 결과: " . print_r($result, true));
                        
                        // 결과 상세 로깅
                        if ($result['success']) {
                            $this->debug_log("쪽지 알림 발송 성공: {$mb_id}");
                            if (isset($result['details'])) {
                                $this->debug_log("발송 상세: " . print_r($result['details'], true));
                            }
                        } else {
                            $this->debug_log("쪽지 알림 발송 실패: " . $result['message']);
                            if (isset($result['details'])) {
                                $this->debug_log("실패 상세: " . print_r($result['details'], true));
                            }
                        }
                        
                    } else {
                        // 기존 함수 fallback
                        $this->debug_log("통합 함수가 없어서 기존 함수 사용");
                        
                        if (function_exists('send_push_to_members')) {
                            $push_options = array(
                                'icon' => get_push_icon_url(),
                                'badge' => get_push_badge_url(),
                                'click_action' => G5_URL."/".G5_BBS_DIR."/memo_view.php?me_id=".$me_id."&kind=recv",
                                'require_interaction' => true,
                                'silent' => false,
                                'tag' => 'memo_' . $me_id
                            );
                            
                            $result = send_push_to_members(
                                array($mb_id), 
                                $push_title, 
                                $push_message, 
                                $push_options
                            );
                            
                            $this->debug_log("기존 함수 사용 결과: " . print_r($result, true));
                        } else {
                            $this->debug_log("사용할 수 있는 푸시 함수가 없음");
                            $result = array('success' => false, 'message' => '푸시 함수 없음');
                        }
                    }
                    
                    // 로그 기록
                    if (function_exists('log_pushmanager_activity')) {
                        if ($result['success']) {
                            log_pushmanager_activity('memo', $push_title, "쪽지 푸시 발송 성공: {$mb_id}");
                        } else {
                            log_pushmanager_activity('error', $push_title, "쪽지 푸시 발송 실패: {$mb_id}", null, null, null, $result['message']);
                        }
                    }
                } else {
                    $this->debug_log("수신자 정보를 찾을 수 없음: {$mb_id}");
                }
            }
        } else {
            $this->debug_log("member_list['id']가 비어있거나 존재하지 않음");
        }
        
        $this->debug_log("=== 쪽지 푸시 종료 ===");
    }

    /**
     * 댓글 작성시 웹푸시 전송 (통합 함수 사용)
     */
    public function comment_push($board, $wr_id, $w, $qstr, $redirect_url){
        global $g5, $is_member, $member, $comment_id; 

        $this->debug_log("=== 댓글 푸시 시작 ===");
        $this->debug_log("board: " . $board['bo_table']);
        $this->debug_log("wr_id: {$wr_id}");
        $this->debug_log("w: {$w}");
        $this->debug_log("comment_id: {$comment_id}");

        if ( ! ($comment_id && $w === 'c') ) {
            $this->debug_log("댓글 작성이 아니므로 종료 (comment_id: {$comment_id}, w: {$w})");
            return;
        }

        // 라이브러리 로드
        if (!$this->load_pushmanager_library()) {
            $this->debug_log("pushmanager 라이브러리 로드 실패 - 댓글 푸시 종료");
            return;
        }

        $bo_table = $board['bo_table'];
        $write_table = $g5['write_prefix'] . $bo_table;
        
        $this->debug_log("게시판: {$bo_table}");
        $this->debug_log("테이블: {$write_table}");

        $wr = get_write($write_table, $wr_id, true);
        $this->debug_log("원글 작성자: " . $wr['mb_id']);

        $request_comment_id = (isset($_POST['comment_id']) && $_POST['comment_id'] ) ? (int) $_POST['comment_id'] : 0;
        $this->debug_log("부모 댓글 ID: {$request_comment_id}");

        $reply_array = $request_comment_id ? get_write($write_table, $request_comment_id, true) : array();
        $comment_wr = get_write($write_table, $comment_id, true);

        $commenter_name = '';
        if( $is_member ){
            $commenter_name = $board['bo_use_name'] ? $member['mb_name'] : $member['mb_nick'];
        } else {
            $commenter_name = $comment_wr['wr_name'];
        }
        
        $this->debug_log("댓글 작성자: {$commenter_name}");

        $push_recipients = array();

        // 댓글에 댓글을 단 경우
        if( isset($reply_array['wr_is_comment']) && $reply_array['wr_is_comment'] && $reply_array['mb_id'] ){
            $this->debug_log("댓글에 답글 - 부모 댓글 작성자: " . $reply_array['mb_id']);
            
            if( $reply_array['mb_id'] !== $member['mb_id'] ) {
                $this->debug_log("부모 댓글 작성자에게 푸시 발송 예정");
                $push_recipients[] = $reply_array['mb_id'];
            } else {
                $this->debug_log("자신의 댓글에 답글 - 푸시 안함");
            }
        }

        // 원글 작성자에게 알림
        if( $wr['mb_id'] && $wr['mb_id'] !== $member['mb_id'] ){
            if( !isset($reply_array['wr_is_comment']) || !$reply_array['wr_is_comment'] || $wr['mb_id'] !== $reply_array['mb_id'] ){
                $this->debug_log("원글 작성자에게 푸시 발송 예정");
                if( !in_array($wr['mb_id'], $push_recipients) ){
                    $push_recipients[] = $wr['mb_id'];
                }
            } else {
                $this->debug_log("원글 작성자와 부모 댓글 작성자가 같음 - 중복 방지");
            }
        } else {
            $this->debug_log("원글 작성자가 없거나 자신이 작성한 글");
        }

        $this->debug_log("푸시 수신자 목록: " . implode(', ', $push_recipients));

        // 푸시 발송
        if( !empty($push_recipients) ){
            
            $board_name = strip_tags($board['bo_subject']);
            $push_title = "새 댓글이 달렸습니다";
            $post_subject = cut_str(strip_tags($wr['wr_subject']), 30);
            $push_message = "{$commenter_name}님이 \"{$post_subject}\" 글에 댓글을 남겼습니다.";
            
            $this->debug_log("푸시 메시지: {$push_message}");
            
            foreach($push_recipients as $recipient_id){
                $this->debug_log("푸시 발송 중: {$recipient_id}");
                
                // 🎯 통합 알림 함수 사용
                if (function_exists('send_notification')) {
                    $this->debug_log("통합 알림 함수로 댓글 알림 발송...");
                    
                    // 댓글 전용 옵션
                    $options = array(
                        'click_action' => G5_URL."/".G5_BBS_DIR."/board.php?bo_table=".$bo_table."&wr_id=".$wr_id."#c_".$comment_id,
                        'tag' => 'comment_' . $bo_table . '_' . $wr_id
                    );
                    
                    $result = send_notification(
                        'comment',
                        array($recipient_id), 
                        $push_title, 
                        $push_message,
                        $options
                    );
                    
                    $this->debug_log("통합 알림 함수 결과: " . print_r($result, true));
                    
                    // 결과 상세 로깅
                    if ($result['success']) {
                        $this->debug_log("댓글 알림 발송 성공: {$recipient_id}");
                    } else {
                        $this->debug_log("댓글 알림 발송 실패 또는 설정으로 인한 건너뜀: " . $result['message']);
                        if (isset($result['details'])) {
                            $this->debug_log("상세 정보: " . print_r($result['details'], true));
                        }
                    }
                    
                } else {
                    // 기존 함수 fallback
                    $this->debug_log("통합 함수가 없어서 기존 함수 사용");
                    
                    if (function_exists('send_push_to_members')) {
                        $push_options = array(
                            'icon' => get_push_icon_url(),
                            'badge' => get_push_badge_url(),
                            'click_action' => G5_URL."/".G5_BBS_DIR."/board.php?bo_table=".$bo_table."&wr_id=".$wr_id."#c_".$comment_id,
                            'require_interaction' => false,
                            'silent' => false,
                            'tag' => 'comment_' . $bo_table . '_' . $wr_id
                        );
                        
                        $result = send_push_to_members(
                            array($recipient_id), 
                            $push_title, 
                            $push_message, 
                            $push_options
                        );
                        
                        $this->debug_log("기존 함수 사용 결과: " . print_r($result, true));
                    } else {
                        $this->debug_log("사용할 수 있는 푸시 함수가 없음");
                        $result = array('success' => false, 'message' => '푸시 함수 없음');
                    }
                }
                
                // 로그 기록
                if (function_exists('log_pushmanager_activity')) {
                    if ($result['success']) {
                        log_pushmanager_activity('comment', $push_title, "댓글 푸시 발송 성공: {$recipient_id} (게시판: {$board_name})");
                    } else {
                        log_pushmanager_activity('error', $push_title, "댓글 푸시 발송 실패: {$recipient_id}", null, null, null, $result['message']);
                    }
                }
            }
        } else {
            $this->debug_log("푸시 수신자가 없음");
        }
        
        $this->debug_log("=== 댓글 푸시 종료 ===");
    }

    /**
     * 글 답변 작성시 웹푸시 전송 (통합 함수 사용)
     */
    public function reply_push($board, $wr_id, $w, $qstr, $redirect_url){
        global $g5, $is_member, $member;

        $this->debug_log("=== 답글 푸시 시작 ===");
        $this->debug_log("w: {$w}");
        $this->debug_log("wr_id: {$wr_id}");
        $this->debug_log("POST wr_id: " . (isset($_POST['wr_id']) ? $_POST['wr_id'] : 'NOT_SET'));

        if( $w !== 'r' || !isset($_POST['wr_id']) || !$_POST['wr_id'] ) {
            $this->debug_log("답글 작성이 아니므로 종료");
            return;
        }

        // 라이브러리 로드
        if (!$this->load_pushmanager_library()) {
            $this->debug_log("pushmanager 라이브러리 로드 실패 - 답글 푸시 종료");
            return;
        }
        
        $original_wr_id = (int) $_POST['wr_id'];
        $this->debug_log("원글 ID: {$original_wr_id}");
        
        $wr = get_write(get_write_table_name($board['bo_table']), $original_wr_id, true);
        $this->debug_log("원글 작성자: " . $wr['mb_id']);
        
        if( $wr['mb_id'] && $member['mb_id'] !== $wr['mb_id'] ){
            $this->debug_log("원글 작성자에게 푸시 발송 진행");

            $recipient = get_member($wr['mb_id'], '*', true);
            if( $recipient && $recipient['mb_id'] ){
                
                $replier_name = '';
                if( $is_member ){
                    $replier_name = $board['bo_use_name'] ? $member['mb_name'] : $member['mb_nick'];
                } else {
                    $reply_wr = get_write(get_write_table_name($board['bo_table']), $wr_id, true);
                    $replier_name = $reply_wr['wr_name'];
                }

                $board_name = strip_tags($board['bo_subject']);
                $post_subject = cut_str(strip_tags($wr['wr_subject']), 30);
                
                $push_title = "글에 답변이 달렸습니다";
                $push_message = "{$replier_name}님이 \"{$post_subject}\" 글에 답변을 작성했습니다.";
                
                $this->debug_log("푸시 메시지: {$push_message}");
                
                // 🎯 통합 알림 함수 사용 (답글은 댓글 알림 설정을 따름)
                if (function_exists('send_notification')) {
                    $this->debug_log("통합 알림 함수로 답글 알림 발송...");
                    
                    // 답글 전용 옵션
                    $options = array(
                        'click_action' => G5_URL."/".G5_BBS_DIR."/board.php?bo_table=".$board['bo_table']."&wr_id=".$wr_id,
                        'tag' => 'reply_' . $board['bo_table'] . '_' . $original_wr_id
                    );
                    
                    $result = send_notification(
                        'comment', // 답글은 댓글 알림 설정을 따름
                        array($wr['mb_id']), 
                        $push_title, 
                        $push_message,
                        $options
                    );
                    
                    $this->debug_log("통합 알림 함수 결과: " . print_r($result, true));
                    
                    // 결과 상세 로깅
                    if ($result['success']) {
                        $this->debug_log("답글 알림 발송 성공: {$wr['mb_id']}");
                    } else {
                        $this->debug_log("답글 알림 발송 실패 또는 설정으로 인한 건너뜀: " . $result['message']);
                        if (isset($result['details'])) {
                            $this->debug_log("상세 정보: " . print_r($result['details'], true));
                        }
                    }
                    
                } else {
                    // 기존 함수 fallback
                    $this->debug_log("통합 함수가 없어서 기존 함수 사용");
                    
                    if (function_exists('send_push_to_members')) {
                        $push_options = array(
                            'icon' => get_push_icon_url(),
                            'badge' => get_push_badge_url(),
                            'click_action' => G5_URL."/".G5_BBS_DIR."/board.php?bo_table=".$board['bo_table']."&wr_id=".$wr_id,
                            'require_interaction' => false,
                            'silent' => false,
                            'tag' => 'reply_' . $board['bo_table'] . '_' . $original_wr_id
                        );
                        
                        $result = send_push_to_members(
                            array($wr['mb_id']), 
                            $push_title, 
                            $push_message, 
                            $push_options
                        );
                        
                        $this->debug_log("기존 함수 사용 결과: " . print_r($result, true));
                    } else {
                        $this->debug_log("사용할 수 있는 푸시 함수가 없음");
                        $result = array('success' => false, 'message' => '푸시 함수 없음');
                    }
                }
                
                // 로그 기록
                if (function_exists('log_pushmanager_activity')) {
                    if ($result['success']) {
                        log_pushmanager_activity('reply', $push_title, "답변 푸시 발송 성공: {$wr['mb_id']} (게시판: {$board_name})");
                    } else {
                        log_pushmanager_activity('error', $push_title, "답변 푸시 발송 실패: {$wr['mb_id']}", null, null, null, $result['message']);
                    }
                }
            } else {
                $this->debug_log("원글 작성자 정보를 찾을 수 없음");
            }
        } else {
            $this->debug_log("원글 작성자가 없거나 자신이 작성한 글");
        }
        
        $this->debug_log("=== 답글 푸시 종료 ===");
    }

    /**
     * 디버그 로그 파일 내용 출력
     */
    public function show_debug_log() {
        if (file_exists($this->log_file)) {
            $content = file_get_contents($this->log_file);
            echo "<pre style='background:#f0f0f0; padding:10px; border:1px solid #ccc; font-size:12px;'>";
            echo htmlspecialchars($content);
            echo "</pre>";
        } else {
            echo "디버그 로그 파일이 없습니다.";
        }
    }

    /**
     * 디버그 로그 파일 초기화
     */
    public function clear_debug_log() {
        if (file_exists($this->log_file)) {
            unlink($this->log_file);
        }
    }
}

// 싱글톤 인스턴스 생성
$g5_push_hook_debug = G5_PUSH_HOOK_DEBUG::getInstance();

// 디버그 페이지 접근용 함수들
function show_push_debug_log() {
    global $g5_push_hook_debug;
    $g5_push_hook_debug->show_debug_log();
}

function clear_push_debug_log() {
    global $g5_push_hook_debug;
    $g5_push_hook_debug->clear_debug_log();
}
?>