<?php
// test_push.php - 테스트 푸시 발송 처리
include_once('./_common.php');

// pushmanager.lib.php 포함
$pushmanager_lib = G5_LIB_PATH . '/pushmanager.lib.php';
if (file_exists($pushmanager_lib)) {
    include_once($pushmanager_lib);
} else {
    echo json_encode(['success' => false, 'message' => 'Push Manager 라이브러리를 찾을 수 없습니다.']);
    exit;
}

// 로그인 체크 (비회원도 허용 - 세션으로 구분)
$user = get_current_user_identifier();
if (!$user['id']) {
    echo json_encode(['success' => false, 'message' => '사용자 식별에 실패했습니다.']);
    exit;
}

// POST 요청만 허용
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => '잘못된 요청입니다.']);
    exit;
}

$action = $_POST['action'] ?? '';
$device_id = $_POST['device_id'] ?? '';

// 테스트 메시지 데이터
$test_messages = [
    [
        'title' => '🧪 테스트 알림',
        'body' => '푸시 알림이 정상적으로 작동하고 있습니다!',
        'icon' => get_push_icon_url(),
        'badge' => get_push_badge_url(),
        'tag' => 'test_' . time(),
        'require_interaction' => false
    ],
    [
        'title' => '✅ 연결 확인',
        'body' => '기기와 서버 간 푸시 연결이 활성화되어 있습니다.',
        'icon' => get_push_icon_url(),
        'badge' => get_push_badge_url(),
        'tag' => 'test_' . time(),
        'require_interaction' => false
    ],
    [
        'title' => '📱 기기 테스트',
        'body' => '이 메시지가 보인다면 푸시 알림 설정이 완료되었습니다.',
        'icon' => get_push_icon_url(),
        'badge' => get_push_badge_url(),
        'tag' => 'test_' . time(),
        'require_interaction' => false
    ],
    [
        'title' => '🎉 알림 성공!',
        'body' => date('H:i') . ' - 테스트 푸시가 성공적으로 전달되었습니다.',
        'icon' => get_push_icon_url(),
        'badge' => get_push_badge_url(),
        'tag' => 'test_' . time(),
        'require_interaction' => false
    ]
];

// 랜덤 테스트 메시지 선택
$test_message = $test_messages[array_rand($test_messages)];
$test_message['timestamp'] = date('Y-m-d H:i:s');

try {
    switch ($action) {
        case 'test_current_device':
            // 현재 브라우저에서 테스트 (IP와 User Agent로 구독 찾기)
            $current_ip = get_real_ip();
            $current_user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
            
            // 현재 브라우저의 구독 정보 조회 (최근 등록된 것 우선)
            $where_conditions = array();
            $where_conditions[] = "is_active = 1";
            $where_conditions[] = "ip_address = '" . addslashes($current_ip) . "'";
            
            if ($user['member_id']) {
                // 회원인 경우: 회원 ID로 조회
                $where_conditions[] = "member_id = '" . addslashes($user['member_id']) . "'";
            } else {
                // 비회원인 경우: IP와 User Agent로 추정
                $where_conditions[] = "member_id IS NULL";
                if ($current_user_agent) {
                    $where_conditions[] = "user_agent = '" . addslashes($current_user_agent) . "'";
                }
            }
            
            $subscription_sql = "SELECT id, endpoint, p256dh_key, auth_key, member_id, user_agent, created_at
                               FROM pushmanager_subscriptions 
                               WHERE " . implode(' AND ', $where_conditions) . "
                               ORDER BY created_at DESC 
                               LIMIT 1";
            
            $subscription_data = sql_fetch($subscription_sql);
            
            if (!$subscription_data) {
                throw new Exception('이 브라우저의 구독 정보를 찾을 수 없습니다. 먼저 푸시 알림을 구독해주세요.');
            }
            
            // 단일 구독으로 배열 구성
            $subscriptions = array(
                array(
                    'endpoint' => $subscription_data['endpoint'],
                    'keys' => array(
                        'p256dh' => $subscription_data['p256dh_key'],
                        'auth' => $subscription_data['auth_key']
                    ),
                    'member_id' => $subscription_data['member_id']
                )
            );
            
            // 푸시 발송
            $result = send_multiple_push_notifications($subscriptions, $test_message, array());
            
            if ($result['success']) {
                // 테스트 발송 로그 기록
                log_pushmanager_activity('test_current', $test_message['title'], $test_message['body'], 1, 1, 0, null);
                
                echo json_encode([
                    'success' => true,
                    'message' => '현재 브라우저로 테스트 푸시가 성공적으로 발송되었습니다.',
                    'subscription_id' => $subscription_data['id'],
                    'user_agent' => substr($subscription_data['user_agent'], 0, 50) . '...'
                ]);
            } else {
                throw new Exception('푸시 발송에 실패했습니다: ' . $result['message']);
            }
            break;
            
        case 'test_single':
            // 개별 기기 테스트
            if (empty($device_id) || !is_numeric($device_id)) {
                throw new Exception('올바르지 않은 기기 ID입니다.');
            }
            
            // 사용자의 기기 목록 조회
            $user_devices = get_user_devices($user['member_id'], $user['session_id']);
            $target_device = null;
            
            foreach ($user_devices as $device) {
                if ($device['id'] == $device_id) {
                    $target_device = $device;
                    break;
                }
            }
            
            if (!$target_device) {
                throw new Exception('기기를 찾을 수 없거나 권한이 없습니다.');
            }
            
            if (!$target_device['is_active']) {
                throw new Exception('비활성화된 기기입니다.');
            }
            
            // 해당 기기의 구독 정보 조회
            $subscription_sql = "SELECT endpoint, p256dh_key, auth_key, member_id 
                               FROM pushmanager_subscriptions 
                               WHERE id = " . intval($device_id) . " AND is_active = 1";
            $subscription_data = sql_fetch($subscription_sql);
            
            if (!$subscription_data) {
                throw new Exception('구독 정보를 찾을 수 없습니다.');
            }
            
            // 단일 구독으로 배열 구성
            $subscriptions = array(
                array(
                    'endpoint' => $subscription_data['endpoint'],
                    'keys' => array(
                        'p256dh' => $subscription_data['p256dh_key'],
                        'auth' => $subscription_data['auth_key']
                    ),
                    'member_id' => $subscription_data['member_id']
                )
            );
            
            // 푸시 발송
            $result = send_multiple_push_notifications($subscriptions, $test_message, array());
            
            if ($result['success']) {
                // 테스트 발송 로그 기록
                log_pushmanager_activity('test_single', $test_message['title'], $test_message['body'], 1, 1, 0, null);
                
                echo json_encode([
                    'success' => true,
                    'message' => '테스트 푸시가 성공적으로 발송되었습니다.',
                    'device_id' => $device_id,
                    'device_name' => $target_device['device_name']
                ]);
            } else {
                throw new Exception('푸시 발송에 실패했습니다: ' . $result['message']);
            }
            break;
            
        case 'test_all':
            // 모든 활성 기기에 테스트 발송
            $user_devices = get_user_devices($user['member_id'], $user['session_id']);
            $active_devices = array_filter($user_devices, function($device) {
                return $device['is_active'];
            });
            
            if (empty($active_devices)) {
                throw new Exception('활성화된 기기가 없습니다.');
            }
            
            // 활성 기기들의 구독 정보 조회
            $device_ids = array_column($active_devices, 'id');
            $device_ids_str = implode(',', array_map('intval', $device_ids));
            
            $subscriptions_sql = "SELECT endpoint, p256dh_key, auth_key, member_id 
                                FROM pushmanager_subscriptions 
                                WHERE id IN ({$device_ids_str}) AND is_active = 1";
            $subscriptions_result = sql_query($subscriptions_sql);
            
            $subscriptions = array();
            while ($row = sql_fetch_array($subscriptions_result)) {
                $subscriptions[] = array(
                    'endpoint' => $row['endpoint'],
                    'keys' => array(
                        'p256dh' => $row['p256dh_key'],
                        'auth' => $row['auth_key']
                    ),
                    'member_id' => $row['member_id']
                );
            }
            
            if (empty($subscriptions)) {
                throw new Exception('발송 가능한 구독 정보가 없습니다.');
            }
            
            // 모든 기기에 푸시 발송
            $result = send_multiple_push_notifications($subscriptions, $test_message, array());
            
            if ($result['success']) {
                $success_count = $result['data']['success_count'] ?? count($subscriptions);
                $error_count = $result['data']['error_count'] ?? 0;
                
                // 테스트 발송 로그 기록
                log_pushmanager_activity('test_all', $test_message['title'], $test_message['body'], 
                                       count($subscriptions), $success_count, $error_count, null);
                
                $message = "총 {$success_count}개 기기에 테스트 푸시가 발송되었습니다.";
                if ($error_count > 0) {
                    $message .= " (실패: {$error_count}개)";
                }
                
                echo json_encode([
                    'success' => true,
                    'message' => $message,
                    'count' => $success_count,
                    'total' => count($subscriptions),
                    'failed' => $error_count
                ]);
            } else {
                throw new Exception('테스트 푸시 발송에 실패했습니다: ' . $result['message']);
            }
            break;
            
        default:
            throw new Exception('올바르지 않은 액션입니다.');
    }
    
} catch (Exception $e) {
    // 에러 로그 기록
    log_pushmanager_activity('test_error', $test_message['title'], $e->getMessage(), 0, 0, 1, $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>