<?php
/**
 * Copyright (c) 2012 Jacob Lee <letsgolee@naver.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA or check at http://www.gnu.org/licenses/gpl.html
 */

/**
 * Advanced Encryption Standard (AES)
 *
 * http://en.wikipedia.org/wiki/Advanced_Encryption_Standard
 *
 * Example)
 *
 * $aes = new AES();
 * $plaintext = 'This is the text';
 * $ciphertext = $aes->encrypt($plaintext, 'password', array('bits'=> 256, // default value
 *                                                          'utf8'=> true // default value
 * ));
 * if ($ciphertext === false) {
 *     echo "Error: ".$aes->displayMsg();
 * }
 * else {
 *      $text = $aes->decrypt($ciphertext, 'password');
 *      if ($text === false) echo "Error: ".$aes->displayMsg();
 *      else {
 *	        if ($text == $plaintext) echo 'OK';
 *		    else echo 'Something's Wrong in the code!;
 *	   }
 * }
 */

define('AES_BLOCK_SIZE', 16);

class AES {
	/**
	 * AES Mode: Cipher-block chaining (CBC)
	 */
	var $Mode = 'CBC';

	/**
	 * Messages that are appended while encrypting & decrypting
	 */
	var $Messages = array();

	/**
	 * S-Box. Predefined Substition Box.
	 * The S-box is generated by determining the multiplicative inverse for a given number in GF(28) = GF(2)[x]/(x8 + x4 + x3 + x + 1),
	 * Rijndael's finite field (zero, which has no inverse, is set to zero).
	 *
	 * http://en.wikipedia.org/wiki/Rijndael_S-box
	 *
	 *     | 0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f
	 *	---|--|--|--|--|--|--|--|--|--|--|--|--|--|--|--|--|
	 *	00 |63 7c 77 7b f2 6b 6f c5 30 01 67 2b fe d7 ab 76 
	 *	10 |ca 82 c9 7d fa 59 47 f0 ad d4 a2 af 9c a4 72 c0 
	 *	20 |b7 fd 93 26 36 3f f7 cc 34 a5 e5 f1 71 d8 31 15 
	 *	30 |04 c7 23 c3 18 96 05 9a 07 12 80 e2 eb 27 b2 75 
	 *	40 |09 83 2c 1a 1b 6e 5a a0 52 3b d6 b3 29 e3 2f 84 
	 *	50 |53 d1 00 ed 20 fc b1 5b 6a cb be 39 4a 4c 58 cf 
	 *	60 |d0 ef aa fb 43 4d 33 85 45 f9 02 7f 50 3c 9f a8 
	 *	70 |51 a3 40 8f 92 9d 38 f5 bc b6 da 21 10 ff f3 d2 
	 *	80 |cd 0c 13 ec 5f 97 44 17 c4 a7 7e 3d 64 5d 19 73 
	 *	90 |60 81 4f dc 22 2a 90 88 46 ee b8 14 de 5e 0b db 
	 *	a0 |e0 32 3a 0a 49 06 24 5c c2 d3 ac 62 91 95 e4 79 
	 *	b0 |e7 c8 37 6d 8d d5 4e a9 6c 56 f4 ea 65 7a ae 08 
	 *	c0 |ba 78 25 2e 1c a6 b4 c6 e8 dd 74 1f 4b bd 8b 8a 
	 *	d0 |70 3e b5 66 48 03 f6 0e 61 35 57 b9 86 c1 1d 9e 
	 *	e0 |e1 f8 98 11 69 d9 8e 94 9b 1e 87 e9 ce 55 28 df 
	 *	f0 |8c a1 89 0d bf e6 42 68 41 99 2d 0f b0 54 bb 16 
	 */
/*
	var $SBOX = array(
		99,  124, 119, 123, 242, 107, 111, 197, 48,  1,   103, 43,  254, 215, 171,
		118, 202, 130, 201, 125, 250, 89,  71,  240, 173, 212, 162, 175, 156, 164,
		114, 192, 183, 253, 147, 38,  54,  63,  247, 204, 52,  165, 229, 241, 113,
		216, 49,  21,  4,   199, 35,  195, 24,  150, 5,   154, 7,   18,  128, 226,
		235, 39,  178, 117, 9,   131, 44,  26,  27,  110, 90,  160, 82,  59,  214,
		179, 41,  227, 47,  132, 83,  209, 0,   237, 32,  252, 177, 91,  106, 203,
		190, 57,  74,  76,  88,  207, 208, 239, 170, 251, 67,  77,  51,  133, 69,
		249, 2,   127, 80,  60,  159, 168, 81,  163, 64,  143, 146, 157, 56,  245,
		188, 182, 218, 33,  16,  255, 243, 210, 205, 12,  19,  236, 95,  151, 68,
		23,  196, 167, 126, 61,  100, 93,  25,  115, 96,  129, 79,  220, 34,  42,
		144, 136, 70,  238, 184, 20,  222, 94,  11,  219, 224, 50,  58,  10,  73,
		6,   36,  92,  194, 211, 172, 98,  145, 149, 228, 121, 231, 200, 55,  109,
		141, 213, 78,  169, 108, 86,  244, 234, 101, 122, 174, 8,   186, 120, 37,
		46,  28,  166, 180, 198, 232, 221, 116, 31,  75,  189, 139, 138, 112, 62,
		181, 102, 72,  3,   246, 14,  97,  53,  87,  185, 134, 193, 29,  158, 225,
		248, 152, 17,  105, 217, 142, 148, 155, 30,  135, 233, 206, 85,  40,  223,
		140, 161, 137, 13,  191, 230, 66,  104, 65,  153, 45,  15,  176, 84,  187,
		22
	);
*/
	var $SBOX = array(
		0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5,
		0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76,
		0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0,
		0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0,
		0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc,
		0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15,
		0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a,
		0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75,
		0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0,
		0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84,
		0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b,
		0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf,
		0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85,
		0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8,
		0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5,
		0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2,
		0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17,
		0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73,
		0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88,
		0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb,
		0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c,
		0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79,
		0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9,
		0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08,
		0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6,
		0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a,
		0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e,
		0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e,
		0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94,
		0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf,
		0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68,
		0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16
	);

	var $SBOX_INVERSE = null;

	//var $SHIFTBY = array(0,5,10,15,4,9,14,3,8,13,2,7,12,1,6,11);
	var $SHIFTBY = array(0x0,0x5,0xa,0xf,0x4,0x9,0xe,0x3,0x8,0xd,0x2,0x7,0xc,0x1,0x6,0xb);

	var $SHIFTBY_INVERSE = null;

	//var $XTIME = null;

	/* Default to 256 Bit Encryption */
	/* refer setKeySize() */
	var $Nr = 14; // number of round
	var $Nk = 8;  // key size


	function AES() {
		$this->SBOX_INVERSE = $this->invertArray($this->SBOX);
		$this->SHIFTBY_INVERSE = $this->invertArray($this->SHIFTBY);
		//$this->XTIME = $this->generateXTIME();
	}

/*
	function generateXTIME() {
		$ret = array();
		for($i = 0; $i < 128; $i++) {
			$ret[$i] = $i << 1;
			$ret[128+$i] = ($i << 1) ^ 0x1b;
		}
		return $ret;
	}
*/
	function xtime($x) {
		return $x < 0x80 ? $x << 1 : (($x - 0x80) << 1) ^ 0x1b;
	}

	function invertArray($array) {
		$ret = array();
		for ($i=0, $cnt=count($array); $i < $cnt; $i++)
			$ret[$array[$i]] = $i;
		return $ret;
	}

	function randomBytesArray($len=8, $start=0, $end=255) {
		if(!$len) $len = 8;
		$bytes = array();
/*
		$field = array();
		for($i = 0; $i < 256; $i++) $field[$i] = $i;
		for($i = 0; $i < $len; $i++)
			$bytes[$i] = $field[rand(0,255)];
*/
		for ($i = 0; $i < $len; $i++)
			$bytes[$i] = mt_rand($start, $end);
		return $bytes;
	}

	function generateSalt() {
		return $this->ByteArray_toString($this->randomBytesArray(8));
	}

	function randomAESKey($len=8) {
		if(!$len) $len = 8;
		// range: 32-126 refer to ASCII code table
		return $this->ByteArray_toString($this->randomBytesArray($len, 32, 126));
	}



	/**
	 * Converts a string into a hexadecimal string
	 * returns the characters of a string to their hexadecimal charcode equivalent
	 * Works only on byte chars with charcode < 256. All others chars are converted
	 * into "xx"
	 *
	 * @return hex string e.g. "hello world" => "68656c6c6f20776f726c64"
	 */
	function Hex_fromString($str) {
		if(!$str) $str = "";
		$hs ='';
		$hv ='';
		for ($i=0, $cnt=strlen($str); $i < $cnt; $i++) {
			$hv = dechex(ord($str[$i]));
			$hs .= (strlen($hv) == 1) ? '0'.$hv : $hv;
		}
		return $hs;
	}

	/**
	 * Converts a hex string into a string
	 * returns the characters of a hex string to their char of charcode
	 *
	 * @return hex string e.g. "68656c6c6f20776f726c64" => "hello world"
	 */
	function Hex_toString($str) {
		if(!$str) $str = "";
		$s = "";
		for($i=0, $cnt=strlen($str); $i < $cnt; $i+=2){
			$s .= chr(hexdec(substr($str, $i,2)));
		}
		return $s;
	}

	/**
	 * Converts a string into an array of char code bytes
	 * returns the characters of a hex string to their char of charcode
	 *
	 * @return hex string e.g. "68656c6c6f20776f726c64" => "hello world"
	 */
	function ByteArray_fromString($s) {
		if(!$s) $s = "";
		$b = array();
		for($i=0, $cnt=strlen($s); $i < $cnt; $i++){
			 $b[$i] = ord($s[$i]);
		}
		return $b;
	}


	function ByteArray_toString($b) {
		$s = '';
		for($i=0, $cnt=count($b); $i < $cnt; $i++){
			 $s .= chr($b[$i]);
		}
		return $s;
	}

	function appendMsg($msg) {
		$this->messages[] = $msg;
	}

	function displayMsg($index = null) {
		$cnt = count($this->Messages);

		if (is_null($index)) $index = $cnt - 1;
		return ($cnt && $index < $cnt) ? $this->Messages[$index] : '';
	}

	function setKeySize($size = 256) {
		switch ($size) {
			case 128:
				$this->Nr = 10;
				$this->Nk = 4;
				break;
			case 192:
				$this->Nr = 12;
				$this->Nk = 6;
				break;
			case 256:
				$this->Nr = 14;
				$this->Nk = 8;
				break;
			default:
				$this->appendMsg('AES.setKeySize encounted an error! Invalid Key Size Specified:' + $size);
				return false;
		}
		return true;
	}



	function encrypt($plaintext, $password, $options = array()) {
		if (!is_array($options)) $options = array();
		$options['bits'] = isset($options['bits']) ? $options['bits'] : 256;
		$options['utf8'] = isset($options['utf8']) ? $options['utf8'] : true;

		if (empty($plaintext) || !is_string($plaintext) || !strlen($plaintext)) {
			$this->appendMsg('The plaintext is blank!');
			return '';
		}

		$salt = $this->generateSalt();

		if (!$this->setKeySize($options['bits'])) return false;

		/* Creates Password Based Encryption KEY and IV(initial vector) */
		$pbe = $this->_createPBEKey($password, $salt);
		if (!$pbe) return false;

		if ($options['utf8'])
			$plaintext = utf8_encode($plaintext);

		$ciphertext = $this->rawEncrypt($plaintext, $pbe['key'], $pbe['iv']);

		return ($ciphertext === false) ? false : base64_encode('Salted__' . $salt . $ciphertext);
	}


	function rawEncrypt($plaintext, $key, $iv) {
		$byteArray = $this->_PKCS5pad($this->ByteArray_fromString($plaintext));
		
		$keySchedule = $this->_createKeySchedule($key);

		$nBlocks = ceil(count($byteArray) / AES_BLOCK_SIZE);

		$cipherText = '';
		$plainBlock = array();
		$state = $this->ByteArray_fromString($iv);

		for ($b = 0; $b < $nBlocks; $b++) {
			/* XOR last block and next data block, then encrypt that */
			$plainBlock = array_slice($byteArray, $b * AES_BLOCK_SIZE, AES_BLOCK_SIZE);
			$state = $this->_xorBlock($state, $plainBlock);
			$state = $this->_encryptBlock($state, $keySchedule); /* encrypt block */
			$cipherText .= $this->ByteArray_toString($state);
		}

		return $cipherText;
	}



	function decrypt($ciphertext, $password, $options=array()) {
		if (!is_array($options)) $options = array();
		$options['bits'] = isset($options['bits']) ? $options['bits'] : 256;
		$options['utf8'] = isset($options['utf8']) ? $options['utf8'] : true;

		if (empty($ciphertext) || !is_string($ciphertext) || !strlen($ciphertext)) {
			$this->appendMsg('The ciphertext is blank!');
			return '';
		}

		$ciphertext = base64_decode($ciphertext);

		if (substr($ciphertext, 0,8) != 'Salted__') {
			$this->appendMsg('The given text is not a AES ciphertext!');
			return $ciphertext;
		}

		$salt = substr($ciphertext, 8, 8); // extract salt from crypted text

		if (!$this->setKeySize($options['bits'])) return false;

		$pbe = $this->_createPBEKey($password, $salt);
		if (!$pbe) return false;

		$plaintext = $this->rawDecrypt(substr($ciphertext, 16) /* remove salt */, $pbe['key'], $pbe['iv']);

		if ($plaintext === false) return false;

		if ($options['utf8'])
			$plaintext = utf8_decode($plaintext);

		return $plaintext;
	}

	function rawDecrypt($ciphertext, $key, $iv) {
		$byteArray = $this->ByteArray_fromString($ciphertext);

		if (strlen($key) != $this->Nk * 4 || strlen($iv) < AES_BLOCK_SIZE) {
			$this->appendMsg('AES.decrypt encountered an error! The length of key or iv may not be correct!');
			return false;
		}		
		if (count($byteArray) % AES_BLOCK_SIZE != 0) {
			$this->appendMsg('AES.decrypt encountered an error! The length of ciphertext is not correct!');
			return false;
		}

		$keySchedule = $this->_createKeySchedule($key);

		/* separate byteArray into block */
		$nBlocks = ceil(count($byteArray) / AES_BLOCK_SIZE);
		
		$plainText = '';
		$state = $this->ByteArray_fromString($iv);
		$cipherBlock = array();
		$dec_state = array();
		for ($b = 0; $b < $nBlocks; $b++){
			$cipherBlock = array_slice($byteArray, $b * AES_BLOCK_SIZE, AES_BLOCK_SIZE);

			$dec_state = $this->_decryptBlock($cipherBlock, $keySchedule); /* decrypt ciphertext block */

			$plainText .= $this->ByteArray_toString(($b == $nBlocks-1) ? $this->_PKCS5unpad($this->_xorBlock($state, $dec_state)) : $this->_xorBlock($state, $dec_state));
/*
			$block = $this->_xorBlock($state, $dec_state);
			if ($b == $nBlocks-1) $block = $this->_PKCS5unpad($block);
			$plainText .= $this->ByteArray_toString($block);
*/
			$state = $cipherBlock; /* save old ciphertext for next round */
		}

		return $plainText;
	}

	function _PKCS5pad($byteArray) {
		/* PKCS#5 padding */
		$padding = AES_BLOCK_SIZE - (count($byteArray) % AES_BLOCK_SIZE);

		for($c = 0; $c < $padding; $c++)
			$byteArray[] = $padding;

		return $byteArray;
	}

	function _PKCS5unpad($block) {
		/* remove PKCS#5 Padding */
		$padding = $block[count($block)-1];

		if ($padding <= 16 && $padding == $block[count($block)-$padding])
			$block = array_slice($block, 0, count($block)-$padding);
		return $block;
	}

	function _createKeySchedule($key) {
		$keyBytes = Array();

		for($i = 0; $i < $this->Nk * 4; $i++) {
			$keyBytes[$i] = is_NaN(ord($key[$i])) ? 0 : ord($key[$i]);
		}
		/* generate key schedule */
		return $this->_expandKey($keyBytes);
	}


	function _expandKey($key) {
		$kl = count($key);
		$Rcon = 1;

		for($i = $kl; $i < 16 * ($this->Nr + 1); $i += 4) {
			$word = array_slice($key, $i-4, $i); /* 4-byte word */
			if ($i % $kl == 0) {
				$word = $this->_subWord($this->_rotWord($word));
				$word[0] ^= $Rcon;

				if (($Rcon <<= 1) >= 256)
					$Rcon ^= 0x11b;  // array(0x1,0x2,0x4,0x8,0x10,0x20,0x40,0x80)
			}
			else if (($kl > 24) && ($i % $kl == 16)) {
				$word = $this->_subWord($word);
			}
			for($j = 0; $j < 4; $j++)
				$key[$i + $j] = $key[$i + $j - $kl] ^ $word[$j];
		}
		return $key;
	}


	function _subWord($w) {
		/* apply SBOX to 4-byte word w */
		for ($i = 0; $i < 4; $i++) {
			$w[$i] = $this->SBOX[$w[$i]];
		}
		return $w;
	}

	function _rotWord($w) {
		/* rotate 4-byte word w left by one byte */
		$tmp = $w[0];
		for ($i = 0; $i < 4; $i++) {
			$w[$i] = ($i==3) ? $tmp : $w[$i+1];
		}
		return $w;
	}

	function _xorBlock($a1, $a2) {
		/* xor the elements of two arrays together */
		 $res = array();
		 for($i=0, $cnt=count($a1); $i < $cnt; $i++)
				$res[$i] = $a1[$i] ^ $a2[$i];
		 return $res;
	}

	function _subBytes($state, $inverse=false) {
		$box = (!$inverse) ? $this->SBOX : $this->SBOX_INVERSE;

		for($i = 0; $i < 16; $i++)
			$state[$i] = $box[$state[$i]];
		return $state;
	}

	function _addRoundKey($state, $rkey) {
		for($i = 0; $i < 16; $i++)
			$state[$i] ^= $rkey[$i];
		return $state;
	}

	function _shiftRows($state, $inverse=false) {
		$shifttab = (!$inverse) ? $this->SHIFTBY : $this->SHIFTBY_INVERSE;
		$h = $state;
		for($i = 0; $i < 16; $i++)
			$state[$i] = $h[$shifttab[$i]];
		return $state;
	}

	function _mixColumns($state, $inverse=false) {
		for($i = 0; $i < 16; $i += 4) {
			$s0 = $state[$i + 0];
			$s1 = $state[$i + 1];
			$s2 = $state[$i + 2];
			$s3 = $state[$i + 3];
			$h = $s0 ^ $s1 ^ $s2 ^ $s3;
/*
			if (!$inverse) {
				$state[$i + 0] ^= $h ^ $this->XTIME[$s0 ^ $s1];
				$state[$i + 1] ^= $h ^ $this->XTIME[$s1 ^ $s2];
				$state[$i + 2] ^= $h ^ $this->XTIME[$s2 ^ $s3];
				$state[$i + 3] ^= $h ^ $this->XTIME[$s3 ^ $s0];
			} else {
				$xh = $this->XTIME[$h];
				$h1 = $this->XTIME[$this->XTIME[$xh ^ $s0 ^ $s2]] ^ $h;
				$h2 = $this->XTIME[$this->XTIME[$xh ^ $s1 ^ $s3]] ^ $h;
				$state[$i + 0] ^= $h1 ^ $this->XTIME[$s0 ^ $s1];
				$state[$i + 1] ^= $h2 ^ $this->XTIME[$s1 ^ $s2];
				$state[$i + 2] ^= $h1 ^ $this->XTIME[$s2 ^ $s3];
				$state[$i + 3] ^= $h2 ^ $this->XTIME[$s3 ^ $s0];
			}			
*/
			if (!$inverse) {
				$state[$i + 0] ^= $h ^ $this->xtime($s0 ^ $s1);
				$state[$i + 1] ^= $h ^ $this->xtime($s1 ^ $s2);
				$state[$i + 2] ^= $h ^ $this->xtime($s2 ^ $s3);
				$state[$i + 3] ^= $h ^ $this->xtime($s3 ^ $s0);
			} else {
				$xh = $this->xtime($h);
				$h1 = $this->xtime($this->xtime($xh ^ $s0 ^ $s2)) ^ $h;
				$h2 = $this->xtime($this->xtime($xh ^ $s1 ^ $s3)) ^ $h;
				$state[$i + 0] ^= $h1 ^ $this->xtime($s0 ^ $s1);
				$state[$i + 1] ^= $h2 ^ $this->xtime($s1 ^ $s2);
				$state[$i + 2] ^= $h1 ^ $this->xtime($s2 ^ $s3);
				$state[$i + 3] ^= $h2 ^ $this->xtime($s3 ^ $s0);
			}			

		}
		return $state;
	}


	function _encryptBlock($block, $key) {
		$len = count($key);
		$block = $this->_addRoundKey($block, array_slice($key, 0, 16));

		for($round = 1; $round < $this->Nr; $round++) {
			$block = $this->_subBytes($block);
			$block = $this->_shiftRows($block);
			$block = $this->_mixColumns($block);
			$block = $this->_addRoundKey($block, array_slice($key, $round * 16, ($round * 16) + 16));
		}
		$block = $this->_subBytes($block);
		$block = $this->_shiftRows($block);
		$block = $this->_addRoundKey($block, array_slice($key, $this->Nr * 16, $len));

/*
		for($round = 1; $round <= $this->Nr; $round++) {
			$block = $this->_subBytes($block);
			$block = $this->_shiftRows($block);
			if ($round < $this->Nr) { // last round? don't mixColumns
				$block = $this->_mixColumns($block);
			}
			$block = $this->_addRoundKey($block, array_slice($key, $round * 16, ($round * 16) + 16));
		}
*/
		return $block;
	}


	function _decryptBlock($block, $key) {
		$len = count($key);

		$block = $this->_addRoundKey($block, array_slice($key, $len - 16, $len));
		$block = $this->_shiftRows($block, 1); // inverse operation
		$block = $this->_subBytes($block, 1); // inverse operation
		for($round = $this->Nr-1; $round >= 1; $round--) {
			$block = $this->_addRoundKey($block, array_slice($key, $round * 16, ($round * 16) + 16));
			$block = $this->_mixColumns($block, 1); // inverse operation
			$block = $this->_shiftRows($block, 1); // inverse operation
			$block = $this->_subBytes($block, 1); // inverse operation
		}
		$block = $this->_addRoundKey($block, array_slice($key, 0, 16));
/*
		for($round = $this->Nr; $round >= 1; $round--) {
			$block = $this->_addRoundKey($block, array_slice($key, $round * 16, ($round * 16) + 16));
			if ($round < $this->Nr) {
				$block = $this->_mixColumns($block, 1); // inverse operation
			}
			$block = $this->_shiftRows($block, 1); // inverse operation
			$block = $this->_subBytes($block, 1); // inverse operation
		}
		$block = $this->_addRoundKey($block, array_slice($key, 0, 16));
*/
		return $block;
	}


	function _createPBEKey($password, $salt) {
		/* creates Password Based Encryption key and initial vector */
		$round = 3;
		$data00 = $password . $salt;
		$result = '';
/*
		$md5_hash = array();

		for($i = 0; $i < $round; $i++){
			$result = md5($this->Hex_toString($result) . $data00);
			$md5_hash[$i] = $result;
		}
		switch($bits){
			case 128://128 bit
				$key = $md5_hash[0];
				$iv = $md5_hash[1];
				break;
			case 192://192 bit
				$key = $md5_hash[0] . substr($md5_hash[1], 0,16);
				$iv = $md5_hash[2];
				break;
			case 256://256 bit
				$key = $md5_hash[0] . $md5_hash[1];
				$iv = $md5_hash[2];
				break;
			default:
				$this->appendMsg('The allowed bits are 128, 192 and 256');
				return false;
		}
		return array('key'=>$this->Hex_toString($key), 'iv'=>$this->Hex_toString($iv));
*/
		$md5_hash = '';

		for ($i = 0; $i < $round; $i++) {
			$result = md5($this->Hex_toString($result) . $data00);
			$md5_hash .= $result;
		}

		// md5 returns 128-bit hash value, which means 32 bytes.
		// 128 bit => 4 * 4(Nk) = 16 bytes key size.
		// 192 bit => 4 * 6(Nk) = 24 bytes key size.
		// 256 bit => 4 * 8(Nk) = 32 bytes key size.
		// iv is always 16 bytes. */
		//
		// Hex_toString function converts two-byte hex to make one-byte string
		// therefore the key length is (4 * Nk) * 2
		// the iv length is 16 * 2
		return array(
			'key'=>$this->Hex_toString(substr($md5_hash, 0, 2 * 4 * $this->Nk)),
			'iv'=> $this->Hex_toString(substr($md5_hash, strlen($md5_hash) - (16 * 2)))
		);
	}

} /* End of AES Class */

?>