<?php
/**
 * Copyright (c) 2012 Jacob Lee <letsgolee@naver.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA or check at http://www.gnu.org/licenses/gpl.html
 */


define('HTTP_OPENSSL_PATH', dirname(__FILE__));

/**
 * RSA에 사용될 키의 길이. 보안을 위해서 최소 512 bit를 사용하고 일반적으로 1024 bit를 사용한다.
 *
 * 최근 1024 bit로 만들어진 키가 몇달이라는 시간으로 크래킹되었다고 한다. 하지만 여전히 1024 bit는
 * 사용하기에 적합하다. 키의 길이가 길 수록 생성시간은 길어지기 때문이며 웹페이지의 보안으로
 * 가장 많이 사용되는 길이는 1024 bit이기 때문이다. 1024 bit는 128 byte의 길이이다.
 *
 * 더 강력한 보안을 위해 2048 bit를 사용할 수 있으나 생성시간이 오래걸려 브라우져나 서버가 다운될 수 있다.
 */
define('RSA_BIT_LENGTH', 1024);

/**
 * RSA 키 생성에 사용될 16진수 지수(exponent) 값. 지수값이 크면 생성 시간이 길어진다.
 * 16진수로 10001은 십진수로 65537값이다.
 */
define('RSA_EXPONENT_HEX_VALUE', '10001');

// RSA class does not use PRNG class  any more
// include_once dirname(__FILE__).'/prng.php';
include_once HTTP_OPENSSL_PATH.'/biginteger.class.php';
include_once HTTP_OPENSSL_PATH.'/aes.class.php';
include_once HTTP_OPENSSL_PATH.'/rsa.class.php';


/**
 * 그누보드4s용 http_openssl 함수
 */

/**
 * $config['cf_http_openssl_use'] 체크유무를 판단하여 http_openssl 라이브러리를 읽어들이고
 * 암호화되어 전송된 aes_key값을 복호화하여 그 값을 이용하여
 * 전송된 변수들 중 암호화 한 것들의 값을 복호화를 시도한다.
 */
function http_openssl_load()
{
	if ($GLOBALS['config']['cf_http_openssl_use']) {

		// 설정에서 http_openssl을 사용하도록 되어 있으나 PEM키를 아직 생성하지 않았을 수 있으므로 확인한다.
		if ($GLOBALS['config']['cf_pem_publickey']) {

			// RSA 생성
			$GLOBALS['rsa'] = new RSA();
			// RSA 개인키 적용
			$result = @$GLOBALS['rsa']->parsePrivatePEM($GLOBALS['config']['cf_pem_privatekey']);

			$GLOBALS['aes'] = new AES();

			// 개인키가 정상적인 경우면 http_openssl을 실행한다.
			if ($result) {

				// 보안을 위한 RSA 및 AES 오브젝트가 만들어졌으므로 http_openssl의 기능을 사용할 수 있다.
				define('HTTP_OPENSSL_START', true);

				// 보안을 위해 $aes_key 초기화
				$GLOBALS['aes_key'] = null;

				// 세션이나 _POST 혹은 _GET으로 encrypted_aes_key가 넘어왔는지 확인한다.
				if (isset($_POST['encrypted_aes_key']) || isset($_SESSION['encrypted_aes_key']) || isset($_GET['encrypted_aes_key'])) {

					// RSA로 암호화된 aes_key를 전송하거나 세션에 저장하는 이유는 값이 탈취당하더라도 PEM Private Key를 알지 못하면
					// 해석하지 못하도록 하기 위함이다. 따라서 PEM키가 탈취되었다면 새로 생성해야 한다.
					// 또한 각 페이지에서 전송되었거나 세션으로 넘어온 encrypted_aes_key는 만일의 경우를 대비해 삭제한다.

					// 일반적으로 인크립트된 encrypted_aes_key는 POST방식으로 다른 데이터와 함께 서버로 전송되나,
					// 몇몇 페이지에서는 서버에서 클라이언트로 데이터를 암호화해서 보내야 할 경우가 생긴다.
					// 그런 경우 그 페이지 이전에 클라이언트에서 aes_key를 생성하여 미리 서버에 전송해야할 필요가 있다.
					//  이 때 간혹 GET방식으로 encrypted_aes_key가 전송될 수 있다.
					// 현재 어드민 페이지 중 member_form.php에서 사용할 암호화 키가 member_list.php에서 미리 생성되어 GET방식으로 전송된다.

					// 마찬가지로 생성된 암호화된 키를 서버에서 세션에 저장하여 다음 페이지에서 사용해야 할 경우가 있으며
					// 클라이언트에서는 암호화되지 않은 키를 쿠키에 저장해야 할 경우가 있다.
					// 클라이언트의 쿠키로 저장된 AES키는 암호화하지 않아도 무방하다. 탈취되어도 의미가 없기 때문이다.
					// 이 키는 접속시마다 바뀐다.
					// 현재 그누보드에서 세션 저장이 필요한 경우는 skin/member/basic/register_update.skin.php와 위에서 언급한 어드민 페이지가 있다.

					// 따라서 클라이언트에서 암호화되지 않은 키를 쿠키에 저장하기 위해 쿠키 클래스가 사용될 것이다.
					// 그누보드에서 제공하는 쿠키 함수는 오류가 있기 때문에 사용하지 않는다.
					$GLOBALS['encrypted_aes_key'] = isset($_POST['encrypted_aes_key']) ? $_POST['encrypted_aes_key'] : (isset($_SESSION['encrypted_aes_key']) ? $_SESSION['encrypted_aes_key'] : $_GET['encrypted_aes_key']);

					// 복호화를 시도하여 $aes_key를 구한다.
					$GLOBALS['aes_key'] = $GLOBALS['rsa']->decrypt($GLOBALS['encrypted_aes_key']);
					// base64_decode 를 실행한다.
					$GLOBALS['aes_key'] = $GLOBALS['aes_key'] ? base64_decode($GLOBALS['aes_key']): false;

					// 만일의 경우를 대비해 삭제
					unset($_POST['encrypted_aes_key']);
					unset($_SESSION['encrypted_aes_key']);
					unset($_GET['encrypted_aes_key']);
					unset($_REQUEST['encrypted_aes_key']); // 확인사살
				}

				// $aes_key가 존재한다는 것은 AES 키로 암호화된 값들이 존재함을 의미한다.
				// AES로 암호화된 값을 복호화하는 경우는 오직 $_POST만 허용한다.
				if ($GLOBALS['aes_key']) {
					foreach ($_POST as $key=>$value) {
						// 보안상 문제가 될 수 있는 것은 배제한다. 
						// 또한 이미 그누보드의 common.php에서 extract($_POST)가 사용되었으므로
						// 존재하는 경우만 허용하자.
						if (in_array($key, array('config', 'member', 'board', 'group', 'g4'))) {
							unset($_POST[$key]);
							continue;
						}
						$new_val = $GLOBALS['aes']->decrypt($_POST[$key], $GLOBALS['aes_key']);

						// 만일 복호화 한 값이 false로 주어지면 AES로 암호화하지 않은 변수 값이라는 것을 의미한다.
						// 그런 경우 그냥 그 값을 돌려주기만 하면 된다. 왜냐면 사용자가 암호화되길 원한다면 
						// 자바스크립트를 죽이지 않을 것이기 때문이다.
						// 복호화와 동시에 extract기능을 사용하자.
						if ($new_val !== false) {
							// g4s에서는 url encoding을 할 때 $_REQUEST['url']를 이용한다...
							$GLOBALS[$key] = $_REQUEST[$key] = $_POST[$key] = $new_val;
						}
					}
				}
			}
		}
	}
}


/**
 * 암호화된 AES 키값을 세션에 저장한다.
 *
 * http_openssl_load() 함수에 의해 세션을 삭제하기 때문에 
 * 필요에 따라 세션에 저장해야 할 경우 사용.
 */
function http_openssl_save_session()
{
	if (strlen($GLOBALS['encrypted_aes_key'])) $_SESSION['encrypted_aes_key'] = $GLOBALS['encrypted_aes_key'];
}


/**
 * 서버에 암호화한 데이터만 전송 하는 경우 사용
 *
 * 이 경우는 서버에서 암호화된 데이터를 받아 처리할 필요 없이 단순하게
 * 클라이언트에서 AES 대칭키를 생성해서 폼 데이터를 암호화한 후 서버에 전송한다.
 */
function http_openssl_simple_prepare($create_cookie=false, $js_code='')
{
	if (!defined('HTTP_OPENSSL_READY')) {
?>

<script type="text/javascript" src="<?php echo G4_URL.'/'.G4_LIB_DIR;?>/http_openssl/prng.class.js"></script>
<script type="text/javascript" src="<?php echo G4_URL.'/'.G4_LIB_DIR;?>/http_openssl/biginteger.class.js"></script>
<script type="text/javascript" src="<?php echo G4_URL.'/'.G4_LIB_DIR;?>/http_openssl/aes.class.js"></script>
<script type="text/javascript" src="<?php echo G4_URL.'/'.G4_LIB_DIR;?>/http_openssl/rsa.class.js"></script>
<script type="text/javascript" src="<?php echo G4_URL.'/'.G4_LIB_DIR;?>/http_openssl/cookie.class.js"></script>
<script type="text/javascript">
var pem_publickey = "<?php echo str_replace("\n", '', $GLOBALS['config']['cf_pem_publickey']);?>";
var rsa = new RSA();
rsa.parsePublicPEM(pem_publickey);

var aes = new AES();
<?php
		if ($GLOBALS['aes_key'] && $_COOKIE[md5('aes_key')] && ($GLOBALS['aes_key'] == base64_decode($_COOKIE[md5('aes_key')]))) {
		//preg_match('/^'.preg_quote(G4_URL.'/'.G4_ADMIN_DIR.'/', '/').'([a-z_0-9]+)\.php/i', $_SERVER['HTTP_REFERER'], $dummy)) {
		// 어드민페이지의 경우 admin.head.php파일에 http_openssl_simple_prepare() 함수가 있는 까닭에
		// 또 어드민 상단 메뉴에 "관리자 정보수정" 링크를 바로 클릭하면 다음페이지에서 다시 이 함수가
		// 호출이 된다. 그런데 GET방식으로 넘어온 aes_key를 다시 사용해야 하는데 
		// 자바스크립트에서 새로 생성하면 서버의 aes_key와 클라이언트의 키가 서로 일치하지 않게 되는 문제가 발생
		// 따라서 서버 상에 $aes_key가 있다면 클라이언트는 키를 쿠키에서 가져와야 함.
		// 문제는 현재의 페이지가 정상경로에서 왔는지가 문제...
		// 천상 HTTP_REFERER를 이용해서 구분해야 한다. 문제는 이게 수정가능하다는 것...

		// 어드민 상단 "관리자 정보수정" 링크를 처음 클릭하면 HTTP_REFERER를 찾지 못함...
		// 쿠키 찾는 것으로 급 수정
?>
var aes_key = BASE64.decode(COOKIE.get(AES.MD5('aes_key', true)));

<?php
		} else {
?>
var aes_key = aes.randomAESKey(32);

<?php
		}

		if ($create_cookie) {
?>
COOKIE.set(AES.MD5('aes_key', true), BASE64.encode(aes_key));

<?php
		} // if ($create_cookie)
	
		if (strlen($js_code)) echo $js_code;
?>

</script>
<noscript>
<div style="color:crimson">자바스크립트 사용이 중지되어 있습니다.<br/>
자바스크립트를 사용하지 않으면 정보가 암호화되지 않기 때문에 보안상 노출될 수 있습니다.
</div>
</noscript>
<?php
	} // if (!defined('HTTP_OPENSSL_READY'))

	if (!defined('HTTP_OPENSSL_READY')) define('HTTP_OPENSSL_READY', true);
}

/*
* register_form.skin.php
*
* 서버에 암호화된 데이터를 받은 후 먼저 복호화 하여 폼에 데이터를 주어야 하는 경우 사용한다.
*
* 예를 들어 회원 정보를 업데이트하는 경우 먼저 서버에서 회원 정보를 받아와야 한다. 
* 그 때 서버에서 암호화한 데이터를 받아오도록 한 후 클라이언트에서 복호화하는 과정을 거치도록 해야 한다.
*/
function http_openssl_prepare(
	$condi_var_name='',        // 조건에 사용될 변수 이름
	$condi_val='',             // 조건에 사용될 변수의 값. 즉 $GLOBALS['condi_var_name'] 값이 $condi_val과 같을 경우를 체크한다.
	$encrypt_var_names='',     // 서버에서 암호화 할 변수들의 이름의 집합
	$save_array_name='',       // 암호화할 array 이름. register_form.skin.php에서는 $member가 사용되므로 'member'가 된다.
	$form_name='fregisterform',// 폼의 이름. 클라이언트에서 암호화 된 데이터를 복호화하여 폼에 입력하기 위해 사용된다.
	$js_code=''                // 폼처리 과정 추가적으로 필요한 javascript 코드를 적어준다. 예로 mb_email 외에 old_email 값도 복호화해야 한다.
) {

	$GLOBALS['http_openssl'] = array();

	// 만일의 경우를 대비해서 $save_array_name(예: $member) array를 직접 다루지 않고 복사를 해서 다루자.
	// 혹은 $member를 임시 변수에 저장한 후에 다 처리가 되면 다시 복구하는 방법을 사용하자.
	if ($work_array_name) {
		$GLOBALS['http_openssl']['save_array'] = $GLOBALS[$save_array_name];
		$GLOBALS['http_openssl']['save_name'] = $save_array_name;
	}

	if ($GLOBALS[$condi_var_name] == $condi_val) {
		// register_form.skin.php의 경우 member_confirm.skin.php를 통해 암호화된 aes_key가 이미 서버에 있다.

		// 만일 업데이트인 경우 즉 $w값이 'u'인경우는 변수값들이 AES 암호화 되어야 하며
		// 폼에 보일 때에는 암호화 된 값들이 복호화 한 후에 표시되도록 해야 한다.

		// 암호화할 변수들
		// 모든 변수를 암호화할 이유가 없다. 필요한 변수만 암호화하여 전송하도록 하자.
		// 또 모두 암호화 되면 문제가 생긴다.
		// ajax로 처리되면 좋겠지만... 그건 나중 문제...
		// $member[mb_id], $member[mb_nick], $member[mb_name], $member[mb_email], $member[mb_birth] 등등이 있다.
		//$encrypt_var_names = array('mb_id', 'mb_nick', 'mb_name', 'mb_email', 'mb_birth', 'mb_homepage', 'mb_tel', 'mb_hp', 'mb_zip1', 'mb_zip2', 'mb_addr1', 'mb_addr2');

		if (is_array($encrypt_var_names)) {
			foreach ($GLOBALS[$save_array_name] as $key=>$val) {
				if (in_array($key, $encrypt_var_names)) {
					$GLOBALS[$save_array_name][$key] = $GLOBALS['aes']->encrypt($val, $GLOBALS['aes_key']);
				}
			}
		}
	}

	if (!defined('HTTP_OPENSSL_READY')) {
?>

<script type="text/javascript" src="<?php echo G4_URL.'/'.G4_LIB_DIR;?>/http_openssl/prng.class.js"></script>
<script type="text/javascript" src="<?php echo G4_URL.'/'.G4_LIB_DIR;?>/http_openssl/biginteger.class.js"></script>
<script type="text/javascript" src="<?php echo G4_URL.'/'.G4_LIB_DIR;?>/http_openssl/aes.class.js"></script>
<script type="text/javascript" src="<?php echo G4_URL.'/'.G4_LIB_DIR;?>/http_openssl/rsa.class.js"></script>
<script type="text/javascript" src="<?php echo G4_URL.'/'.G4_LIB_DIR;?>/http_openssl/cookie.class.js"></script>
<script type="text/javascript">
var pem_publickey = "<?php echo str_replace("\n", '', $GLOBALS['config']['cf_pem_publickey']);?>";
var rsa = new RSA();
rsa.parsePublicPEM(pem_publickey);

var aes = new AES();

<?php
	} else {//if (!defined('HTTP_OPENSSL_READY'))
?>
<script type="text/javascript">
<?php
	}

	if ($GLOBALS[$condi_var_name] == $condi_val) { 
// 업데이트인 경우 AES 대칭키를 쿠키에서 가져와야 한다.
?>
if (!aes_key) {
	var aes_key = BASE64.decode(COOKIE.get(AES.MD5('aes_key', true)));
}

if (!aes_key.length) {
	alert('보안 키가 생성되지 않았습니다. 정상적인 경로를 이용해 주세요.');
	history.go(-1);
}

<?php
	} else { 
// AES 대칭키를 생성한다. 
// 이 대칭키는 데이터를 암호화 하는 데 사용이 되고, 이 키 역시 RSA 공개키로 암호화되어 전송될 것이다.
// 대칭키 생성 후 쿠키로 저장한다.
		if (!defined('HTTP_OPENSSL_READY')) {
?>
var aes_key = aes.randomAESKey(32);

<?php
		} //if (!defined('HTTP_OPENSSL_READY'))
?>
COOKIE.set(AES.MD5('aes_key', true), aes_key);
<?php 
	} // else

	if ($GLOBALS[$condi_var_name] == $condi_val) { ?>
$(document).ready(function() {
	var f = document.getElementById('<?php echo $form_name;?>');
	var encrypt_data = {
<?php
		// 폼의 input에 설정된 maxlength 값에 의해 암호화된 글자의 크기가 잘려나가는 현상이 생긴다. 
		// 그러므로 따로 암호화 된 값을 저장해야 할 필요가 있다.
		foreach ($encrypt_var_names as $key=>$val) {
			echo "\t\t'".$val."':'".$GLOBALS[$save_array_name][$val]."'".($key == count($encrypt_var_names)-1 ? "" : ",")."\n";
		}
?>
	};

	for (var i in encrypt_data) {
		if (f.elements[i]) f.elements[i].value = aes.decrypt(encrypt_data[i], aes_key);

<?php
		echo $js_code;
?>

	}
});
<?php
	} //if ($GLOBALS[$condi_var_name] == $condi_val)
	
	if (!defined('HTTP_OPENSSL_READY')) {
?>

</script>
<noscript>
<div style="color:crimson">자바스크립트 사용이 중지되어 있습니다.<br/>
자바스크립트를 사용하지 않으면 정보가 암호화되지 않기 때문에 보안상 노출될 수 있습니다.
</div>
</noscript>
<?php
	} else {//if (!defined('HTTP_OPENSSL_READY'))
?>
</script>
<?php
	}

	if (!defined('HTTP_OPENSSL_READY')) define('HTTP_OPENSSL_READY', true);
}


/*
* http_openssl 폼 처리 함수
* submit 함수 안에 넣어주어야 하며 "return true;"를 대체하여 주면 된다.
*
* 변수로는 onsubmit 함수의 폼 이름을 준다.
*/
function http_openssl_js_form_submit($form_name='f')
{
	if (defined('HTTP_OPENSSL_READY')) {
?>

	var form = document.createElement('form');
	var div = document.createElement('div');
	div.style.display = 'none';
	div.style.visibility = 'hidden';
	div.appendChild(form);
	document.body.appendChild(div);

	for (var i = 0; i < <?php echo $form_name;?>.elements.length; i++) {
		var el = <?php echo $form_name;?>.elements[i];

		if (el.type != 'button' && el.type != 'image' && el.type != 'submit' && el.name) { 
			var input = document.createElement('input');
			input.name = el.name;
			input.type = el.type;
			input.value = aes.encrypt(el.value, aes_key);
			if (el.type == 'checkbox' || el.type == 'radio') input.checked = el.checked;
			form.appendChild(input);
		}			
	}

	form.enctype = <?php echo $form_name;?>.enctype;
	form.action = <?php echo $form_name;?>.action;
	form.method = <?php echo $form_name;?>.method;
	form.name = <?php echo $form_name;?>.name + '-2';

	var input = document.createElement('input');
	input.type = 'hidden';
	input.name = 'encrypted_aes_key';
	input.value = rsa.encrypt(BASE64.encode(aes_key));
	form.appendChild(input);


<?php
	if (is_array($GLOBALS['http_openssl']) && is_array($GLOBALS['http_openssl']['save_array'])) {
		// 암호화 한 변수를 다음 사용을 위해 복원해야 하는 경우가 있다.
		// 예로 register_form.skin.php에서 $member의 일부 데이터를 암호화 한 후 클라이언트로 전송했는데
		// 작업이 끝나면 혹 다른 스킨이나 프로그램에서 $member를 사용할 수 있으므로 원래의 값으로 
		// 되돌리는 작업이 필요로 한다.
		$GLOBALS[$GLOBALS['http_openssl']['save_name']] = $GLOBALS['http_openssl']['save_array'];
		unset($GLOBALS['http_openssl']);
	}
?>

	form.submit();

	<?php echo $form_name;?>.disabled = true;

	return false;

	<?php } else { ?>

    return true;

	<?php
	}
}



?>
