<?php
/**
 * JavaScript RSA Encryption library.
 * Depends on biginteger.php(phpseclib's Math_BigInteger) and rng.class.php
 *
 * Copyright (c) 2012 Jacob Lee <letsgolee@naver.com>
 *
 * $this program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * $this program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with $this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA or check at http://www.gnu.org/licenses/gpl.html
 */

/*

RSAPublicKey::=SEQUENCE{
	modulus INTEGER, -- n
	publicExponent INTEGER -- e
}


RSAPrivateKey ::= SEQUENCE {
	version Version,
	modulus INTEGER, -- n
	publicExponent INTEGER, -- e
	privateExponent INTEGER, -- d
	prime1 INTEGER, -- p
	prime2 INTEGER, -- q
	exponent1 INTEGER, -- d mod (p-1)
	exponent2 INTEGER, -- d mod (q-1)
	coefficient INTEGER -- (inverse of q) mod p
}
*/


class RSA {
	function RSA()
	{
		$this->n = null;
		$this->e = null; /* Public exponent */
		$this->d = null;
		$this->p = null;
		$this->q = null;
		$this->dmp1 = null; /* D mod (P-1) */
		$this->dmq1 = null; /* D mod (Q-1) */
		$this->coeff = null; /* 1/Q mod P */
	}

	// PKCS#1 (type 2, random) pad input string s to n bytes, and return a bigint
	function PKCS1pad2($s,$n) {
		if($n < strlen($s) + 11) {
			trigger_error("Message too long for RSA", E_USER_ERROR);
			return null;
		}
		$ba = array();

		$i = strlen($s) - 1;
		while($i >= 0 && $n > 0) { // encode using utf-8
			$c = ord($s[$i--]);
			if($c < 128) {
				$ba[--$n] = $c;
			}
			else if(($c > 127) && ($c < 2048)) {
				$ba[--$n] = ($c & 63) | 128;
				$ba[--$n] = ($c >> 6) | 192;
			}
			else {
				$ba[--$n] = ($c & 63) | 128;
				$ba[--$n] = (($c >> 6) & 63) | /* 128 */192;
				$ba[--$n] = ($c >> 12) | 224;
			}
		}
		$ba[--$n] = 0;
/*
		$rng = new PRNG();
		while($n > 2) { // random non-zero pad
			while(0 == ($x = $rng->getByte()));
			$ba[--$n] = $x;
		}
*/
		while($n > 2) { // random no-zero pad
			$x = mt_rand(0x01,0xff);
			$ba[--$n] = $x;
		}
		$ba[--$n] = 2;
		$ba[--$n] = 0;

		return new Math_BigInteger(RSA_ByteArray_toString($ba), 256);
	}

	// Undo PKCS#1 (type 2, random) padding and, if valid, return the plaintext
	function PKCS1unpad2($d,$n) {
		$b = RSA_ByteArray_fromString($d->toBytes());
		$i = 0;
		while($i < count($b) && $b[$i] == 0) ++$i;
		if(count($b)-$i != $n-1 || $b[$i] != 2)
			return null;
		++$i;
		while($b[$i] != 0)
			if(++$i >= count($b)) return null;
		$ret = "";
		while(++$i < count($b)) {
			$c = $b[$i] & 255;
			if($c < 128) { // utf-8 decode
				$ret .= chr($c);
			}
			else if(($c > 191) && ($c < 224)) {
				$ret .= chr((($c & 31) << 6) | ($b[$i+1] & 63));
				++$i;
			}
			else {
				$ret .= chr((($c & 15) << 12) | (($b[$i+1] & 63) << 6) | ($b[$i+2] & 63));
				$i += 2;
			}
		}

		return $ret;
	}


	// Set the public key fields N and e from hex strings
	function setPublicKey($N,$E) {
		if(!is_null($N) && !is_null($E) && strlen($N) > 0 && strlen($E) > 0) {
			$N = preg_replace('/^00/', '', $N);
			$E = preg_replace('/^00/', '', $E);

			$this->n = new Math_BigInteger($N,16);
			$this->e = new Math_BigInteger($E,16);
		}
		else
			trigger_error("Invalid RSA public key", E_USER_ERROR);
	}

	// protected
	// Perform raw public operation on "x": return x^e (mod n)
	function _rawEncrypt($x) {
		return $x->modPow($this->e, $this->n);
	}

	// Return the PKCS#1 RSA encryption of "text" as an even-length hex string
	function encrypt($text) {
		$m = $this->PKCS1pad2($text, strlen($this->n->toBytes()));
		if(is_null($m)) return null;
		$c = $this->_rawEncrypt($m);
		if(is_null($c)) return null;
		$h = $c->toString(16);
		if((strlen($h) & 1) == 0) return $h; else return "0" . $h;
	}

	/*
	setPrivateKey function is not recommended
	for if p and q are known then
	ciphermessage ^ D mod n is the only calculation method and it consumes a lot of time!
	*/
	// Set the private key fields N, e, and d from hex strings
	function setPrivateKey($N,$E,$D) {
		if(!is_null($N) && !is_null($E) && strlen($N) > 0 && strlen($E) > 0) {
			$N = preg_replace('/^00/', '', $N);
			$E = preg_replace('/^00/', '', $E);
			$D = preg_replace('/^00/', '', $D);

			$this->n = new Math_BigInteger($N,16);
			$this->e = new Math_BigInteger($E,16);
			$this->d = new Math_BigInteger($D,16);

			return true;
		}
		else {
			trigger_error("Invalid RSA private key", E_USER_ERROR);
			return false;
		}
	}


	// Set the private key fields D, P, and Q from hex strings
	function setPrivateKey2($D,$P,$Q) {
		if(!is_null($D) && !is_null($P) && !is_null($Q)) {
			$D = preg_replace('/^00/', '', $D);
			$P = preg_replace('/^00/', '', $P);
			$Q = preg_replace('/^00/', '', $Q);

			$this->d = new Math_BigInteger($D,16);
			$this->p = new Math_BigInteger($P,16);
			$this->q = new Math_BigInteger($Q,16);

			$p1 = $this->p->subtract(new Math_BigInteger(1));
			$q1 = $this->q->subtract(new Math_BigInteger(1));

			$this->n = $this->p->multiply($this->q);
			$this->dmp1 = $this->d->mod($p1);
			$this->dmq1 = $this->d->mod($q1);
			$this->coeff = $this->q->modInverse($this->p);

			return true;
		}
		else {
			trigger_error("Invalid RSA private key", E_USER_ERROR);
			return false;
		}
	}



	// Set the private key fields N, e, d and CRT params from hex strings
	function setPrivateKeyEx($N,$E,$D,$P,$Q,$DP,$DQ,$C) {
		if(!is_null($D) && !is_null($P) && !is_null($Q)) {
			$N = preg_replace('/^00/', '', $N);
			$E = preg_replace('/^00/', '', $E);
			$D = preg_replace('/^00/', '', $D);
			$P = preg_replace('/^00/', '', $P);
			$Q = preg_replace('/^00/', '', $Q);
			$DP = preg_replace('/^00/', '', $DP);
			$DQ = preg_replace('/^00/', '', $DQ);
			$C = preg_replace('/^00/', '', $C);

			$this->n = new Math_BigInteger($N,16);
			$this->e = new Math_BigInteger($E,16);
			$this->d = new Math_BigInteger($D,16);
			$this->p = new Math_BigInteger($P,16);
			$this->q = new Math_BigInteger($Q,16);
			$this->dmp1 = new Math_BigInteger($DP,16);
			$this->dmq1 = new Math_BigInteger($DQ,16);
			$this->coeff = new Math_BigInteger($C,16);
			
			return tru;
		}
		else {
			trigger_error("Invalid RSA private key", E_USER_ERROR);
			return false;
		}
	}


	// protected
	// Perform raw private operation on "x": return x^d (mod n)
	function _rawDecrypt($x) {
		if(is_null($this->p) || is_null($this->q))
			return $x->modPow($this->d, $this->n);

		// TODO: re-calculate any missing CRT params
		$xp = $x->mod($this->p)->modPow($this->dmp1, $this->p);
		$xq = $x->mod($this->q)->modPow($this->dmq1, $this->q);

		while($xp->compare($xq) < 0) $xp = $xp->add($this->p);
		return $xp->subtract($xq)->multiply($this->coeff)->mod($this->p)->multiply($this->q)->add($xq);
	}

	// Return the PKCS#1 RSA decryption of "ctext".
	// "ctext" is an even-length hex string and the output is a plain string.
	function decrypt($ctext) {
		$c = new Math_BigInteger($ctext, 16);
		$m = $this->_rawDecrypt($c);
		if(is_null($m)) return null;
		return $this->PKCS1unpad2($m, strlen($this->n->toBytes()));
	}


	// Generate a new random private key B bits long, using public expt E
	function generateKey($B,$E) {
		$this->e = new Math_BigInteger($E, 16);

		for(;;) {
			$this->p = new Math_BigInteger();
			$this->p = $this->p->randomPrime(new Math_BigInteger(str_pad('1', ($B / 8), '0'), 16),new Math_BigInteger(str_pad('f', ($B / 8), 'f'), 16));

			$this->q = new Math_BigInteger();
			$this->q = $this->q->randomPrime(new Math_BigInteger(str_pad('1', ($B / 8), '0'), 16),new Math_BigInteger(str_pad('f', ($B / 8), 'f'), 16));


			for(;;) {
				if($this->p->subtract(new Math_BigInteger(1))->gcd($this->e)->compare(new Math_BigInteger(1)) == 0) break;
				else $this->p = $this->p->subtract(new Math_BigInteger(1));
			}
			for(;;) {
				if($this->q->subtract(new Math_BigInteger(1))->gcd($this->e)->compare(new Math_BigInteger(1)) == 0) break;
				else $this->q = $this->q->subtract(new Math_BigInteger(1));
			}

			if($this->p->compare($this->q) <= 0) {
				$t = $this->p;
				$this->p = $this->q;
				$this->q = $t;
			}
			$p1 = $this->p->subtract(new Math_BigInteger(1));
			$q1 = $this->q->subtract(new Math_BigInteger(1));
			$phi = $p1->multiply($q1);

			if($phi->gcd($this->e)->compare(new Math_BigInteger(1)) == 0) {
				$this->n = $this->p->multiply($this->q);
				$this->d = $this->e->modInverse($phi);
				$this->dmp1 = $this->d->mod($p1);
				$this->dmq1 = $this->d->mod($q1);
				$this->coeff = $this->q->modInverse($this->p);
				break;
			}
		}
	}



	function getPrivatePEM() {
		$pem = '';
		$pem = '020100';
		$pem .= RSA_ASN1_intValue($this->n, true);
		$pem .= RSA_ASN1_intValue($this->e, false);
		$pem .= RSA_ASN1_intValue($this->d, false);
		$pem .= RSA_ASN1_intValue($this->p, true);
		$pem .= RSA_ASN1_intValue($this->q, true);
		$pem .= RSA_ASN1_intValue($this->dmp1, true);
		$pem .= RSA_ASN1_intValue($this->dmq1, false);
		$pem .= RSA_ASN1_intValue($this->coeff, false);
		$pem = '30' . RSA_ASN1_lengthValue($pem) . $pem;
		return "-----BEGIN RSA PRIVATE KEY-----\n" . RSA_lineBreak(base64_encode(RSA_Hex_decode($pem)),64) . "\n-----END RSA PRIVATE KEY-----";
	}

	function getPublicPEM() {
		$pem = '';
		$pem = RSA_ASN1_intValue($this->n, true);
		$pem .= RSA_ASN1_intValue($this->e, false);
		$pem = '30' . RSA_ASN1_lengthValue($pem) . $pem;
		$pem = '03' . RSA_ASN1_lengthValue($pem, 1) . '00' . $pem;
		$pem = '300d06092a864886f70d0101010500' . $pem; // 1.2.840.113549.1.1.1
		$pem = '30' . RSA_ASN1_lengthValue($pem) . $pem;
		return "-----BEGIN PUBLIC KEY-----\n" . RSA_lineBreak(base64_encode(RSA_Hex_decode($pem)),64) . "\n-----END PUBLIC KEY-----";
	}

	function parsePublicPEM($pem) {
		$pem = preg_replace("/\n/s", '', preg_replace('/-----(BEGIN|END) (RSA )?(PUBLIC|PRIVATE) KEY-----/s', '', $pem));
		$asn1 = new RSA_ASN1();
		$data = $asn1->parse(base64_decode($pem));

		if($data[0][0][0]=="1.2.840.113549.1.1.1")
			return $this->setPublicKey(RSA_Hex_encode($data[0][1][0][0]), RSA_Hex_encode($data[0][1][0][1]));
		return false;
	}

	function parsePrivatePEM($pem) {
		$pem = preg_replace("/\n/s", '', preg_replace('/-----(BEGIN|END) (RSA )?(PUBLIC|PRIVATE) KEY-----/s', '', $pem));
		$asn1 = new RSA_ASN1();
		$data = $asn1->parse(base64_decode($pem));

		return $this->setPrivateKeyEx(RSA_Hex_encode($data[0][1]), RSA_Hex_encode($data[0][2]), RSA_Hex_encode($data[0][3]), RSA_Hex_encode($data[0][4]), RSA_Hex_encode($data[0][5]), RSA_Hex_encode($data[0][6]), RSA_Hex_encode($data[0][7]), RSA_Hex_encode($data[0][8]));
	}


	// Generate a new random PEM public/private keypair, which keys are B bits long, using public expt E
	function generatePEMKey($B,$E)
	{
		$this->generateKey($B,$E);

		$pemKeys = array();
		$pemKeys['public'] = $this->getPublicPEM();
		$pemKeys['private'] = $this->getPrivatePEM();

		return $pemKeys;
	}

}




class RSA_ASN1 {
	var $error = false;
	var $data = null;

	function RSA_ASN1($data = null) {
		if (!is_null($data) && !empty($data))
			$this->data = $this->parse($data);
	}

	function parse($data) {
		if (!$data) {
			$this->error = true;
			return null;
		}
		$result = array();
		while(strlen($data) > 0) {
			// get the tag
			$tag = ord($data[0]);
			$data = substr($data, 1);
			// get length
			$length = 0;
			// ignore any null tag
			if (($tag & 31) == 0x5) $data = substr($data,1);
			else {
				if (ord($data[0]) & 128) {
					$lengthSize = ord($data[0]) & 127;
					$data = substr($data,1);
					if($lengthSize > 0) $length = ord($data[0]);
					if($lengthSize > 1)	$length = (($length << 8) | ord($data[1]));
					if($lengthSize > 2) {
						$this->error = true;
						return null;
					}
					$data = substr($data,$lengthSize);
				} else {
					$length = ord($data[0]);
					$data = substr($data,1);
				}
			}
			// get value
			$value = "";
			if($length) {
				if ($length > strlen($data)){
					$this->error = true;
					return null;
				}
				$value = substr($data, 0, $length);
				$data = substr($data, $length);
			}
			if ($tag & 32)
				array_push($result, $this->parse($value)); // sequence
			else
				array_push($result, $this->_getValue(($tag & 128) ? 4 : ($tag & 31), $value));
		}
		return $result;
	}

	function _getValue($tag, $data) {
		if ($tag == 1)
			return $data ? true : false;
		else if ($tag == 2) //integer
			return $data;
		else if ($tag == 3) //bit string
			return $this->parse(substr($data,1));
		else if ($tag == 5) //null
			return null;
		else if ($tag == 6){ //ID
			$res = array();
			$d0 = ord($data[0]);
			array_push($res,floor($d0 / 40));
			array_push($res,$d0 - $res[0]*40);
			$stack = array();
			$powNum = 0;
			for($i=1;$i<strlen($data);$i++){
				$token = ord($data[$i]);
				array_push($stack,$token & 127);
				if ( $token & 128 )
					$powNum++;
				else {
					$sum = 0;
					for($j=0;$j<count($stack);$j++)
						$sum += $stack[$j] * pow(128, $powNum--);
					array_push($res, $sum);
					$powNum = 0;
					$stack = array();
				}
			}
			return implode(".", $res);
		}
		return null;
	}
}


/* functions that are used in RSA and RSA_ASN1 class */


function RSA_ASN1_int2hex($integer) {
	if (gettype($integer) == 'object') $int = $integer->toString(16); // bigInteger
	else if (is_int($integer)) $int = dechex($integer);
	else $int = $integer; // already hex value

	if (strlen($int) % 2 !== 0) {
		$int = '0' . $int;
	}
	return $int;
}

function RSA_ASN1_lengthValue($content, $extra = 0) {
	if (is_null($extra) || empty($extra)) {
		$extra = 0;
	}
	$len = (strlen($content) / 2) + $extra;
	if ($len > 127) {
		$len = RSA_ASN1_int2hex($len);
		return RSA_ASN1_int2hex(0x80 + (strlen($len) / 2)) . $len;
	} else {
		return RSA_ASN1_int2hex($len);
	}
}

function RSA_ASN1_intValue($integer, $nullPrefixed = 0) {
	$integer = RSA_ASN1_int2hex($integer);
	if ($nullPrefixed) {
		$integer = '00' . $integer;
	}
	return '02' . RSA_ASN1_lengthValue($integer) . $integer;
}



function RSA_Hex_encode($input) {
	$hex = "0123456789abcdef";

	if(!$input) return false;
	$output = "";
	$i = 0;
	do {
		$k = ord($input[$i++]);
		$output .= $hex[($k >> 4) & 0xf] . $hex[$k & 0xf];
	} while ($i < strlen($input));
	return $output;
}

function RSA_Hex_decode($input) {
	$hex = "0123456789abcdef";

	if(!$input) return false;
	$input = preg_replace('/[^0-9abcdef]/s', '', $input);
	$output = "";
	$i = 0;
	do {
		$output .= chr(((strpos($hex, $input[$i++]) << 4) & 0xf0) | (strpos($hex, $input[$i++]) & 0xf));
	} while ($i < strlen($input));
	return $output;
}


function RSA_ByteArray_fromString($s) {
	if(!$s) $s = "";
	$b = array();
	for($i=0, $cnt=strlen($s); $i < $cnt; $i++){
		 $b[$i] = ord($s[$i]);
	}
	return $b;
}


function RSA_ByteArray_toString($b) {
	$s = '';
	for($i=0, $cnt=count($b); $i < $cnt; $i++){
		 $s .= chr($b[$i]);
	}
	return $s;
}


function RSA_lineBreak($s,$n) {
	$ret = "";
	$i = 0;
	while($i + $n < strlen($s)) {
		$ret .= substr($s, $i, $n) . "\n";
		$i += $n;
	}
	return $ret . substr($s, $i);
}



?>