<?php
declare(strict_types=1);

/**
 * 간단한 5카드 드로우 포커 게임
 * PHP 8.2 호환
 */

// 세션 시작
session_start();

// 카드 클래스
class Card {
    public function __construct(
        private string $suit,
        private string $value
    ) {}
    
    public function getSuit(): string {
        return $this->suit;
    }
    
    public function getValue(): string {
        return $this->value;
    }
    
    public function toString(): string {
        return $this->suit . ' ' . $this->value;
    }
}

// 덱 클래스
class Deck {
    private array $cards = [];
    
    public function __construct() {
        $this->initialize();
    }
    
    public function initialize(): void {
        $this->cards = [];
        $suits = ['♥', '♦', '♠', '♣'];
        $values = ['2', '3', '4', '5', '6', '7', '8', '9', '10', 'J', 'Q', 'K', 'A'];
        
        foreach ($suits as $suit) {
            foreach ($values as $value) {
                $this->cards[] = new Card($suit, $value);
            }
        }
    }
    
    public function shuffle(): void {
        shuffle($this->cards);
    }
    
    public function deal(int $count = 1): array {
        $dealtCards = [];
        
        for ($i = 0; $i < $count; $i++) {
            if (count($this->cards) > 0) {
                $dealtCards[] = array_pop($this->cards);
            }
        }
        
        return $dealtCards;
    }
    
    public function getCount(): int {
        return count($this->cards);
    }
}

// 핸드 클래스
class Hand {
    private array $cards = [];
    private array $handRanks = [
        'Royal Flush' => 9,
        'Straight Flush' => 8,
        'Four of a Kind' => 7,
        'Full House' => 6,
        'Flush' => 5,
        'Straight' => 4,
        'Three of a Kind' => 3,
        'Two Pair' => 2,
        'Pair' => 1,
        'High Card' => 0
    ];
    
    public function __construct(array $cards = []) {
        $this->cards = $cards;
    }
    
    public function addCards(array $cards): void {
        $this->cards = array_merge($this->cards, $cards);
    }
    
    public function getCards(): array {
        return $this->cards;
    }
    
    public function discard(array $indices): array {
        $discarded = [];
        rsort($indices); // 높은 인덱스부터 제거하기 위해 정렬
        
        foreach ($indices as $index) {
            if (isset($this->cards[$index])) {
                $discarded[] = array_splice($this->cards, $index, 1)[0];
            }
        }
        
        return $discarded;
    }
    
    public function evaluateHand(): array {
        // 값과 무늬로 그룹화
        $valueGroups = [];
        $suitGroups = [];
        
        foreach ($this->cards as $card) {
            $value = $card->getValue();
            $suit = $card->getSuit();
            
            $valueGroups[$value] = ($valueGroups[$value] ?? 0) + 1;
            $suitGroups[$suit] = ($suitGroups[$suit] ?? 0) + 1;
        }
        
        // 카드 값 배열 생성
        $values = array_map(fn($card) => $card->getValue(), $this->cards);
        $numericValues = $this->convertToNumericValues($values);
        
        // 플러시 체크
        $isFlush = count($suitGroups) === 1;
        
        // 스트레이트 체크
        sort($numericValues);
        $isStraight = $this->isStraight($numericValues);
        
        // 로얄 플러시
        if ($isFlush && $isStraight && max($numericValues) === 14) {
            return ['rank' => 'Royal Flush', 'value' => $this->handRanks['Royal Flush']];
        }
        
        // 스트레이트 플러시
        if ($isFlush && $isStraight) {
            return ['rank' => 'Straight Flush', 'value' => $this->handRanks['Straight Flush']];
        }
        
        // 포카드
        if (in_array(4, $valueGroups)) {
            return ['rank' => 'Four of a Kind', 'value' => $this->handRanks['Four of a Kind']];
        }
        
        // 풀하우스
        if (in_array(3, $valueGroups) && in_array(2, $valueGroups)) {
            return ['rank' => 'Full House', 'value' => $this->handRanks['Full House']];
        }
        
        // 플러시
        if ($isFlush) {
            return ['rank' => 'Flush', 'value' => $this->handRanks['Flush']];
        }
        
        // 스트레이트
        if ($isStraight) {
            return ['rank' => 'Straight', 'value' => $this->handRanks['Straight']];
        }
        
        // 트리플
        if (in_array(3, $valueGroups)) {
            return ['rank' => 'Three of a Kind', 'value' => $this->handRanks['Three of a Kind']];
        }
        
        // 투페어
        $pairCount = array_count_values($valueGroups)[2] ?? 0;
        if ($pairCount === 2) {
            return ['rank' => 'Two Pair', 'value' => $this->handRanks['Two Pair']];
        }
        
        // 원페어
        if ($pairCount === 1) {
            return ['rank' => 'Pair', 'value' => $this->handRanks['Pair']];
        }
        
        // 하이카드
        return ['rank' => 'High Card', 'value' => $this->handRanks['High Card']];
    }
    
    private function convertToNumericValues(array $values): array {
        $valueMap = [
            '2' => 2, '3' => 3, '4' => 4, '5' => 5, '6' => 6, '7' => 7, '8' => 8, 
            '9' => 9, '10' => 10, 'J' => 11, 'Q' => 12, 'K' => 13, 'A' => 14
        ];
        
        return array_map(fn($value) => $valueMap[$value], $values);
    }
    
    private function isStraight(array $numericValues): bool {
        // A-5 스트레이트 특수 케이스 확인
        if (
            count(array_intersect($numericValues, [2, 3, 4, 5, 14])) === 5 &&
            count(array_unique($numericValues)) === 5
        ) {
            return true;
        }
        
        // 일반 스트레이트 확인
        for ($i = 1; $i < count($numericValues); $i++) {
            if ($numericValues[$i] !== $numericValues[$i - 1] + 1) {
                return false;
            }
        }
        
        return count(array_unique($numericValues)) === 5;
    }
}

// 게임 클래스
class PokerGame {
    private Deck $deck;
    private Hand $playerHand;
    private Hand $computerHand;
    private bool $gameStarted = false;
    private bool $playerDiscarded = false;
    private string $result = '';
    
    public function __construct() {
        $this->deck = new Deck();
        $this->playerHand = new Hand();
        $this->computerHand = new Hand();
    }
    
    public function startGame(): void {
        $this->resetGame();
        $this->deck->shuffle();
        
        // 플레이어와 컴퓨터에게 각각 5장의 카드를 나눠줌
        $this->playerHand->addCards($this->deck->deal(5));
        $this->computerHand->addCards($this->deck->deal(5));
        
        $this->gameStarted = true;
    }
    
    public function resetGame(): void {
        $this->deck->initialize();
        $this->playerHand = new Hand();
        $this->computerHand = new Hand();
        $this->gameStarted = false;
        $this->playerDiscarded = false;
        $this->result = '';
    }
    
    public function playerDiscard(array $indices): void {
        if (!$this->gameStarted || $this->playerDiscarded) {
            return;
        }
        
        $discarded = $this->playerHand->discard($indices);
        $this->playerHand->addCards($this->deck->deal(count($discarded)));
        $this->playerDiscarded = true;
        
        // 컴퓨터는 자동으로 카드를 교체
        $this->computerDiscard();
        
        // 결과 계산
        $this->determineWinner();
    }
    
    private function computerDiscard(): void {
        // 간단한 AI 전략: 페어나 트리플을 유지하고 나머지 카드 교체
        $cards = $this->computerHand->getCards();
        $valueGroups = [];
        
        foreach ($cards as $index => $card) {
            $value = $card->getValue();
            $valueGroups[$value][] = $index;
        }
        
        $keepIndices = [];
        foreach ($valueGroups as $indices) {
            if (count($indices) >= 2) {
                $keepIndices = array_merge($keepIndices, $indices);
            }
        }
        
        $discardIndices = array_diff(range(0, 4), $keepIndices);
        
        if (count($discardIndices) > 0) {
            $discarded = $this->computerHand->discard($discardIndices);
            $this->computerHand->addCards($this->deck->deal(count($discarded)));
        }
    }
    
    private function determineWinner(): void {
        $playerHandRank = $this->playerHand->evaluateHand();
        $computerHandRank = $this->computerHand->evaluateHand();
        
        if ($playerHandRank['value'] > $computerHandRank['value']) {
            $this->result = "플레이어 승리! 플레이어: {$playerHandRank['rank']}, 컴퓨터: {$computerHandRank['rank']}";
        } elseif ($playerHandRank['value'] < $computerHandRank['value']) {
            $this->result = "컴퓨터 승리! 플레이어: {$playerHandRank['rank']}, 컴퓨터: {$computerHandRank['rank']}";
        } else {
            $this->result = "무승부! 양쪽 모두: {$playerHandRank['rank']}";
        }
    }
    
    public function getPlayerCards(): array {
        return $this->playerHand->getCards();
    }
    
    public function getComputerCards(): array {
        return $this->computerHand->getCards();
    }
    
    public function isGameStarted(): bool {
        return $this->gameStarted;
    }
    
    public function hasPlayerDiscarded(): bool {
        return $this->playerDiscarded;
    }
    
    public function getResult(): string {
        return $this->result;
    }
}

// 게임 인스턴스를 세션에 저장
if (!isset($_SESSION['poker_game'])) {
    $_SESSION['poker_game'] = new PokerGame();
}

$game = $_SESSION['poker_game'];

// 액션 처리
$message = '';

if (isset($_POST['action'])) {
    switch ($_POST['action']) {
        case 'start':
            $game->startGame();
            break;
        case 'discard':
            $indices = [];
            if (isset($_POST['discard']) && is_array($_POST['discard'])) {
                $indices = array_keys($_POST['discard']);
            }
            $game->playerDiscard($indices);
            break;
        case 'reset':
            $game->resetGame();
            break;
    }
    
    $_SESSION['poker_game'] = $game;
}

// HTML 출력
?>

<!DOCTYPE html>
<html lang="ko">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PHP 포커 게임</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }
        h1 {
            text-align: center;
        }
        .game-board {
            display: flex;
            flex-direction: column;
            gap: 20px;
        }
        .hand {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
        }
        .card {
            width: 80px;
            height: 120px;
            border: 1px solid #000;
            border-radius: 5px;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            background-color: white;
            position: relative;
        }
        .card.red {
            color: red;
        }
        .card-value {
            font-size: 24px;
            font-weight: bold;
        }
        .card-suit {
            font-size: 28px;
        }
        .actions {
            display: flex;
            gap: 10px;
            margin-top: 20px;
        }
        .actions button {
            padding: 8px 16px;
            cursor: pointer;
        }
        .result {
            margin-top: 20px;
            font-weight: bold;
            font-size: 18px;
            text-align: center;
        }
    </style>
</head>
<body>
    <h1>PHP 포커 게임</h1>
    
    <div class="game-board">
        <h2>컴퓨터의 패</h2>
        <div class="hand">
            <?php if ($game->isGameStarted() && $game->hasPlayerDiscarded()): ?>
                <?php foreach ($game->getComputerCards() as $index => $card): ?>
                    <div class="card <?php echo in_array($card->getSuit(), ['♥', '♦']) ? 'red' : ''; ?>">
                        <div class="card-value"><?php echo htmlspecialchars($card->getValue()); ?></div>
                        <div class="card-suit"><?php echo htmlspecialchars($card->getSuit()); ?></div>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <?php for ($i = 0; $i < 5; $i++): ?>
                    <div class="card" style="background-color: #6c757d; color: white;">
                        <div class="card-value">?</div>
                        <div class="card-suit">?</div>
                    </div>
                <?php endfor; ?>
            <?php endif; ?>
        </div>
        
        <h2>플레이어의 패</h2>
        <form method="post">
            <div class="hand">
                <?php if ($game->isGameStarted()): ?>
                    <?php foreach ($game->getPlayerCards() as $index => $card): ?>
                        <div class="card <?php echo in_array($card->getSuit(), ['♥', '♦']) ? 'red' : ''; ?>">
                            <div class="card-value"><?php echo htmlspecialchars($card->getValue()); ?></div>
                            <div class="card-suit"><?php echo htmlspecialchars($card->getSuit()); ?></div>
                            <?php if (!$game->hasPlayerDiscarded()): ?>
                                <div style="position: absolute; bottom: 5px;">
                                    <input type="checkbox" name="discard[<?php echo $index; ?>]" id="discard-<?php echo $index; ?>">
                                    <label for="discard-<?php echo $index; ?>">교체</label>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <?php for ($i = 0; $i < 5; $i++): ?>
                        <div class="card" style="background-color: #6c757d; color: white;">
                            <div class="card-value">?</div>
                            <div class="card-suit">?</div>
                        </div>
                    <?php endfor; ?>
                <?php endif; ?>
            </div>
            
            <div class="actions">
                <?php if (!$game->isGameStarted()): ?>
                    <button type="submit" name="action" value="start">게임 시작</button>
                <?php elseif (!$game->hasPlayerDiscarded()): ?>
                    <button type="submit" name="action" value="discard">카드 교체</button>
                <?php endif; ?>
                <button type="submit" name="action" value="reset">게임 리셋</button>
            </div>
        </form>
        
        <?php if ($game->hasPlayerDiscarded()): ?>
            <div class="result">
                <?php echo htmlspecialchars($game->getResult()); ?>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>
