<?php
// 한국 장기 게임 구현
// PHP 8.2 기반

// 세션 시작
session_start();

// 클래스 정의
// 장기 말 클래스
class Piece {
    public string $type;     // 말의 종류 (차, 포, 마, 상, 사, 왕, 졸/병)
    public string $team;     // 팀 (한 또는 초)
    public bool $selected = false;  // 선택 여부
    
    public function __construct(string $type, string $team) {
        $this->type = $type;
        $this->team = $team;
    }
    
    // 말의 표시 이름을 반환
    public function getDisplayName(): string {
        return match($this->type) {
            'cha' => $this->team === 'han' ? '차' : '車',
            'po' => $this->team === 'han' ? '포' : '包',
            'ma' => $this->team === 'han' ? '마' : '馬',
            'sang' => $this->team === 'han' ? '상' : '象',
            'sa' => $this->team === 'han' ? '사' : '士',
            'king' => $this->team === 'han' ? '왕' : '楚',
            'jol' => $this->team === 'han' ? '졸' : '兵',
            default => '?'
        };
    }
}

// 장기판 클래스
class Board {
    public array $grid = [];
    public string $currentTurn = 'han';  // 한팀이 첫 턴
    public ?array $selectedPosition = null;
    public array $validMoves = [];
    
    public function __construct() {
        // 9x10 보드 초기화
        for ($y = 0; $y < 10; $y++) {
            $this->grid[$y] = [];
            for ($x = 0; $x < 9; $x++) {
                $this->grid[$y][$x] = null;
            }
        }
        
        // 초기 말 배치
        $this->setupInitialPieces();
    }
    
    // 초기 말 배치 메소드
    private function setupInitialPieces(): void {
        // 한팀(아래) 배치
        // 차
        $this->grid[9][0] = new Piece('cha', 'han');
        $this->grid[9][8] = new Piece('cha', 'han');
        // 포
        $this->grid[7][1] = new Piece('po', 'han');
        $this->grid[7][7] = new Piece('po', 'han');
        // 마
        $this->grid[9][1] = new Piece('ma', 'han');
        $this->grid[9][7] = new Piece('ma', 'han');
        // 상
        $this->grid[9][2] = new Piece('sang', 'han');
        $this->grid[9][6] = new Piece('sang', 'han');
        // 사
        $this->grid[9][3] = new Piece('sa', 'han');
        $this->grid[9][5] = new Piece('sa', 'han');
        // 왕
        $this->grid[9][4] = new Piece('king', 'han');
        // 졸
        $this->grid[6][0] = new Piece('jol', 'han');
        $this->grid[6][2] = new Piece('jol', 'han');
        $this->grid[6][4] = new Piece('jol', 'han');
        $this->grid[6][6] = new Piece('jol', 'han');
        $this->grid[6][8] = new Piece('jol', 'han');
        
        // 초팀(위) 배치
        // 차
        $this->grid[0][0] = new Piece('cha', 'cho');
        $this->grid[0][8] = new Piece('cha', 'cho');
        // 포
        $this->grid[2][1] = new Piece('po', 'cho');
        $this->grid[2][7] = new Piece('po', 'cho');
        // 마
        $this->grid[0][1] = new Piece('ma', 'cho');
        $this->grid[0][7] = new Piece('ma', 'cho');
        // 상
        $this->grid[0][2] = new Piece('sang', 'cho');
        $this->grid[0][6] = new Piece('sang', 'cho');
        // 사
        $this->grid[0][3] = new Piece('sa', 'cho');
        $this->grid[0][5] = new Piece('sa', 'cho');
        // 왕
        $this->grid[0][4] = new Piece('king', 'cho');
        // 병
        $this->grid[3][0] = new Piece('jol', 'cho');
        $this->grid[3][2] = new Piece('jol', 'cho');
        $this->grid[3][4] = new Piece('jol', 'cho');
        $this->grid[3][6] = new Piece('jol', 'cho');
        $this->grid[3][8] = new Piece('jol', 'cho');
    }
    
    // 말 선택 메소드
    public function selectPiece(int $x, int $y): bool {
        $piece = $this->grid[$y][$x] ?? null;
        
        // 해당 위치에 말이 있고, 현재 턴의 팀인 경우만 선택 가능
        if ($piece && $piece->team === $this->currentTurn) {
            // 이전에 선택된 말이 있으면 선택 해제
            if ($this->selectedPosition) {
                $selectedPiece = $this->grid[$this->selectedPosition[1]][$this->selectedPosition[0]] ?? null;
                if ($selectedPiece) {
                    $selectedPiece->selected = false;
                }
            }
            
            // 새 말 선택
            $piece->selected = true;
            $this->selectedPosition = [$x, $y];
            
            // 이동 가능한 위치 계산
            $this->calculateValidMoves($x, $y, $piece);
            
            return true;
        }
        
        return false;
    }
    
    // 말 이동 메소드
    public function movePiece(int $toX, int $toY): bool {
        // 선택된 말이 없으면 이동 불가
        if (!$this->selectedPosition) {
            return false;
        }
        
        $fromX = $this->selectedPosition[0];
        $fromY = $this->selectedPosition[1];
        $piece = $this->grid[$fromY][$fromX];
        
        // 이동 가능한 위치인지 확인
        $canMove = false;
        foreach ($this->validMoves as $move) {
            if ($move[0] === $toX && $move[1] === $toY) {
                $canMove = true;
                break;
            }
        }
        
        if ($canMove) {
            // 말 이동
            $piece->selected = false;
            $this->grid[$toY][$toX] = $piece;
            $this->grid[$fromY][$fromX] = null;
            
            // 턴 변경
            $this->currentTurn = ($this->currentTurn === 'han') ? 'cho' : 'han';
            
            // 선택 초기화
            $this->selectedPosition = null;
            $this->validMoves = [];
            
            return true;
        }
        
        return false;
    }
    
    // 이동 가능한 위치 계산 메소드
    private function calculateValidMoves(int $x, int $y, Piece $piece): void {
        $this->validMoves = [];
        
        // 말 종류에 따른 이동 규칙 적용
        switch ($piece->type) {
            case 'cha': // 차
                $this->addStraightMoves($x, $y, $piece);
                break;
                
            case 'po': // 포
                $this->addCannonMoves($x, $y, $piece);
                break;
                
            case 'ma': // 마
                $this->addHorseMoves($x, $y, $piece);
                break;
                
            case 'sang': // 상
                $this->addElephantMoves($x, $y, $piece);
                break;
                
            case 'sa': // 사
                $this->addGuardMoves($x, $y, $piece);
                break;
                
            case 'king': // 왕
                $this->addKingMoves($x, $y, $piece);
                break;
                
            case 'jol': // 졸/병
                $this->addPawnMoves($x, $y, $piece);
                break;
        }
    }
    
    // 차(직선 이동) 룰
    private function addStraightMoves(int $x, int $y, Piece $piece): void {
        $directions = [[0, 1], [1, 0], [0, -1], [-1, 0]]; // 상하좌우
        
        foreach ($directions as $dir) {
            $dx = $dir[0];
            $dy = $dir[1];
            
            $currX = $x + $dx;
            $currY = $y + $dy;
            
            // 보드 끝까지 이동
            while ($currX >= 0 && $currX < 9 && $currY >= 0 && $currY < 10) {
                $targetPiece = $this->grid[$currY][$currX] ?? null;
                
                if (!$targetPiece) {
                    // 빈 칸이면 이동 가능
                    $this->validMoves[] = [$currX, $currY];
                } else if ($targetPiece->team !== $piece->team) {
                    // 상대편 말이면 잡을 수 있고 더 이상 이동 불가
                    $this->validMoves[] = [$currX, $currY];
                    break;
                } else {
                    // 같은 편 말이면 이동 불가
                    break;
                }
                
                $currX += $dx;
                $currY += $dy;
            }
        }
        
        // 궁성 대각선 이동 (차는 궁성에서 대각선으로도 이동 가능)
        $palaceMoves = $this->getPalaceDiagonalMoves($x, $y, $piece);
        foreach ($palaceMoves as $move) {
            $this->validMoves[] = $move;
        }
    }
    
    // 포(대포) 룰
    private function addCannonMoves(int $x, int $y, Piece $piece): void {
        $directions = [[0, 1], [1, 0], [0, -1], [-1, 0]]; // 상하좌우
        
        foreach ($directions as $dir) {
            $dx = $dir[0];
            $dy = $dir[1];
            
            $jumped = false;  // 다른 말을 뛰어넘었는지 여부
            $currX = $x + $dx;
            $currY = $y + $dy;
            
            while ($currX >= 0 && $currX < 9 && $currY >= 0 && $currY < 10) {
                $targetPiece = $this->grid[$currY][$currX] ?? null;
                
                if (!$jumped) {
                    // 아직 말을 뛰어넘지 않았을 경우
                    if ($targetPiece) {
                        // 말을 만나면 뛰어넘음
                        $jumped = true;
                    }
                } else {
                    // 이미 말을 뛰어넘은 경우
                    if (!$targetPiece) {
                        // 빈 칸이면 이동 가능
                        $this->validMoves[] = [$currX, $currY];
                    } else if ($targetPiece->team !== $piece->team) {
                        // 상대편 말이면 잡을 수 있고 더 이상 이동 불가
                        $this->validMoves[] = [$currX, $currY];
                        break;
                    } else {
                        // 같은 편 말이면 이동 불가
                        break;
                    }
                }
                
                $currX += $dx;
                $currY += $dy;
            }
        }
    }
    
    // 마(말) 룰
    private function addHorseMoves(int $x, int $y, Piece $piece): void {
        // 마는 직선으로 한 칸 이동 후, 대각선으로 한 칸 이동
        $directions = [
            [0, -1, -1, -1], [0, -1, 1, -1], // 위로 한 칸, 왼쪽/오른쪽 대각선
            [0, 1, -1, 1], [0, 1, 1, 1],     // 아래로 한 칸, 왼쪽/오른쪽 대각선
            [-1, 0, -1, -1], [-1, 0, -1, 1], // 왼쪽으로 한 칸, 위/아래 대각선
            [1, 0, 1, -1], [1, 0, 1, 1]      // 오른쪽으로 한 칸, 위/아래 대각선
        ];
        
        foreach ($directions as $dir) {
            $dx1 = $dir[0]; // 첫 번째 이동 x 방향
            $dy1 = $dir[1]; // 첫 번째 이동 y 방향
            $dx2 = $dir[2]; // 두 번째 이동 x 방향
            $dy2 = $dir[3]; // 두 번째 이동 y 방향
            
            $midX = $x + $dx1;
            $midY = $y + $dy1;
            
            // 첫 번째 이동 위치가 보드 안에 있고, 해당 위치가 비어있는지 확인 (장애물 체크)
            if ($midX >= 0 && $midX < 9 && $midY >= 0 && $midY < 10 && !$this->grid[$midY][$midX]) {
                $toX = $midX + $dx2;
                $toY = $midY + $dy2;
                
                // 최종 이동 위치가 보드 안에 있는지 확인
                if ($toX >= 0 && $toX < 9 && $toY >= 0 && $toY < 10) {
                    $targetPiece = $this->grid[$toY][$toX] ?? null;
                    
                    // 빈 칸이거나 상대편 말이면 이동 가능
                    if (!$targetPiece || $targetPiece->team !== $piece->team) {
                        $this->validMoves[] = [$toX, $toY];
                    }
                }
            }
        }
    }
    
    // 상(코끼리) 룰
    private function addElephantMoves(int $x, int $y, Piece $piece): void {
        // 상은 대각선으로 두 칸 이동 (총 3칸 이동)
        $directions = [
            [-1, -1, -2, -2], [1, -1, 2, -2], // 왼쪽 위, 오른쪽 위 대각선
            [-1, 1, -2, 2], [1, 1, 2, 2]      // 왼쪽 아래, 오른쪽 아래 대각선
        ];
        
        foreach ($directions as $dir) {
            $dx1 = $dir[0]; // 첫 번째 이동 x 방향
            $dy1 = $dir[1]; // 첫 번째 이동 y 방향
            $dx2 = $dir[2]; // 최종 x 방향
            $dy2 = $dir[3]; // 최종 y 방향
            
            $midX = $x + $dx1;
            $midY = $y + $dy1;
            
            // 첫 번째 이동 위치가 보드 안에 있고, 해당 위치가 비어있는지 확인 (장애물 체크)
            if ($midX >= 0 && $midX < 9 && $midY >= 0 && $midY < 10 && !$this->grid[$midY][$midX]) {
                $toX = $x + $dx2;
                $toY = $y + $dy2;
                
                // 최종 이동 위치가 보드 안에 있는지 확인
                if ($toX >= 0 && $toX < 9 && $toY >= 0 && $toY < 10) {
                    // 중간 위치도 확인 (두 번째 장애물 체크)
                    $mid2X = $x + $dx1 * 2;
                    $mid2Y = $y + $dy1 * 2;
                    
                    if ($mid2X >= 0 && $mid2X < 9 && $mid2Y >= 0 && $mid2Y < 10 && !$this->grid[$mid2Y][$mid2X]) {
                        $targetPiece = $this->grid[$toY][$toX] ?? null;
                        
                        // 빈 칸이거나 상대편 말이면 이동 가능
                        if (!$targetPiece || $targetPiece->team !== $piece->team) {
                            $this->validMoves[] = [$toX, $toY];
                        }
                    }
                }
            }
        }
    }
    
    // 사(신하) 룰
    private function addGuardMoves(int $x, int $y, Piece $piece): void {
        // 사는 궁성 안에서만 대각선 포함 한 칸씩 이동 가능
        $palace = ($piece->team === 'han') ? [
            [3, 7], [4, 7], [5, 7],
            [3, 8], [4, 8], [5, 8],
            [3, 9], [4, 9], [5, 9]
        ] : [
            [3, 0], [4, 0], [5, 0],
            [3, 1], [4, 1], [5, 1],
            [3, 2], [4, 2], [5, 2]
        ];
        
        // 상하좌우 및 대각선 방향
        $directions = [[0, 1], [1, 0], [0, -1], [-1, 0], [1, 1], [1, -1], [-1, 1], [-1, -1]];
        
        foreach ($directions as $dir) {
            $toX = $x + $dir[0];
            $toY = $y + $dir[1];
            
            // 이동 위치가 궁성 안에 있는지 확인
            $inPalace = false;
            foreach ($palace as $pos) {
                if ($pos[0] === $toX && $pos[1] === $toY) {
                    $inPalace = true;
                    break;
                }
            }
            
            if ($inPalace) {
                $targetPiece = $this->grid[$toY][$toX] ?? null;
                
                // 빈 칸이거나 상대편 말이면 이동 가능
                if (!$targetPiece || $targetPiece->team !== $piece->team) {
                    $this->validMoves[] = [$toX, $toY];
                }
            }
        }
    }
    
    // 왕(장군) 룰
    private function addKingMoves(int $x, int $y, Piece $piece): void {
        // 왕은 궁성 안에서만 상하좌우 및 대각선으로 한 칸씩 이동 가능
        // 사(신하)와 동일한 이동 패턴
        $this->addGuardMoves($x, $y, $piece);
    }
    
    // 졸/병(보병) 룰
    private function addPawnMoves(int $x, int $y, Piece $piece): void {
        // 졸/병은 전진만 가능하지만, 상대 진영에 들어가면 좌우도 가능
        $directions = [];
        
        if ($piece->team === 'han') {
            // 한팀은 위로 이동
            $directions[] = [0, -1]; // 위로
            
            // 상대 진영에 있으면 좌우도 가능
            if ($y <= 4) {
                $directions[] = [-1, 0]; // 왼쪽
                $directions[] = [1, 0];  // 오른쪽
            }
        } else {
            // 초팀은 아래로 이동
            $directions[] = [0, 1]; // 아래로
            
            // 상대 진영에 있으면 좌우도 가능
            if ($y >= 5) {
                $directions[] = [-1, 0]; // 왼쪽
                $directions[] = [1, 0];  // 오른쪽
            }
        }
        
        foreach ($directions as $dir) {
            $toX = $x + $dir[0];
            $toY = $y + $dir[1];
            
            // 이동 위치가 보드 안에 있는지 확인
            if ($toX >= 0 && $toX < 9 && $toY >= 0 && $toY < 10) {
                $targetPiece = $this->grid[$toY][$toX] ?? null;
                
                // 빈 칸이거나 상대편 말이면 이동 가능
                if (!$targetPiece || $targetPiece->team !== $piece->team) {
                    $this->validMoves[] = [$toX, $toY];
                }
            }
        }
    }
    
    // 궁성 대각선 이동 확인 (차, 왕 등이 사용)
    private function getPalaceDiagonalMoves(int $x, int $y, Piece $piece): array {
        $moves = [];
        
        // 궁성 위치 정의
        $palace = ($piece->team === 'han') ? [
            [3, 7], [4, 7], [5, 7],
            [3, 8], [4, 8], [5, 8],
            [3, 9], [4, 9], [5, 9]
        ] : [
            [3, 0], [4, 0], [5, 0],
            [3, 1], [4, 1], [5, 1],
            [3, 2], [4, 2], [5, 2]
        ];
        
        // 현재 위치가 궁성 안에 있는지 확인
        $inPalace = false;
        foreach ($palace as $pos) {
            if ($pos[0] === $x && $pos[1] === $y) {
                $inPalace = true;
                break;
            }
        }
        
        if ($inPalace) {
            // 궁성 안에 있으면 대각선 이동 가능
            $diagonals = [[1, 1], [1, -1], [-1, 1], [-1, -1]];
            
            foreach ($diagonals as $dir) {
                $toX = $x + $dir[0];
                $toY = $y + $dir[1];
                
                // 대각선 이동 위치가 궁성 안에 있는지 확인
                $targetInPalace = false;
                foreach ($palace as $pos) {
                    if ($pos[0] === $toX && $pos[1] === $toY) {
                        $targetInPalace = true;
                        break;
                    }
                }
                
                if ($targetInPalace) {
                    $targetPiece = $this->grid[$toY][$toX] ?? null;
                    
                    // 빈 칸이거나 상대편 말이면 이동 가능
                    if (!$targetPiece || $targetPiece->team !== $piece->team) {
                        $moves[] = [$toX, $toY];
                    }
                }
            }
        }
        
        return $moves;
    }
    
    // 승자 확인 메소드
    public function checkWinner(): ?string {
        $hanKingAlive = false;
        $choKingAlive = false;
        
        // 보드 전체를 검사하여 양 팀의 왕이 살아있는지 확인
        for ($y = 0; $y < 10; $y++) {
            for ($x = 0; $x < 9; $x++) {
                $piece = $this->grid[$y][$x] ?? null;
                if ($piece && $piece->type === 'king') {
                    if ($piece->team === 'han') {
                        $hanKingAlive = true;
                    } else {
                        $choKingAlive = true;
                    }
                }
            }
        }
        
        // 승자 결정
        if (!$hanKingAlive) {
            return 'cho';
        } else if (!$choKingAlive) {
            return 'han';
        }
        
        return null; // 아직 승자 없음
    }
}

// 게임 상태 관리
if (!isset($_SESSION['janggi_board'])) {
    $_SESSION['janggi_board'] = new Board();
}

$board = $_SESSION['janggi_board'];

// 액션 처리
if (isset($_GET['action'])) {
    $action = $_GET['action'];
    
    if ($action === 'select' && isset($_GET['x']) && isset($_GET['y'])) {
        $x = (int)$_GET['x'];
        $y = (int)$_GET['y'];
        
        // 말 선택 시도
        if (!$board->selectPiece($x, $y)) {
            // 이미 선택된 말이 있고, 이동 가능한 위치를 클릭한 경우 말 이동
            if ($board->selectedPosition) {
                $board->movePiece($x, $y);
            }
        }
    } else if ($action === 'reset') {
        // 게임 재시작
        $_SESSION['janggi_board'] = new Board();
        $board = $_SESSION['janggi_board'];
    }
    
    // 세션에 보드 상태 저장
    $_SESSION['janggi_board'] = $board;
    
    // AJAX 요청인 경우 JSON 응답 반환
    if (isset($_GET['ajax'])) {
        header('Content-Type: application/json');
        
        $response = [
            'grid' => [],
            'currentTurn' => $board->currentTurn,
            'selectedPosition' => $board->selectedPosition,
            'validMoves' => $board->validMoves,
            'winner' => $board->checkWinner()
        ];
        
        // 그리드 데이터 변환
        for ($y = 0; $y < 10; $y++) {
            $response['grid'][$y] = [];
            for ($x = 0; $x < 9; $x++) {
                $piece = $board->grid[$y][$x];
                if ($piece) {
                    $response['grid'][$y][$x] = [
                        'type' => $piece->type,
                        'team' => $piece->team,
                        'selected' => $piece->selected,
                        'displayName' => $piece->getDisplayName()
                    ];
                } else {
                    $response['grid'][$y][$x] = null;
                }
            }
        }
        
        echo json_encode($response);
        exit;
    }
    
    // 일반 요청인 경우 같은 페이지로 리다이렉트하여 새로고침
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit;
}
?>


<!DOCTYPE html>
<html lang="ko">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>한국 장기 게임</title>
    <style>
        /* 기본 스타일 */
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'Malgun Gothic', 'Nanum Gothic', 'Dotum', sans-serif;
            background-color: #f8f1e5;
            color: #333;
            padding: 20px;
            display: flex;
            flex-direction: column;
            align-items: center;
            line-height: 1.6;
        }
        
        .container {
            max-width: 800px;
            width: 100%;
        }
        
        /* 제목 스타일 */
        h1 {
            color: #8B4513;
            text-align: center;
            margin-bottom: 20px;
            font-size: 28px;
            font-weight: bold;
            text-shadow: 1px 1px 2px rgba(0,0,0,0.1);
        }
        
        /* 게임 정보 영역 */
        .game-info {
            display: flex;
            flex-direction: column;
            margin-bottom: 20px;
            padding: 10px;
            background-color: #fff;
            border-radius: 8px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        
        .turn-indicator {
            padding: 10px;
            border-radius: 5px;
            font-weight: bold;
            text-align: center;
            margin-bottom: 10px;
            font-size: 16px;
        }
        
        .turn-han {
            background-color: #FFD700;
            color: #333;
        }
        
        .turn-cho {
            background-color: #4169E1;
            color: white;
        }
        
        .message {
            padding: 10px;
            border-radius: 5px;
            background-color: #f0f0f0;
            min-height: 20px;
            text-align: center;
        }
        
        /* 장기판 스타일 */
        .board-container {
            position: relative;
            margin: 0 auto 20px;
            padding: 10px;
            background-color: #d9bfa3;
            border-radius: 10px;
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        }
        
        .board {
            background-color: #DEB887;
            border: 4px solid #8B4513;
            border-radius: 5px;
            margin: 0 auto;
            position: relative;
            overflow: hidden;
        }
        
        .grid {
            display: grid;
            grid-template-columns: repeat(9, 1fr);
            grid-template-rows: repeat(10, 1fr);
            gap: 1px;
            background-color: #8B4513;
        }
        
        .cell {
            width: 40px;
            height: 40px;
            background-color: #DEB887;
            display: flex;
            justify-content: center;
            align-items: center;
            position: relative;
            cursor: pointer;
        }
        
        .cell:hover {
            background-color: rgba(255, 215, 0, 0.3);
        }
        
        .cell.valid-move {
            background-color: rgba(0, 255, 0, 0.3);
        }
        
        /* 장기 말 스타일 */
        .piece {
            width: 36px;
            height: 36px;
            border-radius: 50%;
            display: flex;
            justify-content: center;
            align-items: center;
            cursor: pointer;
            font-weight: bold;
            transition: all 0.3s ease;
            box-shadow: 0 2px 4px rgba(0,0,0,0.2);
        }
        
        .piece.han {
            background-color: #FFD700;
            color: #333;
        }
        
        .piece.cho {
            background-color: #4169E1;
            color: white;
        }
        
        .piece.selected {
            box-shadow: 0 0 0 3px red, 0 0 10px 3px rgba(255, 0, 0, 0.5);
            transform: scale(1.1);
        }
        
        /* 궁성 스타일 */
        .palace {
            position: absolute;
            width: 122px;
            height: 122px;
            background-color: rgba(255, 215, 0, 0.1);
            border: 1px dashed #8B4513;
            z-index: 0;
        }
        
        .palace-cho {
            top: 0;
            left: 120px;
        }
        
        .palace-han {
            bottom: 0;
            left: 120px;
        }
        
        /* 궁성 대각선 */
        .diagonal {
            position: absolute;
            width: 170px;
            height: 1px;
            background-color: #8B4513;
            transform-origin: center;
            z-index: 0;
        }
        
        .diagonal-1-han {
            bottom: 60px;
            left: 120px;
            transform: rotate(45deg);
        }
        
        .diagonal-2-han {
            bottom: 60px;
            left: 120px;
            transform: rotate(-45deg);
        }
        
        .diagonal-1-cho {
            top: 60px;
            left: 120px;
            transform: rotate(45deg);
        }
        
        .diagonal-2-cho {
            top: 60px;
            left: 120px;
            transform: rotate(-45deg);
        }
        
        /* 컨트롤 버튼 */
        .controls {
            display: flex;
            justify-content: center;
            margin: 20px 0;
            flex-wrap: wrap;
        }
        
        .btn {
            padding: 10px 20px;
            background-color: #8B4513;
            color: white;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 16px;
            font-weight: bold;
            margin: 5px;
            text-decoration: none;
            transition: background-color 0.3s;
        }
        
        .btn:hover {
            background-color: #A0522D;
        }
        
        .btn.active {
            background-color: #006400;
        }
        
        /* 승리 모달 */
        .winner-modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.7);
            z-index: 100;
            justify-content: center;
            align-items: center;
            animation: fadeIn 0.5s ease;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        .winner-content {
            background-color: white;
            padding: 30px;
            border-radius: 10px;
            text-align: center;
            box-shadow: 0 5px 15px rgba(0,0,0,0.3);
            max-width: 90%;
            width: 400px;
        }
        
        .winner-message {
            font-size: 24px;
            margin-bottom: 20px;
            font-weight: bold;
            color: #8B4513;
        }
        
        .winner-message button {
            margin-top: 20px;
        }
        
        /* 게임 가이드 */
        .game-guide {
            background-color: white;
            border-radius: 8px;
            padding: 15px;
            margin-top: 20px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        
        .guide-title {
            font-weight: bold;
            font-size: 18px;
            color: #8B4513;
            margin-bottom: 10px;
            border-bottom: 1px solid #ddd;
            padding-bottom: 5px;
        }
        
        .guide-section {
            margin-bottom: 15px;
        }
        
        .guide-subtitle {
            font-weight: bold;
            margin: 10px 0 5px;
            color: #333;
        }
        
        .guide-list {
            list-style-type: disc;
            padding-left: 20px;
            margin: 5px 0;
        }
        
        .guide-list li {
            margin-bottom: 5px;
        }
        
        /* 반응형 디자인 */
        @media screen and (max-width: 600px) {
            .cell {
                width: 32px;
                height: 32px;
            }
            
            .piece {
                width: 28px;
                height: 28px;
                font-size: 14px;
            }
            
            .palace-han, .palace-cho {
                width: 98px;
                height: 98px;
                left: 96px;
            }
            
            .diagonal {
                width: 136px;
            }
            
            .diagonal-1-han, .diagonal-2-han {
                bottom: 48px;
                left: 96px;
            }
            
            .diagonal-1-cho, .diagonal-2-cho {
                top: 48px;
                left: 96px;
            }
            
            h1 {
                font-size: 24px;
            }
            
            .guide-title {
                font-size: 16px;
            }
        }
        
        @media screen and (max-width: 400px) {
            .cell {
                width: 28px;
                height: 28px;
            }
            
            .piece {
                width: 24px;
                height: 24px;
                font-size: 12px;
            }
            
            .palace-han, .palace-cho {
                width: 86px;
                height: 86px;
                left: 84px;
            }
            
            .diagonal {
                width: 120px;
            }
            
            .diagonal-1-han, .diagonal-2-han {
                bottom: 42px;
                left: 84px;
            }
            
            .diagonal-1-cho, .diagonal-2-cho {
                top: 42px;
                left: 84px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>한국 장기 게임</h1>
        
        <div class="game-info">
            <div class="turn-indicator <?php echo $board->currentTurn === 'han' ? 'turn-han' : 'turn-cho'; ?>">
                현재 턴: <?php echo $board->currentTurn === 'han' ? '한팀(황색)' : '초팀(청색)'; ?>
                <?php if (isset($board->gameMode) && $board->gameMode === 'pvc' && $board->currentTurn === 'cho'): ?>
                    (컴퓨터 차례)
                <?php endif; ?>
            </div>
            <div class="message">
                <?php echo isset($board->message) ? $board->message : ''; ?>
            </div>
        </div>
        
        <div class="board-container">
            <div class="board">
                <!-- 궁성 표시 -->
                <div class="palace palace-cho"></div>
                <div class="palace palace-han"></div>
                
                <!-- 궁성 대각선 -->
                <div class="diagonal diagonal-1-cho"></div>
                <div class="diagonal diagonal-2-cho"></div>
                <div class="diagonal diagonal-1-han"></div>
                <div class="diagonal diagonal-2-han"></div>
                
                <div class="grid">
                    <?php for ($y = 0; $y < 10; $y++): ?>
                        <?php for ($x = 0; $x < 9; $x++): ?>
                            <?php
                            $piece = $board->grid[$y][$x] ?? null;
                            $isValidMove = false;
                            foreach ($board->validMoves as $move) {
                                if ($move[0] === $x && $move[1] === $y) {
                                    $isValidMove = true;
                                    break;
                                }
                            }
                            ?>
                            <div class="cell <?php echo $isValidMove ? 'valid-move' : ''; ?>" 
                                 data-x="<?php echo $x; ?>" 
                                 data-y="<?php echo $y; ?>">
                                <?php if ($piece): ?>
                                    <div class="piece <?php echo $piece->team; ?> <?php echo $piece->selected ? 'selected' : ''; ?>">
                                        <?php echo $piece->getDisplayName(); ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endfor; ?>
                    <?php endfor; ?>
                </div>
            </div>
        </div>
        
        <div class="controls">
            <a href="?action=setmode&mode=pvc" class="btn <?php echo (isset($board->gameMode) && $board->gameMode === 'pvc') ? 'active' : ''; ?>">컴퓨터와 대결</a>
            <a href="?action=setmode&mode=pvp" class="btn <?php echo (isset($board->gameMode) && $board->gameMode === 'pvp') ? 'active' : ''; ?>">2인 플레이</a>
            <a href="?action=reset&mode=<?php echo isset($board->gameMode) ? $board->gameMode : 'pvc'; ?>" class="btn">게임 재시작</a>
        </div>
        
        <div class="game-guide">
            <div class="guide-title">장기 게임 가이드</div>
            
            <div class="guide-section">
                <div class="guide-subtitle">게임 방법</div>
                <ul class="guide-list">
                    <li>자신의 말을 클릭하여 선택합니다.</li>
                    <li>초록색으로 표시된 위치를 클릭하여 말을 이동합니다.</li>
                    <li>상대방의 왕(楚/왕)을 잡으면 승리합니다.</li>
                </ul>
            </div>
            
            <div class="guide-section">
                <div class="guide-subtitle">말의 이동 규칙</div>
                <ul class="guide-list">
                    <li><strong>차(車)</strong>: 상하좌우 직선으로 이동 가능, 궁성에서는 대각선도 이동 가능</li>
                    <li><strong>포(包)</strong>: 다른 말을 뛰어넘어 이동 또는 공격 가능 (상대방 말도 잡을 수 있음)</li>
                    <li><strong>마(馬)</strong>: 직선으로 1칸, 대각선으로 1칸 이동 (장애물에 막히면 이동 불가)</li>
                    <li><strong>상(象)</strong>: 대각선 방향으로 2칸씩 이동 (장애물에 막히면 이동 불가)</li>
                    <li><strong>사(士)</strong>: 궁성 안에서만 1칸 이동 (대각선 포함)</li>
                    <li><strong>왕(楚/왕)</strong>: 궁성 안에서만 1칸 이동 (대각선 포함)</li>
                    <li><strong>졸/병(兵/졸)</strong>: 앞으로 1칸, 상대 진영에서는 좌우로도 1칸 이동 가능</li>
                </ul>
            </div>
            
            <div class="guide-section">
                <div class="guide-subtitle">게임 모드</div>
                <ul class="guide-list">
                    <li><strong>컴퓨터와 대결</strong>: 사용자가 한팀(황색)을 조작하고, 컴퓨터가 초팀(청색)을 조작합니다.</li>
                    <li><strong>2인 플레이</strong>: 두 사람이 번갈아가며 한팀과 초팀을 조작합니다.</li>
                </ul>
            </div>
        </div>
    </div>
    
    <!-- 승리 모달 -->
    <div id="winnerModal" class="winner-modal">
        <div class="winner-content">
            <div id="winnerMessage" class="winner-message"></div>
            <a href="?action=reset&mode=<?php echo isset($board->gameMode) ? $board->gameMode : 'pvc'; ?>" class="btn">새 게임 시작</a>
        </div>
    </div>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // 초기 상태 확인
            checkWinner('<?php echo $board->checkWinner(); ?>');
            
            // 컴퓨터 턴이면 컴퓨터 이동 트리거
            <?php if (isset($board->gameMode) && $board->gameMode === 'pvc' && $board->currentTurn === 'cho' && !$board->checkWinner()): ?>
            setTimeout(function() {
                window.location.href = '?action=select&x=0&y=0';
            }, 1000);
            <?php endif; ?>
            
            // 셀 클릭 이벤트 등록
            const cells = document.querySelectorAll('.cell');
            cells.forEach(cell => {
                cell.addEventListener('click', function() {
                    const x = this.getAttribute('data-x');
                    const y = this.getAttribute('data-y');
                    
                    // 컴퓨터 턴이면 클릭 무시
                    <?php if (isset($board->gameMode)): ?>
                    if ('<?php echo $board->gameMode; ?>' === 'pvc' && '<?php echo $board->currentTurn; ?>' === 'cho') {
                        return;
                    }
                    <?php endif; ?>
                    
                    // 서버에 액션 요청
                    window.location.href = `?action=select&x=${x}&y=${y}`;
                });
            });
            
            // 승자 확인 및 승리 모달 표시
            function checkWinner(winner) {
                if (winner) {
                    const winnerModal = document.getElementById('winnerModal');
                    const winnerMessage = document.getElementById('winnerMessage');
                    
                    winnerMessage.textContent = winner === 'han' ? '한팀(황색) 승리!' : '초팀(청색) 승리!';
                    winnerModal.style.display = 'flex';
                }
            }
        });
    </script>
    </script>
</body>
</html>