<?php
// index.php - 메인 게임 파일

session_start();

// 게임 초기화 또는 게임 상태 로드
if (!isset($_SESSION['chess_game']) || isset($_POST['reset'])) {
    initializeGame();
}

// 사용자 이동 처리
if (isset($_POST['from']) && isset($_POST['to'])) {
    $fromRow = intval($_POST['from']['row']);
    $fromCol = intval($_POST['from']['col']);
    $toRow = intval($_POST['to']['row']);
    $toCol = intval($_POST['to']['col']);
    
    makePlayerMove($fromRow, $fromCol, $toRow, $toCol);
    
    // 플레이어 이동 후 게임 오버 체크
    if (!$_SESSION['chess_game']['game_over']) {
        // 컴퓨터 이동 실행
        makeComputerMove();
    }
}

/**
 * 게임 초기화 함수
 */
function initializeGame() {
    // 초기 체스 보드 설정
    $initialBoard = [
        ['r', 'n', 'b', 'q', 'k', 'b', 'n', 'r'],
        ['p', 'p', 'p', 'p', 'p', 'p', 'p', 'p'],
        [null, null, null, null, null, null, null, null],
        [null, null, null, null, null, null, null, null],
        [null, null, null, null, null, null, null, null],
        [null, null, null, null, null, null, null, null],
        ['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P'],
        ['R', 'N', 'B', 'Q', 'K', 'B', 'N', 'R']
    ];
    
    $_SESSION['chess_game'] = [
        'board' => $initialBoard,
        'turn' => 'white', // white 또는 black
        'status' => '게임 시작! 흰색(대문자) 차례입니다.',
        'game_over' => false,
        'captured_pieces' => [
            'white' => [], // 흰색이 잡은 말
            'black' => []  // 검은색이 잡은 말
        ],
        'selected_piece' => null,
        'valid_moves' => []
    ];
}

/**
 * 플레이어 이동 처리 함수
 */
function makePlayerMove($fromRow, $fromCol, $toRow, $toCol) {
    $game = &$_SESSION['chess_game'];
    $board = &$game['board'];
    
    if ($game['game_over'] || $game['turn'] !== 'white') {
        return false;
    }
    
    $piece = $board[$fromRow][$fromCol];
    
    // 유효한 이동인지 확인
    $validMoves = calculateValidMoves($fromRow, $fromCol, $piece);
    $isValidMove = false;
    
    foreach ($validMoves as $move) {
        if ($move['row'] === $toRow && $move['col'] === $toCol) {
            $isValidMove = true;
            break;
        }
    }
    
    if (!$isValidMove) {
        return false;
    }
    
    // 말 이동 전에 캡처 확인
    if ($board[$toRow][$toCol] !== null) {
        $capturedPiece = $board[$toRow][$toCol];
        if ($capturedPiece === strtolower($capturedPiece)) { // 검은색 말을 캡처
            $game['captured_pieces']['white'][] = $capturedPiece;
            
            // 킹 캡처 체크
            if ($capturedPiece === 'k') {
                $game['status'] = '게임 종료! 플레이어(흰색)의 승리입니다!';
                $game['game_over'] = true;
            }
        }
    }
    
    // 말 이동
    $board[$toRow][$toCol] = $board[$fromRow][$fromCol];
    $board[$fromRow][$fromCol] = null;
    
    // 턴 변경
    if (!$game['game_over']) {
        $game['turn'] = 'black';
        $game['status'] = '컴퓨터(검은색)의 차례입니다...';
    }
    
    return true;
}

/**
 * 컴퓨터 이동 처리 함수
 */
function makeComputerMove() {
    $game = &$_SESSION['chess_game'];
    $board = &$game['board'];
    
    if ($game['game_over'] || $game['turn'] !== 'black') {
        return false;
    }
    
    // 모든 검은색 말 위치 찾기
    $blackPieces = [];
    foreach ($board as $rowIndex => $row) {
        foreach ($row as $colIndex => $piece) {
            if ($piece !== null && $piece === strtolower($piece)) {
                $blackPieces[] = [
                    'piece' => $piece,
                    'row' => $rowIndex,
                    'col' => $colIndex
                ];
            }
        }
    }
    
    // 가능한 모든 이동 수집
    $allPossibleMoves = [];
    foreach ($blackPieces as $pieceInfo) {
        $piece = $pieceInfo['piece'];
        $row = $pieceInfo['row'];
        $col = $pieceInfo['col'];
        
        $moves = calculateValidMoves($row, $col, $piece);
        
        foreach ($moves as $move) {
            $allPossibleMoves[] = [
                'from' => ['row' => $row, 'col' => $col],
                'to' => $move,
                'piece' => $piece
            ];
        }
    }
    
    // 컴퓨터가 이동할 수 없는 경우 게임 종료
    if (empty($allPossibleMoves)) {
        $game['status'] = '컴퓨터가 움직일 수 있는 말이 없습니다. 플레이어의 승리입니다!';
        $game['game_over'] = true;
        return false;
    }
    
    // 랜덤으로 이동 선택
    $randomMove = $allPossibleMoves[array_rand($allPossibleMoves)];
    $fromRow = $randomMove['from']['row'];
    $fromCol = $randomMove['from']['col'];
    $toRow = $randomMove['to']['row'];
    $toCol = $randomMove['to']['col'];
    $piece = $randomMove['piece'];
    
    // 말 이동 전에 캡처 확인
    if ($board[$toRow][$toCol] !== null) {
        $capturedPiece = $board[$toRow][$toCol];
        if ($capturedPiece === strtoupper($capturedPiece)) { // 흰색 말을 캡처
            $game['captured_pieces']['black'][] = $capturedPiece;
            
            // 킹 캡처 체크
            if ($capturedPiece === 'K') {
                $game['status'] = '게임 종료! 컴퓨터(검은색)의 승리입니다!';
                $game['game_over'] = true;
            }
        }
    }
    
    // 말 이동
    $board[$toRow][$toCol] = $board[$fromRow][$fromCol];
    $board[$fromRow][$fromCol] = null;
    
    // 턴 변경
    if (!$game['game_over']) {
        $game['turn'] = 'white';
        $game['status'] = '플레이어(흰색)의 차례입니다.';
    }
    
    return true;
}

/**
 * 유효한 이동 위치 계산 함수
 */
function calculateValidMoves($row, $col, $piece) {
    $game = $_SESSION['chess_game'];
    $board = $game['board'];
    $moves = [];
    
    if ($piece === null) {
        return $moves;
    }
    
    $pieceType = strtolower($piece);
    $isWhite = $piece === strtoupper($piece);
    
    // 폰 이동
    if ($pieceType === 'p') {
        $direction = $isWhite ? -1 : 1;
        $startRow = $isWhite ? 6 : 1;
        
        // 전진
        if ($row + $direction >= 0 && $row + $direction < 8 && $board[$row + $direction][$col] === null) {
            $moves[] = ['row' => $row + $direction, 'col' => $col];
            
            // 처음 이동시 2칸 가능
            if ($row === $startRow && $board[$row + 2 * $direction][$col] === null) {
                $moves[] = ['row' => $row + 2 * $direction, 'col' => $col];
            }
        }
        
        // 대각선 캡처
        $captureDirections = [
            ['row' => $direction, 'col' => -1],
            ['row' => $direction, 'col' => 1]
        ];
        
        foreach ($captureDirections as $dir) {
            $newRow = $row + $dir['row'];
            $newCol = $col + $dir['col'];
            
            if ($newRow >= 0 && $newRow < 8 && $newCol >= 0 && $newCol < 8) {
                if ($board[$newRow][$newCol] !== null && isOpponent($piece, $board[$newRow][$newCol])) {
                    $moves[] = ['row' => $newRow, 'col' => $newCol];
                }
            }
        }
    }
    
    // 룩 이동
    if ($pieceType === 'r') {
        $directions = [
            ['row' => -1, 'col' => 0],  // 위
            ['row' => 1, 'col' => 0],   // 아래
            ['row' => 0, 'col' => -1],  // 왼쪽
            ['row' => 0, 'col' => 1]    // 오른쪽
        ];
        
        foreach ($directions as $dir) {
            $moves = array_merge($moves, getLineMoves($board, $row, $col, $dir['row'], $dir['col'], $piece));
        }
    }
    
    // 나이트 이동
    if ($pieceType === 'n') {
        $knightMoves = [
            ['row' => -2, 'col' => -1],
            ['row' => -2, 'col' => 1],
            ['row' => -1, 'col' => -2],
            ['row' => -1, 'col' => 2],
            ['row' => 1, 'col' => -2],
            ['row' => 1, 'col' => 2],
            ['row' => 2, 'col' => -1],
            ['row' => 2, 'col' => 1]
        ];
        
        foreach ($knightMoves as $move) {
            $newRow = $row + $move['row'];
            $newCol = $col + $move['col'];
            
            if ($newRow >= 0 && $newRow < 8 && $newCol >= 0 && $newCol < 8) {
                if ($board[$newRow][$newCol] === null || isOpponent($piece, $board[$newRow][$newCol])) {
                    $moves[] = ['row' => $newRow, 'col' => $newCol];
                }
            }
        }
    }
    
    // 비숍 이동
    if ($pieceType === 'b') {
        $directions = [
            ['row' => -1, 'col' => -1],  // 왼쪽 위
            ['row' => -1, 'col' => 1],   // 오른쪽 위
            ['row' => 1, 'col' => -1],   // 왼쪽 아래
            ['row' => 1, 'col' => 1]     // 오른쪽 아래
        ];
        
        foreach ($directions as $dir) {
            $moves = array_merge($moves, getLineMoves($board, $row, $col, $dir['row'], $dir['col'], $piece));
        }
    }
    
    // 퀸 이동 (룩 + 비숍)
    if ($pieceType === 'q') {
        $directions = [
            ['row' => -1, 'col' => 0],   // 위
            ['row' => 1, 'col' => 0],    // 아래
            ['row' => 0, 'col' => -1],   // 왼쪽
            ['row' => 0, 'col' => 1],    // 오른쪽
            ['row' => -1, 'col' => -1],  // 왼쪽 위
            ['row' => -1, 'col' => 1],   // 오른쪽 위
            ['row' => 1, 'col' => -1],   // 왼쪽 아래
            ['row' => 1, 'col' => 1]     // 오른쪽 아래
        ];
        
        foreach ($directions as $dir) {
            $moves = array_merge($moves, getLineMoves($board, $row, $col, $dir['row'], $dir['col'], $piece));
        }
    }
    
    // 킹 이동
    if ($pieceType === 'k') {
        $directions = [
            ['row' => -1, 'col' => 0],   // 위
            ['row' => 1, 'col' => 0],    // 아래
            ['row' => 0, 'col' => -1],   // 왼쪽
            ['row' => 0, 'col' => 1],    // 오른쪽
            ['row' => -1, 'col' => -1],  // 왼쪽 위
            ['row' => -1, 'col' => 1],   // 오른쪽 위
            ['row' => 1, 'col' => -1],   // 왼쪽 아래
            ['row' => 1, 'col' => 1]     // 오른쪽 아래
        ];
        
        foreach ($directions as $dir) {
            $newRow = $row + $dir['row'];
            $newCol = $col + $dir['col'];
            
            if ($newRow >= 0 && $newRow < 8 && $newCol >= 0 && $newCol < 8) {
                if ($board[$newRow][$newCol] === null || isOpponent($piece, $board[$newRow][$newCol])) {
                    $moves[] = ['row' => $newRow, 'col' => $newCol];
                }
            }
        }
    }
    
    return $moves;
}

/**
 * 직선(상하좌우) 또는 대각선 방향으로 이동 가능한 위치 계산
 */
function getLineMoves($board, $row, $col, $rowDir, $colDir, $piece) {
    $moves = [];
    $newRow = $row + $rowDir;
    $newCol = $col + $colDir;
    
    while ($newRow >= 0 && $newRow < 8 && $newCol >= 0 && $newCol < 8) {
        if ($board[$newRow][$newCol] === null) {
            // 빈 칸이면 이동 가능
            $moves[] = ['row' => $newRow, 'col' => $newCol];
        } else if (isOpponent($piece, $board[$newRow][$newCol])) {
            // 상대 말이면 캡처 가능하고 멈춤
            $moves[] = ['row' => $newRow, 'col' => $newCol];
            break;
        } else {
            // 아군 말이면 멈춤
            break;
        }
        
        $newRow += $rowDir;
        $newCol += $colDir;
    }
    
    return $moves;
}

/**
 * 두 말이 서로 상대편인지 확인
 */
function isOpponent($piece1, $piece2) {
    $isWhite1 = $piece1 === strtoupper($piece1);
    $isWhite2 = $piece2 === strtoupper($piece2);
    return $isWhite1 !== $isWhite2;
}

/**
 * 체스말 HTML 표현
 */
function renderPiece($piece) {
    if ($piece === null) {
        return '';
    }
    
    $pieceMap = [
        'P' => '♙', 'p' => '♟', // 폰
        'R' => '♖', 'r' => '♜', // 룩
        'N' => '♘', 'n' => '♞', // 나이트
        'B' => '♗', 'b' => '♝', // 비숍
        'Q' => '♕', 'q' => '♛', // 퀸
        'K' => '♔', 'k' => '♚'  // 킹
    ];
    
    $symbol = isset($pieceMap[$piece]) ? $pieceMap[$piece] : $piece;
    $color = $piece === strtoupper($piece) ? 'piece-white' : 'piece-black';
    
    return "<span class=\"{$color}\">{$symbol}</span>";
}

/**
 * 이동 가능 위치 계산 및 JSON 반환 (AJAX용)
 */
if (isset($_POST['action']) && $_POST['action'] === 'get_moves') {
    $row = intval($_POST['row']);
    $col = intval($_POST['col']);
    $piece = $_SESSION['chess_game']['board'][$row][$col];
    
    $validMoves = [];
    
    // 플레이어 턴이고 플레이어의 말인 경우만 이동 계산
    if ($_SESSION['chess_game']['turn'] === 'white' && $piece === strtoupper($piece)) {
        $validMoves = calculateValidMoves($row, $col, $piece);
    }
    
    header('Content-Type: application/json');
    echo json_encode($validMoves);
    exit;
}

$game = $_SESSION['chess_game'];
?>
<!DOCTYPE html>
<html lang="ko">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PHP 체스 게임</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }
        h1 {
            text-align: center;
        }
        .status {
            text-align: center;
            margin-bottom: 20px;
            font-weight: bold;
        }
        .captured-pieces {
            display: flex;
            justify-content: space-between;
            margin-bottom: 20px;
        }
        .captured-container {
            flex: 1;
            text-align: center;
            margin: 0 10px;
        }
        .chessboard {
            display: grid;
            grid-template-columns: repeat(8, 50px);
            grid-template-rows: repeat(8, 50px);
            border: 2px solid #333;
            width: 400px;
            margin: 0 auto;
        }
        .square {
            width: 50px;
            height: 50px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 28px;
            cursor: pointer;
        }
        .black {
            background-color: #999;
        }
        .white {
            background-color: #eee;
        }
        .selected {
            background-color: #ffcf75;
        }
        .valid-move {
            background-color: #90ee90;
        }
        .piece-white {
            color: blue;
        }
        .piece-black {
            color: red;
        }
        .controls {
            margin-top: 20px;
            text-align: center;
        }
        .reset-button {
            padding: 10px 20px;
            background-color: #4CAF50;
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 16px;
        }
        .reset-button:hover {
            background-color: #45a049;
        }
        .rules {
            margin-top: 20px;
            font-size: 14px;
        }
    </style>
</head>
<body>
    <h1>PHP 체스 게임</h1>
    
    <div class="status"><?php echo $game['status']; ?></div>
    
    <div class="captured-pieces">
        <div class="captured-container">
            <h3>컴퓨터의 잡은 말</h3>
            <div class="captured">
                <?php foreach ($game['captured_pieces']['black'] as $piece): ?>
                    <?php echo renderPiece($piece); ?>
                <?php endforeach; ?>
            </div>
        </div>
        <div class="captured-container">
            <h3>플레이어의 잡은 말</h3>
            <div class="captured">
                <?php foreach ($game['captured_pieces']['white'] as $piece): ?>
                    <?php echo renderPiece($piece); ?>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
    
    <div class="chessboard">
        <?php for ($row = 0; $row < 8; $row++): ?>
            <?php for ($col = 0; $col < 8; $col++): ?>
                <?php
                $squareColor = ($row + $col) % 2 === 0 ? 'white' : 'black';
                $piece = $game['board'][$row][$col];
                $pieceHtml = renderPiece($piece);
                $dataAttrs = "data-row=\"{$row}\" data-col=\"{$col}\"";
                ?>
                <div class="square <?php echo $squareColor; ?>" <?php echo $dataAttrs; ?>>
                    <?php echo $pieceHtml; ?>
                </div>
            <?php endfor; ?>
        <?php endfor; ?>
    </div>
    
    <div class="controls">
        <form method="post">
            <button type="submit" name="reset" class="reset-button">게임 재시작</button>
        </form>
    </div>
    
    <div class="rules">
        <h3>체스 말 이동 규칙:</h3>
        <p>♙♟: 폰 - 앞으로 1칸, 첫 이동시 2칸 가능, 대각선으로만 공격</p>
        <p>♖♜: 룩 - 상하좌우 이동</p>
        <p>♘♞: 나이트 - L자 이동 (장애물 건너뛸 수 있음)</p>
        <p>♗♝: 비숍 - 대각선 이동</p>
        <p>♕♛: 퀸 - 상하좌우, 대각선 이동</p>
        <p>♔♚: 킹 - 모든 방향 1칸 이동</p>
    </div>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            let selectedPiece = null;
            const squares = document.querySelectorAll('.square');
            
            squares.forEach(square => {
                square.addEventListener('click', function() {
                    const row = parseInt(this.dataset.row);
                    const col = parseInt(this.dataset.col);
                    
                    if (selectedPiece === null) {
                        // 말 선택
                        // AJAX로 유효한 이동 위치 가져오기
                        const formData = new FormData();
                        formData.append('action', 'get_moves');
                        formData.append('row', row);
                        formData.append('col', col);
                        
                        fetch('', {
                            method: 'POST',
                            body: formData
                        })
                        .then(response => response.json())
                        .then(validMoves => {
                            if (validMoves.length > 0) {
                                // 선택된 말 표시
                                this.classList.add('selected');
                                selectedPiece = { row, col };
                                
                                // 유효한 이동 위치 표시
                                validMoves.forEach(move => {
                                    const moveSquare = document.querySelector(`.square[data-row="${move.row}"][data-col="${move.col}"]`);
                                    if (moveSquare) {
                                        moveSquare.classList.add('valid-move');
                                    }
                                });
                            }
                        });
                    } else {
                        // 이미 말이 선택된 상태에서 이동 처리
                        const isValidMove = this.classList.contains('valid-move');
                        
                        if (isValidMove) {
                            // 폼 데이터로 이동 정보 전송
                            const formData = new FormData();
                            formData.append('from[row]', selectedPiece.row);
                            formData.append('from[col]', selectedPiece.col);
                            formData.append('to[row]', row);
                            formData.append('to[col]', col);
                            
                            fetch('', {
                                method: 'POST',
                                body: formData
                            })
                            .then(() => {
                                // 페이지 새로고침하여 업데이트된 보드 표시
                                window.location.reload();
                            });
                        } else {
                            // 유효하지 않은 이동 위치면 선택 해제
                            resetSelection();
                        }
                    }
                });
            });
            
            function resetSelection() {
                squares.forEach(square => {
                    square.classList.remove('selected');
                    square.classList.remove('valid-move');
                });
                selectedPiece = null;
            }
        });
    </script>
</body>
</html>
