<?php
declare(strict_types=1);

/**
 * 섯다 게임 - 컴퓨터와 대결하는 한국 전통 카드 게임
 * PHP 8.2 호환
 */

// 세션 시작
session_start();

// 섯다 카드 클래스
class SutdaCard {
    public function __construct(
        private int $month,
        private bool $isSpecial = false
    ) {}
    
    public function getMonth(): int {
        return $this->month;
    }
    
    public function isSpecial(): bool {
        return $this->isSpecial;
    }
    
    public function getImageName(): string {
        return $this->month . ($this->isSpecial ? 'k' : '');
    }
    
    public function toString(): string {
        return $this->month . ($this->isSpecial ? '(특)' : '');
    }
}

// 덱 클래스
class SutdaDeck {
    private array $cards = [];
    
    public function __construct() {
        $this->initialize();
    }
    
    public function initialize(): void {
        $this->cards = [];
        
        // 섯다는 1~10월 카드가 각 2장씩(일반 카드와 특수 카드)
        for ($month = 1; $month <= 10; $month++) {
            // 일반 카드
            $this->cards[] = new SutdaCard($month);
            
            // 특수 카드(광) - 1월, 3월, 8월만 특수 카드가 있음
            $isSpecial = in_array($month, [1, 3, 8]);
            $this->cards[] = new SutdaCard($month, $isSpecial);
        }
    }
    
    public function shuffle(): void {
        shuffle($this->cards);
    }
    
    public function deal(int $count = 1): array {
        $dealtCards = [];
        
        for ($i = 0; $i < $count; $i++) {
            if (count($this->cards) > 0) {
                $dealtCards[] = array_pop($this->cards);
            }
        }
        
        return $dealtCards;
    }
    
    public function getCount(): int {
        return count($this->cards);
    }
}

// 핸드 클래스
class SutdaHand {
    private array $cards = [];
    
    public function __construct(array $cards = []) {
        $this->cards = $cards;
    }
    
    public function addCards(array $cards): void {
        $this->cards = array_merge($this->cards, $cards);
    }
    
    public function getCards(): array {
        return $this->cards;
    }
    
    public function evaluateHand(): array {
        if (count($this->cards) !== 2) {
            return ['rank' => '노 페어', 'value' => 0, 'name' => '노 페어'];
        }
        
        $card1 = $this->cards[0];
        $card2 = $this->cards[1];
        
        $month1 = $card1->getMonth();
        $month2 = $card2->getMonth();
        $isSpecial1 = $card1->isSpecial();
        $isSpecial2 = $card2->isSpecial();
        
        // 38광땡 (3월 광 + 8월 광)
        if (
            ($month1 === 3 && $month2 === 8 && $isSpecial1 && $isSpecial2) ||
            ($month1 === 8 && $month2 === 3 && $isSpecial1 && $isSpecial2)
        ) {
            return ['rank' => '38광땡', 'value' => 50, 'name' => '38광땡'];
        }
        
        // 광땡 (1월 광 + 3월 광 or 1월 광 + 8월 광)
        if (
            ($month1 === 1 && $month2 === 3 && $isSpecial1 && $isSpecial2) ||
            ($month1 === 3 && $month2 === 1 && $isSpecial1 && $isSpecial2) ||
            ($month1 === 1 && $month2 === 8 && $isSpecial1 && $isSpecial2) ||
            ($month1 === 8 && $month2 === 1 && $isSpecial1 && $isSpecial2)
        ) {
            return ['rank' => '광땡', 'value' => 40, 'name' => '광땡'];
        }
        
        // 땡 (같은 월 카드 2장)
        if ($month1 === $month2) {
            $name = $month1 . '땡';
            return ['rank' => '땡', 'value' => 30 + $month1, 'name' => $name];
        }
        
        // 알리 (1+2)
        if (
            ($month1 === 1 && $month2 === 2) ||
            ($month1 === 2 && $month2 === 1)
        ) {
            return ['rank' => '알리', 'value' => 20, 'name' => '알리'];
        }
        
        // 독사 (1+4)
        if (
            ($month1 === 1 && $month2 === 4) ||
            ($month1 === 4 && $month2 === 1)
        ) {
            return ['rank' => '독사', 'value' => 19, 'name' => '독사'];
        }
        
        // 구삥 (1+9)
        if (
            ($month1 === 1 && $month2 === 9) ||
            ($month1 === 9 && $month2 === 1)
        ) {
            return ['rank' => '구삥', 'value' => 18, 'name' => '구삥'];
        }
        
        // 장삥 (1+10)
        if (
            ($month1 === 1 && $month2 === 10) ||
            ($month1 === 10 && $month2 === 1)
        ) {
            return ['rank' => '장삥', 'value' => 17, 'name' => '장삥'];
        }
        
        // 장사 (4+10)
        if (
            ($month1 === 4 && $month2 === 10) ||
            ($month1 === 10 && $month2 === 4)
        ) {
            return ['rank' => '장사', 'value' => 16, 'name' => '장사'];
        }
        
        // 세륙 (4+6)
        if (
            ($month1 === 4 && $month2 === 6) ||
            ($month1 === 6 && $month2 === 4)
        ) {
            return ['rank' => '세륙', 'value' => 15, 'name' => '세륙'];
        }
        
        // 끗 (두 카드의 합의 일의 자리)
        $sum = ($month1 + $month2) % 10;
        if ($sum === 0) {
            $sum = 10; // 갑오
        }
        
        return ['rank' => '끗', 'value' => $sum, 'name' => $sum . '끗'];
    }
}

// 게임 클래스
class SutdaGame {
    private SutdaDeck $deck;
    private SutdaHand $playerHand;
    private SutdaHand $computerHand;
    private bool $gameStarted = false;
    private string $result = '';
    private int $playerScore = 0;
    private int $computerScore = 0;
    
    public function __construct() {
        $this->deck = new SutdaDeck();
        $this->playerHand = new SutdaHand();
        $this->computerHand = new SutdaHand();
    }
    
    public function startGame(): void {
        $this->resetRound();
        $this->deck->shuffle();
        
        // 플레이어와 컴퓨터에게 각각 2장의 카드를 나눠줌
        $this->playerHand->addCards($this->deck->deal(2));
        $this->computerHand->addCards($this->deck->deal(2));
        
        $this->gameStarted = true;
        
        // 결과 계산
        $this->determineWinner();
    }
    
    public function resetRound(): void {
        $this->deck->initialize();
        $this->playerHand = new SutdaHand();
        $this->computerHand = new SutdaHand();
        $this->gameStarted = false;
        $this->result = '';
    }
    
    public function resetGame(): void {
        $this->resetRound();
        $this->playerScore = 0;
        $this->computerScore = 0;
    }
    
    private function determineWinner(): void {
        $playerHandRank = $this->playerHand->evaluateHand();
        $computerHandRank = $this->computerHand->evaluateHand();
        
        if ($playerHandRank['value'] > $computerHandRank['value']) {
            $this->result = "플레이어 승리! 플레이어: {$playerHandRank['name']}, 컴퓨터: {$computerHandRank['name']}";
            $this->playerScore++;
        } elseif ($playerHandRank['value'] < $computerHandRank['value']) {
            $this->result = "컴퓨터 승리! 플레이어: {$playerHandRank['name']}, 컴퓨터: {$computerHandRank['name']}";
            $this->computerScore++;
        } else {
            $this->result = "무승부! 양쪽 모두: {$playerHandRank['name']}";
        }
    }
    
    public function getPlayerCards(): array {
        return $this->playerHand->getCards();
    }
    
    public function getComputerCards(): array {
        return $this->computerHand->getCards();
    }
    
    public function isGameStarted(): bool {
        return $this->gameStarted;
    }
    
    public function getResult(): string {
        return $this->result;
    }
    
    public function getPlayerScore(): int {
        return $this->playerScore;
    }
    
    public function getComputerScore(): int {
        return $this->computerScore;
    }
}

// 게임 인스턴스를 세션에 저장
if (!isset($_SESSION['sutda_game'])) {
    $_SESSION['sutda_game'] = new SutdaGame();
}

$game = $_SESSION['sutda_game'];

// 액션 처리
$message = '';

if (isset($_POST['action'])) {
    switch ($_POST['action']) {
        case 'start':
            $game->startGame();
            break;
        case 'reset_round':
            $game->resetRound();
            break;
        case 'reset_game':
            $game->resetGame();
            break;
    }
    
    $_SESSION['sutda_game'] = $game;
}

// HTML 출력
?>

<!DOCTYPE html>
<html lang="ko">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PHP 섯다 게임</title>
    <style>
        body {
            font-family: 'Malgun Gothic', Arial, sans-serif;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
            background-color: #f5f5f5;
        }
        h1 {
            text-align: center;
            color: #333;
        }
        .game-board {
            display: flex;
            flex-direction: column;
            gap: 20px;
            background-color: #fff;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
        }
        .hand {
            display: flex;
            gap: 15px;
            margin-bottom: 20px;
            justify-content: center;
        }
        .card {
            width: 100px;
            height: 150px;
            border: 2px solid #333;
            border-radius: 10px;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            background-color: white;
            position: relative;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
            font-size: 24px;
            font-weight: bold;
        }
        .card .month {
            font-size: 36px;
            margin-bottom: 5px;
        }
        .card .special {
            color: red;
            font-size: 16px;
        }
        .card-back {
            background-color: #8B4513;
            background-image: linear-gradient(45deg, #8B4513 25%, #A0522D 25%, #A0522D 50%, #8B4513 50%, #8B4513 75%, #A0522D 75%, #A0522D);
            background-size: 20px 20px;
        }
        .actions {
            display: flex;
            gap: 10px;
            margin-top: 20px;
            justify-content: center;
        }
        .actions button {
            padding: 10px 20px;
            cursor: pointer;
            background-color: #4CAF50;
            color: white;
            border: none;
            border-radius: 5px;
            font-size: 16px;
            transition: background-color 0.3s;
        }
        .actions button:hover {
            background-color: #45a049;
        }
        .result {
            margin-top: 20px;
            font-weight: bold;
            font-size: 18px;
            text-align: center;
            padding: 10px;
            background-color: #f0f0f0;
            border-radius: 5px;
        }
        .score {
            display: flex;
            justify-content: space-around;
            margin-top: 20px;
            font-size: 18px;
            font-weight: bold;
        }
        .player-section, .computer-section {
            text-align: center;
            padding: 10px;
            background-color: #e9e9e9;
            border-radius: 5px;
            width: 45%;
        }
        .instructions {
            margin-top: 30px;
            padding: 15px;
            background-color: #e6f7ff;
            border-radius: 5px;
            border-left: 5px solid #1890ff;
        }
        .instructions h3 {
            margin-top: 0;
            color: #1890ff;
        }
        .instructions ul {
            padding-left: 20px;
        }
    </style>
</head>
<body>
    <h1>PHP 섯다 게임</h1>
    
    <div class="game-board">
        <div class="score">
            <div class="player-section">
                <div>플레이어 점수: <?php echo $game->getPlayerScore(); ?></div>
            </div>
            <div class="computer-section">
                <div>컴퓨터 점수: <?php echo $game->getComputerScore(); ?></div>
            </div>
        </div>
        
        <h2 style="text-align: center;">컴퓨터의 패</h2>
        <div class="hand">
            <?php if ($game->isGameStarted()): ?>
                <?php foreach ($game->getComputerCards() as $card): ?>
                    <div class="card">
                        <div class="month"><?php echo $card->getMonth(); ?></div>
                        <?php if ($card->isSpecial()): ?>
                            <div class="special">광</div>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <?php for ($i = 0; $i < 2; $i++): ?>
                    <div class="card card-back"></div>
                <?php endfor; ?>
            <?php endif; ?>
        </div>
        
        <h2 style="text-align: center;">플레이어의 패</h2>
        <div class="hand">
            <?php if ($game->isGameStarted()): ?>
                <?php foreach ($game->getPlayerCards() as $card): ?>
                    <div class="card">
                        <div class="month"><?php echo $card->getMonth(); ?></div>
                        <?php if ($card->isSpecial()): ?>
                            <div class="special">광</div>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <?php for ($i = 0; $i < 2; $i++): ?>
                    <div class="card card-back"></div>
                <?php endfor; ?>
            <?php endif; ?>
        </div>
        
        <form method="post">
            <div class="actions">
                <button type="submit" name="action" value="start">카드 받기</button>
                <button type="submit" name="action" value="reset_round">라운드 리셋</button>
                <button type="submit" name="action" value="reset_game">게임 리셋</button>
            </div>
        </form>
        
        <?php if ($game->isGameStarted()): ?>
            <div class="result">
                <?php echo htmlspecialchars($game->getResult()); ?>
            </div>
        <?php endif; ?>
        
        <div class="instructions">
            <h3>게임 규칙</h3>
            <ul>
                <li><strong>38광땡</strong>: 3월 광 + 8월 광 (최고 족보)</li>
                <li><strong>광땡</strong>: 1월 광 + 3월 광 또는 1월 광 + 8월 광</li>
                <li><strong>땡</strong>: 같은 월 카드 2장 (숫자가 클수록 높음)</li>
                <li><strong>알리</strong>: 1월 + 2월</li>
                <li><strong>독사</strong>: 1월 + 4월</li>
                <li><strong>구삥</strong>: 1월 + 9월</li>
                <li><strong>장삥</strong>: 1월 + 10월</li>
                <li><strong>장사</strong>: 4월 + 10월</li>
                <li><strong>세륙</strong>: 4월 + 6월</li>
                <li><strong>끗</strong>: 두 카드의 합이 일의 자리 수 (갑오: 10끗이 가장 높음)</li>
            </ul>
        </div>
    </div>
</body>
</html>
