<?php
declare(strict_types=1);

namespace Damoang\G5Plugin\Reaction\Models;

use Damoang\G5Plugin\Reaction\Reaction;
use Damoang\G5Plugin\Reaction\ReactionHelper;
use Exception;

/**
 * @template TReactionItem of array{
 *     reaction: string,
 *     category: string,
 *     reactionId: string,
 *     count: int,
 *     choose: bool,
 * }
 */
class ReactionModel
{
    /**
     * @return string[]
     */
    public static function getLogs(string $targetId, string $memberId): array
    {
        $db = Reaction::db();
        try {
            $stmt = $db->prepare('SELECT `reaction` FROM `' . Reaction::$tableChoose . '`
                WHERE
                    `member_id` = ?
                    AND `target_id` = ?
            ');
            $stmt->bind_param('ss', $memberId, $targetId);
            $stmt->execute();
            $result = $stmt->get_result();
            $result = $result->fetch_all(\MYSQLI_ASSOC);
        } catch (Exception $e) {
            return [];
        }

        $reactions = [];
        array_map(function ($row) use (&$reactions) {
            $reactions[] = $row['reaction'];
        }, $result);

        return $reactions;
    }

    /**
     * @return array<string,string[]>
     */
    public static function getLogsByParentId(string $parentId, string $memberId): array
    {
        $db = Reaction::db();

        try {
            $stmt = $db->prepare('SELECT `reaction`, `target_id`
                FROM `' . Reaction::$tableChoose . '`
                WHERE
                    `member_id` = ?
                    AND `parent_id` = ?
            ');
            if (!$stmt) {
                throw new Exception('리액션 데이터를 가져오는데 실패했습니다.');
            }
            $stmt->bind_param('ss', $memberId, $parentId);
            $stmt->execute();
            $result = $stmt->get_result();
            $result = $result->fetch_all(\MYSQLI_ASSOC);
        } catch (Exception $e) {
            return [];
        }

        $reactions = [];
        array_map(function ($row) use (&$reactions) {
            if (!is_array($reactions[$row['target_id']] ?? null)) {
                $reactions[$row['target_id']] = [];
            }
            $reactions[$row['target_id']][] = $row['reaction'];
        }, $result);

        return $reactions;
    }

    /**
     * 대상의 리액션 목록 반환
     * @return array<string,TReactionItem[]>
     */
    public static function getReactions(string $targetId, ?string $memberId): array
    {
        $reactions = [];

        $stmt = Reaction::db()->prepare(
            'SELECT `parent_id`, `target_id`, `reaction`, `reaction_count`
                FROM `' . Reaction::$tableReaction . '`
                WHERE `target_id` = ?
                ORDER BY `id` ASC
            '
        );
        $stmt->bind_param('s', $targetId);
        $stmt->execute();
        $result = $stmt->get_result();
        $result = $result->fetch_all(\MYSQLI_ASSOC);

        $memberActions = [];
        if ($memberId) {
            $memberActions = self::getLogs($targetId, $memberId);
        }

        foreach ($result as $row) {
            $reactionData = ReactionHelper::parseReaction($row['reaction'], intval($row['reaction_count'] ?? 0));
            if ($memberId && $memberActions) {
                $reactionData['choose'] = in_array($reactionData['reaction'], $memberActions);
            }
            $reactions[$row['target_id']][] = $reactionData;
        }

        return $reactions;
    }

    /**
     * 대상의 리액션 목록 반환
     * @return array<string,TReactionItem[]>
     */
    public static function getReactionsByParentId(string $parentId, ?string $memberId): array
    {
        $reactions = [];

        try {
            $table = Reaction::$tableReaction;
            $stmt = Reaction::db()->prepare("SELECT `parent_id`, `target_id`, `reaction`, `reaction_count`
                FROM `{$table}`
                WHERE `parent_id` = ?
            ");
            if (!$stmt) {
                throw new Exception('리액션 데이터를 가져오는데 실패했습니다.');
            }
            $stmt->bind_param('s', $parentId);
            $stmt->execute();
            $result = $stmt->get_result();
            $result = $result->fetch_all(\MYSQLI_ASSOC);
        } catch (Exception $e) {
            return [];
        }

        $memberActions = [];
        if ($memberId) {
            $memberActions = self::getLogsByParentId($parentId, $memberId);
        }

        foreach ($result as $row) {
            $reactionData = ReactionHelper::parseReaction($row['reaction'], intval($row['reaction_count'] ?? 0));
            if ($memberId && $memberActions) {
                $reactionData['choose'] = in_array($reactionData['reaction'], $memberActions[$row['target_id']] ?? []);
            }
            $reactions[$row['target_id']][] = $reactionData;
        }

        return $reactions;
    }

    /**
     * 리액션을 추가, 취소할 수 있는지 반환
     *
     * - 추가 및 취소 불가능: 0 (Reaction::NOT_REACTABLE)
     * - 추가 가능: 1 (Reaction::REACTABLE_ADD)
     * - 취소 가능: 2 (Reaction::REACTABLE_REVOKE)
     * - 추가 및 취소 가능: 3 (Reaction::REACTABLE)
     *
     * @param string[] $member
     */
    public static function reactable(string $targetId, string $reaction, array $member): int
    {
        global $is_admin;

        $reactable = Reaction::NOT_REACTABLE;
        // FIXME
        $reactionLimit = 20;

        if (!$member['mb_id']) {
            return Reaction::NOT_REACTABLE;
        }

        $db = Reaction::db();
        try {
            // 리액션 횟수 제한 확인
            $table = Reaction::$tableReaction;
            $stmt = $db->prepare("SELECT COUNT(*) AS `reactionRows` FROM `{$table}`
                WHERE `target_id` = ?
            ");
            if (!$stmt) {
                throw new Exception($db->error);
            }
            $stmt->bind_param('s', $targetId);
            $stmt->execute();
            $result = $stmt->get_result();
            $reactionRows = $result->fetch_assoc()['reactionRows'] ?? 0;
            $result = null;

            // 토글 모드일 때 취소할 수 있는지 확인
            $table = Reaction::$tableChoose;
            $stmt = $db->prepare("SELECT COUNT(*) AS `count` FROM `{$table}`
                WHERE
                    `member_id` = ?
                    AND `target_id` = ?
                    AND `reaction` = ?
            ");
            if (!$stmt) {
                throw new Exception($db->error);
            }
            $stmt->bind_param('sss', $member['mb_id'], $targetId, $reaction);
            $stmt->execute();
            $result = $stmt->get_result();
            $choose = boolval($result->fetch_assoc()['count'] ?? 0);
            $result = null;
        } catch (Exception $e) {
            $message = $is_admin ? "관리페이지에서 'DB 업그레이드'를 완료하세요. 오류 : {$e->getMessage()}" : '리액션을 추가할 수 없습니다.';
            throw new Exception($message, 0, $e);
        }

        if ($choose) {
            $reactable |= Reaction::REACTABLE_REVOKE;
        } else if ($reactionRows < $reactionLimit) {
            $reactable |= Reaction::REACTABLE_ADD;
        } else if ($is_admin) {
            $reactable |= Reaction::REACTABLE_ADD;
        }

        return $reactable;
    }

    /**
     * 리액션 추가
     */
    public static function addReaction(string $memberId, string $reaction, string $targetId, ?string $parentId): bool
    {
        ReactionHelper::validateReactionId($reaction);
        ReactionHelper::validateTargetId($targetId);
        if ($parentId !== null) {
            ReactionHelper::validateTargetId($parentId);
        } else {
            $parentId = $targetId;
        }

        $datetime = date('YmdHis');
        $db = Reaction::db();

        try {
            // 로그
            $table = Reaction::$tableChoose;
            $stmt = $db->prepare("INSERT INTO `{$table}`
                SET
                    `member_id` = ?,
                    `reaction` = ?,
                    `target_id` = ?,
                    `parent_id` = ?,
                    `created_at` = ?
            ");
            $stmt->bind_param('sssss', $memberId, $reaction, $targetId, $parentId, $datetime);
            $stmt->execute();
        } catch (Exception $e) {
            throw new Exception('리액션 기록에 실패했습니다.', 0, $e);
        }

        // 리액션 카운트 업데이트 및 추가
        $table = Reaction::$tableReaction;
        // 카운트 증가
        $stmt = $db->prepare("UPDATE `{$table}`
            SET
                `reaction_count` = `reaction_count` + 1
            WHERE
                `target_id` = ?
                AND `reaction` = ?
        ");
        $stmt->bind_param('ss', $targetId, $reaction);
        $stmt->execute();

        // 없으면 추가
        if (!$stmt->affected_rows) {
            $stmt = $db->prepare("INSERT INTO `{$table}`
                SET
                    `reaction` = ?,
                    `target_id` = ?,
                    `parent_id` = ?,
                    `reaction_count` = 1
            ");
            $stmt->bind_param('sss', $reaction, $targetId, $parentId);
            $stmt->execute();
        }

        return true;
    }

    public static function revokeReaction(string $memberId, string $reaction, string $targetId): bool
    {
        ReactionHelper::validateReactionId($reaction);
        ReactionHelper::validateTargetId($targetId);

        $db = Reaction::db();

        // choose 테이블에서 리액션 삭제
        $table = Reaction::$tableChoose;
        $stmt = $db->prepare("DELETE FROM `{$table}` WHERE `member_id` = ? AND `target_id` = ? AND `reaction` = ?");
        $stmt->bind_param('sss', $memberId, $targetId, $reaction);
        $result = $stmt->execute();

        if ($result) {
            // reaction 테이블에서 카운트 감소
            $table = Reaction::$tableReaction;
            $stmt = $db->prepare("UPDATE `{$table}`
                SET
                    `reaction_count` = `reaction_count` - 1
                WHERE
                    `target_id` = ?
                    AND `reaction` = ?
            ");
            $stmt->bind_param('ss', $targetId, $reaction);
            $stmt->execute();

            // 리액션이 없는 경우 삭제
            $stmt = $db->prepare("DELETE FROM `{$table}`
                WHERE
                    `target_id` = ?
                    AND `reaction` = ?
                    AND `reaction_count` <= 0
            ");
            $stmt->bind_param('ss', $targetId, $reaction);
            $stmt->execute();
        }

        return true;
    }
}
