<?php
declare(strict_types=1);

namespace Damoang\G5Plugin\Reaction;

use stdClass;

/**

 */
class ReactionHelper
{
    /**
     * @param ?array<mixed> $data
     */
    public static function generateIdDocument(string $boardId, $data): ?string
    {
        if (!isset($data['wr_id'])) {
            return null;
        }

        return self::sanitizeTargetId(implode(':', ['document', $boardId, $data['wr_id']]));
    }

    /**
     * @param ?array<mixed> $data
     */
    public static function generateIdComment(string $boardId, $data): ?string
    {
        if (!isset($data['wr_id'])) {
            return null;
        }

        $targetId = self::sanitizeTargetId(implode(':', ['comment', $boardId, $data['wr_id']]));
        $parentId = self::sanitizeTargetId(implode(':', ['document', $boardId, $data['wr_parent']]));

        return "{$targetId}@{$parentId}";
    }

    public static function generateIdContent(string $contentId): ?string
    {
        if (!$contentId) {
            return null;
        }

        return self::sanitizeTargetId(implode(':', ['content', $contentId]));
    }

    public static function validateReactionId(string $reactionId): void
    {
        if (strlen($reactionId) > 250) {
            throw new Exceptions\ReactionIdTooLongException();
        }
    }

    public static function validateTargetId(string $targetId): void
    {
        if (strlen($targetId) > 250) {
            throw new Exceptions\TargetIdTooLongException();
        }
    }

    /**
     * @return array{
     *     reaction: string,
     *     category: string,
     *     reactionId: string,
     *     count: int,
     *     choose: bool,
     * }
     */
    public static function parseReaction(string $reaction, int $count = 0): array
    {
        $result = [];
        $parts = explode(':', $reaction, 2);
        $result['reaction'] = $reaction;
        $result['category'] = $parts[0];
        $result['reactionId'] = $parts[1];
        $result['count'] = $count;
        $result['choose'] = false;

        return $result;
    }

    /**
     * 이미지 이모티콘 가져오기
     *
     * @return array{
     *     reaction: string,
     *     url: string,
     * }[]
     */
    public static function getImportImages(): array
    {
        $imageList = array_reduce(
            glob(DA_PLUGIN_REACTION_PATH . '/public/emoticon-images/*'),
            function ($carry, $file) {
                $fileinfo = pathinfo($file);
                $validExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'];
                if (in_array(strtolower($fileinfo['extension']), $validExtensions)) {
                    $filename = preg_replace('/[^a-zA-Z0-9_-]/', '', $fileinfo['filename']);
                    $filename = strtolower($filename);
                    $carry[] = [
                        'reaction' => "import-image:{$filename}",
                        'url' => str_replace(G5_PATH, G5_URL, $file),
                    ];
                }
                return $carry;
            },
            []
        );

        return $imageList;
    }

    public static function sanitizeTargetId(string $id): string
    {
        return preg_replace('/[^a-z0-9:_-]/', '', strtolower($id));
    }

}
