<?php
startSession();
//핀 - 로그인시 입력해야 합니다. 원하는 번호, 문자로 설정하세요.
define('_SFTP_PIN', '12345678');
//암호화키 32자리 영문숫자, 임의로 변경해서 사용하세요. 노출금지
define('_SFTP_ENC_KEY', '1234567891234567891234');

//편집가능 파일 - .으로 시작하는 파일은 .을 제외한 파일명을 입력하세요. 예(.htaccess)
define('_SFTP_OPEN_FILES', array('php', 'txt', 'htm', 'html', 'css', 'js', 'htaccess'));

//ssh2 라이브러리 체크
if (!function_exists('ssh2_connect')) {
    die("ssh2 extension not installed");
}
/**
 * SSH2 SFTP 연결을 설정하고 유지하는 함수
 *
 * @param string $host SFTP 서버 호스트 이름
 * @param int $port SFTP 서버 포트 번호
 * @param string $username SFTP 서버 계정 이름
 * @param string $password SFTP 서버 계정 비밀번호
 * @param int $timeout SSH2 연결 제한 시간 (초)
 * @return resource|null SSH2 SFTP 연결 핸들러. 실패 시 null 반환
 */
function establishSftpConnection($host, $port, $username, $password, $timeout = 10)
{
    // 이전 연결 로그아웃 확인
    if (isset($_SESSION['_sftp_logout']) && $_SESSION['_sftp_logout']) {
        return null;
    }

    // SSH2 연결 시도
    $connection = ssh2_connect($host, $port, ['timeout' => $timeout]);

    if (!is_resource($connection)) {
        // SSH2 연결 실패 시 에러 처리
        handleConnectionError('SSH2 연결에 실패하였습니다.');
        return null;
    }

    // SSH2 인증 시도
    if (!ssh2_auth_password($connection, $username, $password)) {
        // SSH2 인증 실패 시 에러 처리
        handleConnectionError('SSH2 인증에 실패하였습니다.');
        return null;
    }

    // SFTP 연결 시도
    $sftp = ssh2_sftp($connection);

    if (!is_resource($sftp)) {
        // SFTP 연결 실패 시 에러 처리
        handleConnectionError('SFTP 연결에 실패하였습니다.');
        return null;
    }

    // 연결 성공 시 세션 정보 저장
    $_SESSION['ssh2_connection'] = $connection;
    $_SESSION['sftp'] = $sftp;

    return $sftp;
}

/**
 * 연결 에러 처리 함수
 *
 * @param string $errorMessage 에러 메시지
 * @return void
 */
function handleConnectionError($errorMessage)
{
    $sessionKeys = ['ssh2_connection', 'ask_sftp_auth', '_sftp_host', '_sftp_id', '_sftp_pw', '_sftp_port', '_sftp_pin', '_message'];

    // 세션 정보 삭제
    foreach ($sessionKeys as $key) {
        unset($_SESSION[$key]);
    }

    $_SESSION['_message_login_error'] = $errorMessage;
    header('Location: ./login.php');
    exit;
}


/*
// 사용자 정의 오류 핸들러 함수
function customErrorHandler($errno, $errstr)
{
    echo '<div style="color:red; font-weight:bold;">' . $errstr . '</div>';
}

// set_error_handler 함수로 사용자 정의 오류 핸들러 등록
set_error_handler("customErrorHandler");
*/

if (isset($_SESSION['ask_sftp_auth']) && $_SESSION['ask_sftp_auth'] == true) {
    //로그인된 상태에서 SFTP 연결 생성.
    //sftp 연결
    $sessionKeys = ['_sftp_host', '_sftp_port', '_sftp_id', '_sftp_pw'];
    $credentials = [];
    foreach ($sessionKeys as $key) {
        $credentials[$key] = _decrypt($_SESSION[$key], _SFTP_ENC_KEY);
    }
    $sftp = establishSftpConnection($credentials['_sftp_host'], $credentials['_sftp_port'], $credentials['_sftp_id'], $credentials['_sftp_pw']);
} else {
    $_SESSION['_sftp_logout'] = true;
    // 로그인 페이지로 이동
    if (strpos($_SERVER['PHP_SELF'], 'login.php') === false) {
        header("location:login.php");
    }
}


/**
 * SFTP 디렉토리 내 파일 리스트 가져오기
 *
 * @param resource $sftp    SFTP 세션 핸들러
 * @param string   $dir     디렉토리 경로
 * @param bool     $is_root 루트 디렉토리 여부
 *
 * @return array 파일 및 디렉토리 목록
 * @throws Exception SFTP 연결 오류 발생 시 예외 처리
 */
function getSFTPFileList($sftp, $dir = '/', $is_root = false)
{

    // 루트 디렉토리 경로 설정
    if ($is_root) {
        $homedir = '/';
    } else {
        $homedir = $dir;
        // 경로의 마지막이 슬래시로 끝나면 제거
        if (substr($homedir, -1) === '/') {
            $homedir = substr($homedir, 0, -1);
        }
    }
    // 유효성 검사
    if (!isValidDirectory($dir)) {
        $errorMessage = "{$dir}는 유효하지 않은 디렉토리명입니다.";
        handleSftpError($errorMessage);
        return;
    }

    // SFTP 디렉토리 내 파일 리스트 가져오기
    $dir = @opendir("ssh2.sftp://{$sftp}{$homedir}");
    // 디렉토리 열기 실패 시 오류 처리
    if ($dir === false) {
        $errorMessage = $homedir . " 디렉토리를 열 수 없습니다.";
        handleSftpError($errorMessage);
        return;
    }

    $files = array();
    while ($file = readdir($dir)) {
        if ($file != "." && $file != "..") {
            $filePath = "ssh2.sftp://{$sftp}{$homedir}/" . $file;
            //directory
            if (is_dir($filePath)) {
                $files['dirs'][] = $file;
            } else {
                //file
                $fileInfo = array(
                    'name' => $file,
                    'size' => filesize($filePath),
                    'created' => date('Y-m-d H:i:s', filemtime($filePath)),
                    'owner' => ''
                );

                $ownerInfo = posix_getpwuid(fileowner($filePath));
                if ($ownerInfo !== false) {
                    $fileInfo['owner'] = $ownerInfo['name'];
                }

                $files['files'][] = $fileInfo;
            }
        }
    }
    closedir($dir);

    // 파일 및 디렉토리 목록 반환
    return $files;
}

/**
 * sftp 세션으로부터 uid 값을 가져오는 함수
 *
 * @param resource $sftp sftp 연결 리소스
 * 
 * @return int|null uid 값이 존재하면 해당 값, 그렇지 않으면 null 반환
 */
function getSftpUid($sftp)
{
    // sftp 연결이 유효한지 체크
    if (!is_resource($sftp) || get_resource_type($sftp) !== 'SSH2 SFTP') {
        return null;
    }

    // 사용자의 홈 디렉토리 경로 가져오기
    $home_dir = ssh2_sftp_realpath($sftp, '.');

    // 사용자 홈 디렉토리의 파일 상태 정보 가져오기
    $stat = ssh2_sftp_stat($sftp, $home_dir);

    // uid 값을 반환
    return isset($stat['uid']) ? intval($stat['uid']) : null;
}


function getFileUid($sftp, $file_path)
{
    // sftp 연결이 유효한지 체크
    if (!is_resource($sftp) || get_resource_type($sftp) !== 'SSH2 SFTP') {
        return null;
    }

    // 파일 정보 가져오기
    $stat = ssh2_sftp_stat($sftp, $file_path);

    // uid 값을 반환
    return isset($stat['uid']) ? intval($stat['uid']) : null;
}


/**
 * 폴더 경로에서 마지막 디렉토리를 제거하고 URL 매개변수로 반환합니다.
 * @param string $folder 현재 폴더 경로를 수정할 문자열
 * @return string|null 수정된 폴더 경로를 URL 매개변수로 반환하며, 입력이 잘못된 경우 null을 반환합니다.
 */
function getPreviousFolderParameter(string $folder): ?string
{
    // 마지막 슬래시가 있는 경우 제거합니다.
    $folder = rtrim($folder, '/');

    // 슬래시로 경로를 분할하고 마지막 디렉토리를 제거합니다.
    $path_parts = explode('/', $folder);
    array_pop($path_parts);

    // 수정된 부분으로 경로를 재구성합니다.
    $modified_path = implode('/', $path_parts);

    // 수정된 경로를 URL 매개변수로 반환합니다.
    return ($modified_path !== '') ? '?folder=' . $modified_path : '';
}


function checkPhpVersion()
{
    $phpVersion = phpversion();
    if (version_compare($phpVersion, '7.0.0') < 0) {
        echo '오류: PHP 7 이상 버전을 사용해주세요.';
        exit;
    }
}

function print_pre($data)
{
    echo '<pre>';
    print_r($data);
    echo '</pre>';
}

//에러 메세지 처리
function handleSftpError($errorMessage)
{
    $_SESSION['_warning_error_message'] = $errorMessage;
}

function isValidDirectory($dir)
{
    // 리눅스 디렉토리 생성 규칙에 맞는 정규식 패턴 (한글 포함)
    $pattern = '/^[\p{L}\p{N}\s\.\-_\/]*$/u';

    // 입력된 $dir 값에 상위 디렉토리 접근이 포함되어 있는지 확인
    if (strpos($dir, '..') !== false) {
        return false; // 상위 디렉토리 접근이 포함된 경우 유효하지 않음
    }

    // 입력된 $dir 값과 정규식 패턴 비교
    if ($dir !== '' && preg_match($pattern, $dir)) {
        return true; // 유효한 디렉토리명인 경우
    } else {
        return false; // 유효하지 않은 디렉토리명인 경우
    }
}

function startSession()
{
    // 세션 시작
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
}


/**
 * 데이터를 암호화합니다.
 *
 * @param string $data 암호화할 데이터
 * @param string $key 암호화에 사용할 키
 * @return string 암호화된 데이터
 */
function _encrypt($data, $key)
{
    $cipher = 'AES-256-CBC';
    $ivSize = openssl_cipher_iv_length($cipher);

    // 초기화 벡터 생성
    $iv = openssl_random_pseudo_bytes($ivSize);

    // 데이터를 암호화
    $encrypted = openssl_encrypt($data, $cipher, $key, OPENSSL_RAW_DATA, $iv);

    // HMAC 생성
    $hmac = hash_hmac('sha256', $encrypted, $key, true);

    // 암호화된 데이터, HMAC, 초기화 벡터를 연결하고 Base64로 인코딩
    return base64_encode($iv . $hmac . $encrypted);
}

/**
 * 데이터를 복호화합니다.
 *
 * @param string $data 복호화할 데이터
 * @param string $key 복호화에 사용할 키
 * @return string|bool 복호화된 데이터 또는 복호화 실패 시 false
 */
function _decrypt($data, $key)
{
    $cipher = 'AES-256-CBC';
    $ivSize = openssl_cipher_iv_length($cipher);

    // Base64 디코딩 및 데이터 분리
    $data = base64_decode($data);
    $iv = substr($data, 0, $ivSize);
    $hmac = substr($data, $ivSize, 32);
    $encrypted = substr($data, $ivSize + 32);

    // HMAC 검증
    $calculatedHmac = hash_hmac('sha256', $encrypted, $key, true);
    if (!hash_equals($hmac, $calculatedHmac)) {
        return false;
    }

    // 데이터를 복호화
    return openssl_decrypt($encrypted, $cipher, $key, OPENSSL_RAW_DATA, $iv);
}
