/**
 * JavaScript BigInteger library - subset useful for RSA encryption.
 *
 * Copyright (c) 2012 Jacob Lee <letsgolee@naver.com>
 *
 * This code works faster than the original work of Tom Wu.
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA or check at http://www.gnu.org/licenses/gpl.html
 */

/**
 * Copyright (c) 2005-2009  Tom Wu
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL TOM WU BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF
 * THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT
 * OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * In addition, the following condition applies:
 *
 * All redistributions must retain an intact copy of this copyright notice
 * and disclaimer.
 */


// JavaScript engine analysis
var canary = 0xdeadbeefcafe;
var j_lm = ((canary&0xffffff)==0xefcafe);


// (public) Constructor
function BigInteger(a,base) {
	this.d = [];
	this.s;
	this.t;
	if(a != null) {
		if("number" == typeof a) {
			this._fromInt(a);
		}
		else if(base == null && "string" != typeof a) /* this._fromString(a,256)*/ this._fromByteArray(a); // byte array
		else this._fromString(a,base);
	}
}

var dbits;

// am: Compute w_j += (x*this_i), propagate carries,
// c is initial carry, returns final carry.
// c < 3*dvalue, x < 2*dvalue, this_i < dvalue
// We need to select the fastest one that works in this environment.

// am1: use a single mult and divide to get the high bits,
// max digit bits should be 26 because
// max internal value = 2*dvalue^2-2*dvalue (< 2^53)
BigInteger.am1 = function(i,x,w,j,c,n) {
	while(--n >= 0) {
		var v = x*this.d[i++]+w.d[j]+c;
		c = Math.floor(v/0x4000000);
		w.d[j++] = v&0x3ffffff;
	}
	return c;
}
// am2 avoids a big mult-and-extract completely.
// Max digit bits should be <= 30 because we do bitwise ops
// on values up to 2*hdvalue^2-hdvalue-1 (< 2^31)
BigInteger.am2 = function(i,x,w,j,c,n) {
	var xl = x&0x7fff, xh = x>>15;
	while(--n >= 0) {
		var l = this.d[i]&0x7fff;
		var h = this.d[i++]>>15;
		var m = xh*l+h*xl;
		l = xl*l+((m&0x7fff)<<15)+w.d[j]+(c&0x3fffffff);
		c = (l>>>30)+(m>>>15)+xh*h+(c>>>30);
		w.d[j++] = l&0x3fffffff;
	}
	return c;
}
// Alternately, set max digit bits to 28 since some
// browsers slow down when dealing with 32-bit numbers.
BigInteger.am3 = function(i,x,w,j,c,n) {
	var xl = x&0x3fff, xh = x>>14;
	while(--n >= 0) {
		var l = this.d[i]&0x3fff;
		var h = this.d[i++]>>14;
		var m = xh*l+h*xl;
		l = xl*l+((m&0x3fff)<<14)+w.d[j]+c;
		c = (l>>28)+(m>>14)+xh*h;
		w.d[j++] = l&0xfffffff;
	}
	return c;
}
if(j_lm && (navigator.appName == "Microsoft Internet Explorer")) {
	BigInteger.prototype.am = BigInteger.am2;
	dbits = 30;
}
else if(j_lm && (navigator.appName != "Netscape")) {
	BigInteger.prototype.am = BigInteger.am1;
	dbits = 26;
}
else { // Mozilla/Netscape seems to prefer am3
	BigInteger.prototype.am = BigInteger.am3;
	dbits = 28;
}

BigInteger.DB = dbits;

BigInteger.DM = ((1<<BigInteger.DB)-1);
BigInteger.DV = (1<<BigInteger.DB);


var BI_FP = 52;
BigInteger.FV = Math.pow(2,BI_FP);
BigInteger.F1 = BI_FP-BigInteger.DB;
BigInteger.F2 = 2*BigInteger.DB-BI_FP;

// Digit conversions
BigInteger.RM = "0123456789abcdefghijklmnopqrstuvwxyz";
BigInteger.RC = new Array();
var rr,vv;
rr = "0".charCodeAt(0);
for(vv = 0; vv <= 9; ++vv) BigInteger.RC[rr++] = vv;
rr = "a".charCodeAt(0);
for(vv = 10; vv < 36; ++vv) BigInteger.RC[rr++] = vv;
rr = "A".charCodeAt(0);
for(vv = 10; vv < 36; ++vv) BigInteger.RC[rr++] = vv;

BigInteger.int2char = function(n) {return BigInteger.RM.charAt(n); };

BigInteger.intAt = function(s,i) {
	var c = BigInteger.RC[s.charCodeAt(i)];
	return (c==null)?-1:c;
};

// (protected) copy this to r
BigInteger.prototype._copyTo = function(r) {
	for(var i = this.t-1; i >= 0; --i) r.d[i] = this.d[i];
	r.t = this.t;
	r.s = this.s;
};

// (protected) set from integer value x, -DV <= x < DV
BigInteger.prototype._fromInt = function(x) {
	this.t = 1;
	this.s = (x<0)?-1:0;
	if(x > 0) this.d[0] = x;
	else if(x < -1) this.d[0] = x+BigInteger.DV;
	else this.t = 0;
};

// return bigint initialized to value
//function nbv(i) { var r = new BigInteger(null); r._fromInt(i); return r; };
BigInteger.havingInt = function(i) { var r = new BigInteger(null); r._fromInt(i); return r; };


BigInteger.prototype._fromByteArray = function(s) {
	this._fromString(s, 256);
}

// (protected) set from string and radix
BigInteger.prototype._fromString = function(s,b) {
	var k;
	if(b == 16) k = 4;
	else if(b == 8) k = 3;
	else if(b == 256) k = 8; // byte array
	else if(b == 2) k = 1;
	else if(b == 32) k = 5;
	else if(b == 4) k = 2;
	else { this._fromRadix(s,b); return; }
	this.t = 0;
	this.s = 0;
	var i = s.length, mi = false, sh = 0;
	while(--i >= 0) {
		var x = (k==8)?s[i]&0xff:BigInteger.intAt(s,i);
		if(x < 0) {
			if(s.charAt(i) == "-") mi = true;
			continue;
		}
		mi = false;
		if(sh == 0)
			this.d[this.t++] = x;
		else if(sh+k > BigInteger.DB) {
			this.d[this.t-1] |= (x&((1<<(BigInteger.DB-sh))-1))<<sh;
			this.d[this.t++] = (x>>(BigInteger.DB-sh));
		}
		else
			this.d[this.t-1] |= x<<sh;
		sh += k;
		if(sh >= BigInteger.DB) sh -= BigInteger.DB;
	}
	if(k == 8 && (s[0]&0x80) != 0) {
		this.s = -1;
		if(sh > 0) this.d[this.t-1] |= ((1<<(BigInteger.DB-sh))-1)<<sh;
	}
	this._clamp();
	if(mi) BigInteger.ZERO._subTo(this,this);
};

// (protected) clamp off excess high words
BigInteger.prototype._clamp = function() {
	var c = this.s&BigInteger.DM;
	while(this.t > 0 && this.d[this.t-1] == c) --this.t;
};

// (public) return string representation in given radix
BigInteger.prototype.toString = function(b) {
	if(this.s < 0) return "-"+this.negate().toString(b);
	var k;
	if(b == 16) k = 4;
	else if(b == 8) k = 3;
	else if(b == 2) k = 1;
	else if(b == 32) k = 5;
	else if(b == 4) k = 2;
	else return this._toRadix(b);
	var km = (1<<k)-1, d, m = false, r = "", i = this.t;
	var p = BigInteger.DB-(i*BigInteger.DB)%k;
	if(i-- > 0) {
		if(p < BigInteger.DB && (d = this.d[i]>>p) > 0) { m = true; r = BigInteger.int2char(d); }
		while(i >= 0) {
			if(p < k) {
				d = (this.d[i]&((1<<p)-1))<<(k-p);
				d |= this.d[--i]>>(p+=BigInteger.DB-k);
			}
			else {
				d = (this.d[i]>>(p-=k))&km;
				if(p <= 0) { p += BigInteger.DB; --i; }
			}
			if(d > 0) m = true;
			if(m) r += BigInteger.int2char(d);
		}
	}
	return m?r:"0";
};

// (public) -this
BigInteger.prototype.negate = function() { var r = new BigInteger(null); BigInteger.ZERO._subTo(this,r); return r; };

// (public) |this|
BigInteger.prototype.abs = function() { return (this.s<0)?this.negate():this; };

// (public) return + if this > a, - if this < a, 0 if equal
BigInteger.prototype.compareTo = function(a) {
	var r = this.s-a.s;
	if(r != 0) return r;
	var i = this.t;
	r = i-a.t;
	if(r != 0) return (this.s<0)?-r:r;
	while(--i >= 0) if((r=this.d[i]-a.d[i]) != 0) return r;
	return 0;
};

// returns bit length of the integer x
BigInteger.nBits = function(x) {
	var r = 1, t;
	if((t=x>>>16) != 0) { x = t; r += 16; }
	if((t=x>>8) != 0) { x = t; r += 8; }
	if((t=x>>4) != 0) { x = t; r += 4; }
	if((t=x>>2) != 0) { x = t; r += 2; }
	if((t=x>>1) != 0) { x = t; r += 1; }
	return r;
};

// (public) return the number of bits in "this"
BigInteger.prototype.bitLength = function() {
	if(this.t <= 0) return 0;
	return BigInteger.DB*(this.t-1)+BigInteger.nBits(this.d[this.t-1]^(this.s&BigInteger.DM));
};

// (protected) r = this << n*DB
BigInteger.prototype._dlShiftTo = function(n,r) {
	var x = this.clone();
	var i;
	for(i = /*this.t*/x.t-1; i >= 0; --i) r.d[i+n] = /*this.d[i]*/x.d[i];
	for(i = n-1; i >= 0; --i) r.d[i] = 0;
	r.t = /*this.t*/x.t+n;
	r.s = /*this.s*/x.s;
};

// (protected) r = this >> n*DB
BigInteger.prototype._drShiftTo = function(n,r) {
	var x = this.clone();
	for(var i = n; i < /*this.t*/x.t; ++i) r.d[i-n] = /*this.d[i]*/x.d[i];
	r.t = Math.max(/*this.t*/x.t-n,0);
	r.s = /*this.s*/x.s;
};

// (protected) r = this << n
BigInteger.prototype._lShiftTo = function(n,r) {
	var x = this.clone();
	var bs = n%BigInteger.DB;
	var cbs = BigInteger.DB-bs;
	var bm = (1<<cbs)-1;
	var ds = Math.floor(n/BigInteger.DB), c = (/*this.s*/x.s<<bs)&BigInteger.DM, i;
	for(i = /*this.t*/x.t-1; i >= 0; --i) {
		r.d[i+ds+1] = (/*this.d[i]*/x.d[i]>>cbs)|c;
		c = (/*this.d[i]*/x.d[i]&bm)<<bs;
	}
	for(i = ds-1; i >= 0; --i) r.d[i] = 0;
	r.d[ds] = c;
	r.t = /*this.t*/x.t+ds+1;
	r.s = /*this.s*/x.s;
	r._clamp();
};

// (protected) r = this >> n
BigInteger.prototype._rShiftTo = function(n,r) {
	var x = this.clone();
	r.s = /*this.s*/x.s;
	var ds = Math.floor(n/BigInteger.DB);
	if(ds >= /*this.t*/x.t) { r.t = 0; return; }
	var bs = n%BigInteger.DB;
	var cbs = BigInteger.DB-bs;
	var bm = (1<<bs)-1;
	r.d[0] = /*this.d[ds]*/x.d[ds]>>bs;
	for(var i = ds+1; i < /*this.t*/x.t; ++i) {
		r.d[i-ds-1] |= (/*this.d[i]*/x.d[i]&bm)<<cbs;
		r.d[i-ds] = /*this.d[i]*/x.d[i]>>bs;
	}
	if(bs > 0) r.d[/*this.t*/x.t-ds-1] |= (/*this.s*/x.s&bm)<<cbs;
	r.t = /*this.t*/x.t-ds;
	r._clamp();
};

// (protected) r = this - a
BigInteger.prototype._subTo = function(a,r) {
	var x = this.clone();// _subTo(this,this). the second this is the result.
	var i = 0, c = 0, m = Math.min(a.t,/*this.t*/x.t);
	while(i < m) {
		c += /*this.d[i]*/x.d[i]-a.d[i];
		r.d[i++] = c&BigInteger.DM;
		c >>= BigInteger.DB;
	}
	if(a.t < /*this.t*/x.t) {
		c -= a.s;
		while(i < /*this.t*/x.t) {
			c += /*this.d[i]*/x.d[i];
			r.d[i++] = c&BigInteger.DM;
			c >>= BigInteger.DB;
		}
		c += /*this.s*/x.s;
	}
	else {
		c += /*this.s*/x.s;
		while(i < a.t) {
			c -= a.d[i];
			r.d[i++] = c&BigInteger.DM;
			c >>= BigInteger.DB;
		}
		c -= a.s;
	}
	r.s = (c<0)?-1:0;
	if(c < -1) r.d[i++] = BigInteger.DV+c;
	else if(c > 0) r.d[i++] = c;
	r.t = i;
	r._clamp();
};

// (protected) r = this * a, r != this,a (HAC 14.12)
// "this" should be the larger one if appropriate.
BigInteger.prototype._multiplyTo = function(a,r) {
	var t = this.clone();
	var x = /*this.abs()*/t.abs(), y = a.abs();
	var i = x.t;
	r.t = i+y.t;
	while(--i >= 0) r.d[i] = 0;
	for(i = 0; i < y.t; ++i) r.d[i+x.t] = x.am(0,y.d[i],r,i,0,x.t);
	r.s = 0;
	r._clamp();
	if(/*this.s*/t.s != a.s) BigInteger.ZERO._subTo(r,r);
};

// (protected) r = this^2, r != this (HAC 14.16)
BigInteger.prototype._squareTo = function(r) {
	var x = this.abs();
	var i = r.t = 2*x.t;
	while(--i >= 0) r.d[i] = 0;
	for(i = 0; i < x.t-1; ++i) {
		var c = x.am(i,x.d[i],r,2*i,0,1);
		if((r.d[i+x.t]+=x.am(i+1,2*x.d[i],r,2*i+1,c,x.t-i-1)) >= BigInteger.DV) {
			r.d[i+x.t] -= BigInteger.DV;
			r.d[i+x.t+1] = 1;
		}
	}
	if(r.t > 0) r.d[r.t-1] += x.am(i,x.d[i],r,2*i,0,1);
	r.s = 0;
	r._clamp();
};

// (protected) divide this by m, quotient and remainder to q, r (HAC 14.20)
// r != q, this != m.	q or r may be null.
BigInteger.prototype._divRemTo = function(m,q,r) {
	var x = this.clone();
	var pm = m.abs();
	if(pm.t <= 0) return;
	var pt = /*this.abs()*/x.abs();
	if(pt.t < pm.t) {
		if(q != null) q._fromInt(0);
		if(r != null) /*this._copyTo(r)*/x._copyTo(r);
		return;
	}
	if(r == null) r = new BigInteger(null);
	var y = new BigInteger(null), ts = /*this.s*/x.s, ms = m.s;
	var nsh = BigInteger.DB-BigInteger.nBits(pm.d[pm.t-1]);	// normalize modulus
	if(nsh > 0) { pm._lShiftTo(nsh,y); pt._lShiftTo(nsh,r); }
	else { pm._copyTo(y); pt._copyTo(r); }
	var ys = y.t;
	var y0 = y.d[ys-1];
	if(y0 == 0) return;
	var yt = y0*(1<<BigInteger.F1)+((ys>1)?y.d[ys-2]>>BigInteger.F2:0);
	var d1 = BigInteger.FV/yt, d2 = (1<<BigInteger.F1)/yt, e = 1<<BigInteger.F2;
	var i = r.t, j = i-ys, t = (q==null)?new BigInteger(null):q;
	y._dlShiftTo(j,t);
	if(r.compareTo(t) >= 0) {
		r.d[r.t++] = 1;
		r._subTo(t,r);
	}
	BigInteger.ONE._dlShiftTo(ys,t);
	t._subTo(y,y);	// "negative" y so we can replace sub with am later
	while(y.t < ys) y.d[y.t++] = 0;
	while(--j >= 0) {
		// Estimate quotient digit
		var qd = (r.d[--i]==y0)?BigInteger.DM:Math.floor(r.d[i]*d1+(r.d[i-1]+e)*d2);
		if((r.d[i]+=y.am(0,qd,r,j,0,ys)) < qd) {	// Try it out
			y._dlShiftTo(j,t);
			r._subTo(t,r);
			while(r.d[i] < --qd) r._subTo(t,r);
		}
	}
	if(q != null) {
		r._drShiftTo(ys,q);
		if(ts != ms) BigInteger.ZERO._subTo(q,q);
	}
	r.t = ys;
	r._clamp();
	if(nsh > 0) r._rShiftTo(nsh,r);	// Denormalize remainder
	if(ts < 0) BigInteger.ZERO._subTo(r,r);
};

// (public) this mod a
BigInteger.prototype.mod = function(a) {
	var r = new BigInteger(null);
	this.abs()._divRemTo(a,null,r);
	if(this.s < 0 && r.compareTo(BigInteger.ZERO) > 0) a._subTo(r,r);
	return r;
};

// Modular reduction using "classic" algorithm
function Classic(m) { this.m = m; };

Classic.prototype.convert = function(x) {
	if(x.s < 0 || x.compareTo(this.m) >= 0) return x.mod(this.m);
	else return x;
};
Classic.prototype.revert = function(x) { return x; };
Classic.prototype.reduce = function(x) { x._divRemTo(this.m,null,x); };
Classic.prototype.mulTo = function(x,y,r) { x._multiplyTo(y,r); this.reduce(r); };
Classic.prototype.sqrTo = function(x,r) { x._squareTo(r); this.reduce(r); };


// (protected) return "-1/this % 2^DB"; useful for Mont. reduction
// justification:
//				 xy == 1 (mod m)
//				 xy =	1+km
//	 xy(2-xy) = (1+km)(1-km)
// x[y(2-xy)] = 1-k^2m^2
// x[y(2-xy)] == 1 (mod m^2)
// if y is 1/x mod m, then y(2-xy) is 1/x mod m^2
// should reduce x and y(2-xy) by m^2 at each step to keep size bounded.
// JS multiply "overflows" differently from C/C++, so care is needed here.
BigInteger.prototype._invDigit = function() {
	if(this.t < 1) return 0;
	var x = this.d[0];
	if((x&1) == 0) return 0;
	var y = x&3;		// y == 1/x mod 2^2
	y = (y*(2-(x&0xf)*y))&0xf;	// y == 1/x mod 2^4
	y = (y*(2-(x&0xff)*y))&0xff;	// y == 1/x mod 2^8
	y = (y*(2-(((x&0xffff)*y)&0xffff)))&0xffff;	// y == 1/x mod 2^16
	// last step - calculate inverse mod DV directly;
	// assumes 16 < DB <= 32 and assumes ability to handle 48-bit ints
	y = (y*(2-x*y%BigInteger.DV))%BigInteger.DV;		// y == 1/x mod 2^dbits
	// we really want the negative inverse, and -DV < y < DV
	return (y>0)?BigInteger.DV-y:-y;
};



// Montgomery reduction
function Montgomery(m) {
	this.m = m;
	this.mp = m._invDigit();
	this.mpl = this.mp&0x7fff;
	this.mph = this.mp>>15;
	this.um = (1<<(BigInteger.DB-15))-1;
	this.mt2 = 2*m.t;
};

// xR mod m
Montgomery.prototype.convert = function(x) {
	var r = new BigInteger(null);
	x.abs()._dlShiftTo(this.m.t,r);
	r._divRemTo(this.m,null,r);
	if(x.s < 0 && r.compareTo(BigInteger.ZERO) > 0) this.m._subTo(r,r);
	return r;
};

// x/R mod m
Montgomery.prototype.revert = function(x) {
	var r = new BigInteger(null);
	x._copyTo(r);
	this.reduce(r);
	return r;
};

// x = x/R mod m (HAC 14.32)
Montgomery.prototype.reduce = function(x) {
	while(x.t <= this.mt2)	// pad x so am has enough room later
		x.d[x.t++] = 0;
	for(var i = 0; i < this.m.t; ++i) {
		// faster way of calculating u0 = x.d[i]*mp mod DV
		var j = x.d[i]&0x7fff;
		var u0 = (j*this.mpl+(((j*this.mph+(x.d[i]>>15)*this.mpl)&this.um)<<15))&BigInteger.DM;
		// use am to combine the multiply-shift-add into one call
		j = i+this.m.t;
		x.d[j] += this.m.am(0,u0,x,i,0,this.m.t);
		// propagate carry
		while(x.d[j] >= BigInteger.DV) { x.d[j] -= BigInteger.DV; x.d[++j]++; }
	}
	x._clamp();
	x._drShiftTo(this.m.t,x);
	if(x.compareTo(this.m) >= 0) x._subTo(this.m,x);
};

// r = "x^2/R mod m"; x != r
Montgomery.prototype.sqrTo = function(x,r) { x._squareTo(r); this.reduce(r); };

// r = "xy/R mod m"; x,y != r
Montgomery.prototype.mulTo = function(x,y,r) { x._multiplyTo(y,r); this.reduce(r); };




// (protected) true if this is even
BigInteger.prototype._isEven = function() { return ((this.t>0)?(this.d[0]&1):this.s) == 0; };

// (protected) this^e, e < 2^32, doing sqr and mul with "r" (HAC 14.79)
BigInteger.prototype._exp = function(e,z) {
	if(e > 0xffffffff || e < 1) return BigInteger.ONE;
	var r = new BigInteger(null), r2 = new BigInteger(null), g = z.convert(this), i = BigInteger.nBits(e)-1;
	g._copyTo(r);
	while(--i >= 0) {
		z.sqrTo(r,r2);
		if((e&(1<<i)) > 0) z.mulTo(r2,g,r);
		else { var t = r; r = r2; r2 = t; }
	}
	return z.revert(r);
};

// (public) this^e % m, 0 <= e < 2^32
BigInteger.prototype.modPowInt = function(e,m) {
	var z;
	if(e < 256 || m._isEven()) z = new Classic(m); else z = new Montgomery(m);
	return this._exp(e,z);
};



// "constants"
BigInteger.ZERO = BigInteger.havingInt(0);
BigInteger.ONE = BigInteger.havingInt(1);



// (public)
BigInteger.prototype.clone = function() { var r = new BigInteger(null); this._copyTo(r); return r; };

// (public) return value as integer
BigInteger.prototype.intValue = function() {
	if(this.s < 0) {
		if(this.t == 1) return this.d[0]-BigInteger.DV;
		else if(this.t == 0) return -1;
	}
	else if(this.t == 1) return this.d[0];
	else if(this.t == 0) return 0;
	// assumes 16 < DB < 32
	return ((this.d[1]&((1<<(32-BigInteger.DB))-1))<<BigInteger.DB)|this.d[0];
};

// (public) return value as byte
BigInteger.prototype.byteValue = function() { return (this.t==0)?this.s:(this.d[0]<<24)>>24; };

// (public) return value as short (assumes DB>=16)
BigInteger.prototype.shortValue = function() { return (this.t==0)?this.s:(this.d[0]<<16)>>16; };

// (protected) return x s.t. r^x < DV
BigInteger.prototype._chunkSize = function(r) { return Math.floor(Math.LN2*BigInteger.DB/Math.log(r)); };

// (public) 0 if this == 0, 1 if this > 0
BigInteger.prototype._sigNum = function() {
	if(this.s < 0) return -1;
	else if(this.t <= 0 || (this.t == 1 && this.d[0] <= 0)) return 0;
	else return 1;
};

// (protected) convert to radix string
BigInteger.prototype._toRadix = function(b) {
	if(b == null) b = 10;
	if(this._sigNum() == 0 || b < 2 || b > 36) return "0";
	var cs = this._chunkSize(b);
	var a = Math.pow(b,cs);
	var d = BigInteger.havingInt(a), y = new BigInteger(null), z = new BigInteger(null), r = "";
	this._divRemTo(d,y,z);
	while(y._sigNum() > 0) {
		r = (a+z.intValue()).toString(b).substr(1) + r;
		y._divRemTo(d,y,z);
	}
	return z.intValue().toString(b) + r;
};

// (protected) convert from radix string
BigInteger.prototype._fromRadix = function(s,b) {
	this._fromInt(0);
	if(b == null) b = 10;
	var cs = this._chunkSize(b);
	var d = Math.pow(b,cs), mi = false, j = 0, w = 0;
	for(var i = 0; i < s.length; ++i) {
		var x = BigInteger.intAt(s,i);
		if(x < 0) {
			if(s.charAt(i) == "-" && this._sigNum() == 0) mi = true;
			continue;
		}
		w = b*w+x;
		if(++j >= cs) {
			this._dMultiply(d);
			this._dAddOffset(w,0);
			j = 0;
			w = 0;
		}
	}
	if(j > 0) {
		this._dMultiply(Math.pow(b,j));
		this._dAddOffset(w,0);
	}
	if(mi) BigInteger.ZERO._subTo(this,this);
};
/*
// (protected) alternate constructor
BigInteger.prototype._fromNumber = function(a,b,c) {
	if("number" == typeof b) {
		// new BigInteger(int,int,RNG)
		if(a < 2) this._fromInt(1);
		else {
			this._fromNumber(a,c);
			if(!this.testBit(a-1))	// force MSB set
				this._bitwiseTo(BigInteger.ONE.shiftLeft(a-1),BigInteger.op_or,this);
			if(this._isEven()) this._dAddOffset(1,0); // force odd
			while(!this.isProbablePrime(b)) {
				this._dAddOffset(2,0);
				if(this.bitLength() > a) this._subTo(BigInteger.ONE.shiftLeft(a-1),this);
			}
		}
	}
	else {
		// new BigInteger(int,RNG)
		this.random(a, b);
	}
};
*/

BigInteger.prototype.random = function(bit, rng) {
	// new BigInteger(int,RNG)
	var x = new Array(), t = bit&7;
	x.length = (bit>>3)+1;
	x = rng.getBytes(x);
	if(t > 0) x[0] &= ((1<<t)-1);
	else x[0] = 0;
	this._fromString(x,256);
};

BigInteger.prototype.randomPrime = function(bit, rng, repeat) {
	// $repeat is repeat times for accuracy of the test
	// http://en.wikipedia.org/wiki/Miller%E2%80%93Rabin_primality_test

	if (typeof repeat == 'undefined' || repeat == null) repeat = 1;

	this.random(bit, rng);
	if(!this.testBit(bit-1))	// force MSB set
		this._bitwiseTo(BigInteger.ONE.shiftLeft(bit-1), BigInteger.op_or, this);
	if(this._isEven()) this._dAddOffset(1,0); // force odd
	while(!this.isProbablePrime(repeat)) {
		this._dAddOffset(2,0);
		if(this.bitLength() > bit) this._subTo(BigInteger.ONE.shiftLeft(bit-1), this);
	}
};


// (public) convert to bigendian byte array
BigInteger.prototype.toByteArray = function() {
	var i = this.t, r = new Array();
	r[0] = this.s;
	var p = BigInteger.DB-(i*BigInteger.DB)%8, d, k = 0;
	if(i-- > 0) {
		if(p < BigInteger.DB && (d = this.d[i]>>p) != (this.s&BigInteger.DM)>>p)
			r[k++] = d|(this.s<<(BigInteger.DB-p));
		while(i >= 0) {
			if(p < 8) {
				d = (this.d[i]&((1<<p)-1))<<(8-p);
				d |= this.d[--i]>>(p+=BigInteger.DB-8);
			}
			else {
				d = (this.d[i]>>(p-=8))&0xff;
				if(p <= 0) { p += BigInteger.DB; --i; }
			}
			if((d&0x80) != 0) d |= -256;
			if(k == 0 && (this.s&0x80) != (d&0x80)) ++k;
			if(k > 0 || d != this.s) r[k++] = d;
		}
	}
	return r;
};

BigInteger.prototype.equals = function(a) { return(this.compareTo(a)==0); };
BigInteger.prototype.min = function(a) { return(this.compareTo(a)<0)?this:a; };
BigInteger.prototype.max = function(a) { return(this.compareTo(a)>0)?this:a; };

// (protected) r = this op a (bitwise)
BigInteger.prototype._bitwiseTo = function(a,op,r) {
	var i, f, m = Math.min(a.t,this.t);
	for(i = 0; i < m; ++i) r.d[i] = op(this.d[i],a.d[i]);
	if(a.t < this.t) {
		f = a.s&BigInteger.DM;
		for(i = m; i < this.t; ++i) r.d[i] = op(this.d[i],f);
		r.t = this.t;
	}
	else {
		f = this.s&BigInteger.DM;
		for(i = m; i < a.t; ++i) r.d[i] = op(f,a.d[i]);
		r.t = a.t;
	}
	r.s = op(this.s,a.s);
	r._clamp();
};

// (public) this & a
BigInteger.op_and = function(x,y) { return x&y; }
BigInteger.prototype.and = function(a) { var r = new BigInteger(null); this._bitwiseTo(a,BigInteger.op_and,r); return r; };

// (public) this | a
BigInteger.op_or = function(x,y) { return x|y; }
BigInteger.prototype.or = function(a) { var r = new BigInteger(null); this._bitwiseTo(a,BigInteger.op_or,r); return r; };

// (public) this ^ a
BigInteger.op_xor = function(x,y) { return x^y; }
BigInteger.prototype.xor = function(a) { var r = new BigInteger(null); this._bitwiseTo(a,BigInteger.op_xor,r); return r; };

// (public) this & ~a
BigInteger.op_andnot = function(x,y) { return x&~y; }
BigInteger.prototype.andNot = function(a) { var r = new BigInteger(null); this._bitwiseTo(a,BigInteger.op_andnot,r); return r; };

// (public) ~this
BigInteger.prototype.not = function() {
	var r = new BigInteger(null);
	for(var i = 0; i < this.t; ++i) r.d[i] = BigInteger.DM&~this.d[i];
	r.t = this.t;
	r.s = ~this.s;
	return r;
};

// (public) this << n
BigInteger.prototype.shiftLeft = function(n) {
	var r = new BigInteger(null);
	if(n < 0) this._rShiftTo(-n,r); else this._lShiftTo(n,r);
	return r;
};

// (public) this >> n
BigInteger.prototype.shiftRight = function(n) {
	var r = new BigInteger(null);
	if(n < 0) this._lShiftTo(-n,r); else this._rShiftTo(n,r);
	return r;
};

// return index of lowest 1-bit in x, x < 2^31
BigInteger.lbit = function(x) {
	if(x == 0) return -1;
	var r = 0;
	if((x&0xffff) == 0) { x >>= 16; r += 16; }
	if((x&0xff) == 0) { x >>= 8; r += 8; }
	if((x&0xf) == 0) { x >>= 4; r += 4; }
	if((x&3) == 0) { x >>= 2; r += 2; }
	if((x&1) == 0) ++r;
	return r;
};

// (public) returns index of lowest 1-bit (or -1 if none)
BigInteger.prototype.getLowestSetBit = function() {
	for(var i = 0; i < this.t; ++i)
		if(this.d[i] != 0) return i*BigInteger.DB+BigInteger.lbit(this.d[i]);
	if(this.s < 0) return this.t*BigInteger.DB;
	return -1;
};

// return number of 1 bits in x
BigInteger.cbit = function(x) {
	var r = 0;
	while(x != 0) { x &= x-1; ++r; }
	return r;
};

// (public) return number of set bits
BigInteger.prototype.bitCount = function() {
	var r = 0, x = this.s&BigInteger.DM;
	for(var i = 0; i < this.t; ++i) r += BigInteger.cbit(this.d[i]^x);
	return r;
};

// (public) true iff nth bit is set
BigInteger.prototype.testBit = function(n) {
	var j = Math.floor(n/BigInteger.DB);
	if(j >= this.t) return(this.s!=0);
	return((this.d[j]&(1<<(n%BigInteger.DB)))!=0);
};

// (protected) this op (1<<n)
BigInteger.prototype._changeBit = function(n,op) {
	var r = BigInteger.ONE.shiftLeft(n);
	this._bitwiseTo(r,op,r);
	return r;
};

// (public) this | (1<<n)
BigInteger.prototype.setBit = function(n) { return this._changeBit(n,BigInteger.op_or); };

// (public) this & ~(1<<n)
BigInteger.prototype.clearBit = function(n) { return this._changeBit(n,BigInteger.op_andnot); };

// (public) this ^ (1<<n)
BigInteger.prototype.flipBit = function(n) { return this._changeBit(n,BigInteger.op_xor); };

// (protected) r = this + a
BigInteger.prototype._addTo = function(a,r) {
	var i = 0, c = 0, m = Math.min(a.t,this.t);
	while(i < m) {
		c += this.d[i]+a.d[i];
		r.d[i++] = c&BigInteger.DM;
		c >>= BigInteger.DB;
	}
	if(a.t < this.t) {
		c += a.s;
		while(i < this.t) {
			c += this.d[i];
			r.d[i++] = c&BigInteger.DM;
			c >>= BigInteger.DB;
		}
		c += this.s;
	}
	else {
		c += this.s;
		while(i < a.t) {
			c += a.d[i];
			r.d[i++] = c&BigInteger.DM;
			c >>= BigInteger.DB;
		}
		c += a.s;
	}
	r.s = (c<0)?-1:0;
	if(c > 0) r.d[i++] = c;
	else if(c < -1) r.d[i++] = BigInteger.DV+c;
	r.t = i;
	r._clamp();
};

// (public) this + a
BigInteger.prototype.add = function(a) { var r = new BigInteger(null); this._addTo(a,r); return r; };

// (public) this - a
BigInteger.prototype.subtract = function(a) { var r = new BigInteger(null); this._subTo(a,r); return r; };

// (public) this * a
BigInteger.prototype.multiply = function(a) { var r = new BigInteger(null); this._multiplyTo(a,r); return r; };

// (public) this^2
BigInteger.prototype.square = function() { var r = new BigInteger(null); this._squareTo(r); return r; };

// (public) this / a
BigInteger.prototype.divide = function(a) { var r = new BigInteger(null); this._divRemTo(a,r,null); return r; };

// (public) this % a
BigInteger.prototype.remainder = function(a) { var r = new BigInteger(null); this._divRemTo(a,null,r); return r; };

// (public) [this/a,this%a]
BigInteger.prototype.divideAndRemainder = function(a) {
	var q = new BigInteger(null), r = new BigInteger(null);
	this._divRemTo(a,q,r);
	return new Array(q,r);
};

// (protected) this *= n, this >= 0, 1 < n < DV
BigInteger.prototype._dMultiply = function(n) {
	this.d[this.t] = this.am(0,n-1,this,0,0,this.t);
	++this.t;
	this._clamp();
};

// (protected) this += n << w words, this >= 0
BigInteger.prototype._dAddOffset = function(n,w) {
	if(n == 0) return;
	while(this.t <= w) this.d[this.t++] = 0;
	this.d[w] += n;
	while(this.d[w] >= BigInteger.DV) {
		this.d[w] -= BigInteger.DV;
		if(++w >= this.t) this.d[this.t++] = 0;
		++this.d[w];
	}
};

// A "null" reducer
function NullExp() {};
NullExp.prototype.convert = function(x) { return x; };
NullExp.prototype.revert = function(x) { return x; };
NullExp.prototype.mulTo = function(x,y,r) { x._multiplyTo(y,r); };
NullExp.prototype.sqrTo = function(x,r) { x._squareTo(r); };




// (public) this^e
BigInteger.prototype.pow = function(e) { return this._exp(e,new NullExp()); };

// (protected) r = lower n words of "this * a", a.t <= n
// "this" should be the larger one if appropriate.
BigInteger.prototype._multiplyLowerTo = function(a,n,r) {
	var i = Math.min(this.t+a.t,n);
	r.s = 0; // assumes a,this >= 0
	r.t = i;
	while(i > 0) r.d[--i] = 0;
	var j;
	for(j = r.t-this.t; i < j; ++i) r.d[i+this.t] = this.am(0,a.d[i],r,i,0,this.t);
	for(j = Math.min(a.t,n); i < j; ++i) this.am(0,a.d[i],r,i,0,n-i);
	r._clamp();
};

// (protected) r = "this * a" without lower n words, n > 0
// "this" should be the larger one if appropriate.
BigInteger.prototype._multiplyUpperTo = function(a,n,r) {
	--n;
	var i = r.t = this.t+a.t-n;
	r.s = 0; // assumes a,this >= 0
	while(--i >= 0) r.d[i] = 0;
	for(i = Math.max(n-this.t,0); i < a.t; ++i)
		r.d[this.t+i-n] = this.am(n-i,a.d[i],r,0,0,this.t+i-n);
	r._clamp();
	r._drShiftTo(1,r);
};

// Barrett modular reduction
function Barrett(m) {
	// setup Barrett
	this.r2 = new BigInteger(null);
	this.q3 = new BigInteger(null);
	BigInteger.ONE._dlShiftTo(2*m.t,this.r2);
	this.mu = this.r2.divide(m);
	this.m = m;
};

Barrett.prototype.convert = function(x) {
	if(x.s < 0 || x.t > 2*this.m.t) return x.mod(this.m);
	else if(x.compareTo(this.m) < 0) return x;
	else { var r = new BigInteger(null); x._copyTo(r); this.reduce(r); return r; }
};

Barrett.prototype.revert = function(x) { return x; };

// x = x mod m (HAC 14.42)
Barrett.prototype.reduce = function(x) {
	x._drShiftTo(this.m.t-1,this.r2);
	if(x.t > this.m.t+1) { x.t = this.m.t+1; x._clamp(); }
	this.mu._multiplyUpperTo(this.r2,this.m.t+1,this.q3);
	this.m._multiplyLowerTo(this.q3,this.m.t+1,this.r2);
	while(x.compareTo(this.r2) < 0) x._dAddOffset(1,this.m.t+1);
	x._subTo(this.r2,x);
	while(x.compareTo(this.m) >= 0) x._subTo(this.m,x);
};

// r = x^2 mod m; x != r
Barrett.prototype.sqrTo = function(x,r) { x._squareTo(r); this.reduce(r); };

// r = x*y mod m; x,y != r
Barrett.prototype.mulTo = function(x,y,r) { x._multiplyTo(y,r); this.reduce(r); };



// (public) this^e % m (HAC 14.85)
BigInteger.prototype.modPow = function(e,m) {
	var i = e.bitLength(), k, r = BigInteger.havingInt(1), z;
	if(i <= 0) return r;
	else if(i < 18) k = 1;
	else if(i < 48) k = 3;
	else if(i < 144) k = 4;
	else if(i < 768) k = 5;
	else k = 6;
	if(i < 8)
		z = new Classic(m);
	else if(m._isEven())
		z = new Barrett(m);
	else
		z = new Montgomery(m);

	// precomputation
	var g = new Array(), n = 3, k1 = k-1, km = (1<<k)-1;
	g[1] = z.convert(this);
	if(k > 1) {
		var g2 = new BigInteger(null);
		z.sqrTo(g[1],g2);
		while(n <= km) {
			g[n] = new BigInteger(null);
			z.mulTo(g2,g[n-2],g[n]);
			n += 2;
		}
	}

	var j = e.t-1, w, is1 = true, r2 = new BigInteger(null), t;
	i = BigInteger.nBits(e.d[j])-1;
	while(j >= 0) {
		if(i >= k1) w = (e.d[j]>>(i-k1))&km;
		else {
			w = (e.d[j]&((1<<(i+1))-1))<<(k1-i);
			if(j > 0) w |= e.d[j-1]>>(BigInteger.DB+i-k1);
		}

		n = k;
		while((w&1) == 0) { w >>= 1; --n; }
		if((i -= n) < 0) { i += BigInteger.DB; --j; }
		if(is1) {	// ret == 1, don't bother squaring or multiplying it
			g[w]._copyTo(r);
			is1 = false;
		}
		else {
			while(n > 1) { z.sqrTo(r,r2); z.sqrTo(r2,r); n -= 2; }
			if(n > 0) z.sqrTo(r,r2); else { t = r; r = r2; r2 = t; }
			z.mulTo(r2,g[w],r);
		}

		while(j >= 0 && (e.d[j]&(1<<i)) == 0) {
			z.sqrTo(r,r2); t = r; r = r2; r2 = t;
			if(--i < 0) { i = BigInteger.DB-1; --j; }
		}
	}
	return z.revert(r);
};

// (public) gcd(this,a) (HAC 14.54)
BigInteger.prototype.gcd = function(a) {
	var x = (this.s<0)?this.negate():this.clone();
	var y = (a.s<0)?a.negate():a.clone();
	if(x.compareTo(y) < 0) { var t = x; x = y; y = t; }
	var i = x.getLowestSetBit(), g = y.getLowestSetBit();
	if(g < 0) return x;
	if(i < g) g = i;
	if(g > 0) {
		x._rShiftTo(g,x);
		y._rShiftTo(g,y);
	}
	while(x._sigNum() > 0) {
		if((i = x.getLowestSetBit()) > 0) x._rShiftTo(i,x);
		if((i = y.getLowestSetBit()) > 0) y._rShiftTo(i,y);
		if(x.compareTo(y) >= 0) {
			x._subTo(y,x);
			x._rShiftTo(1,x);
		}
		else {
			y._subTo(x,y);
			y._rShiftTo(1,y);
		}
	}
	if(g > 0) y._lShiftTo(g,y);
	return y;
};

// (protected) this % n, n < 2^26
BigInteger.prototype._modInt = function(n) {
	if(n <= 0) return 0;
	var d = BigInteger.DV%n, r = (this.s<0)?n-1:0;
	if(this.t > 0)
		if(d == 0) r = this.d[0]%n;
		else for(var i = this.t-1; i >= 0; --i) r = (d*r+this.d[i])%n;
	return r;
};

// (public) 1/this % m (HAC 14.61)
BigInteger.prototype.modInverse = function(m) {
	var ac = m._isEven();
	if((this._isEven() && ac) || m._sigNum() == 0) return BigInteger.ZERO;
	var u = m.clone(), v = this.clone();
	var a = BigInteger.havingInt(1), b = BigInteger.havingInt(0), c = BigInteger.havingInt(0), d = BigInteger.havingInt(1);
	while(u._sigNum() != 0) {
		while(u._isEven()) {
			u._rShiftTo(1,u);
			if(ac) {
				if(!a._isEven() || !b._isEven()) { a._addTo(this,a); b._subTo(m,b); }
				a._rShiftTo(1,a);
			}
			else if(!b._isEven()) b._subTo(m,b);
			b._rShiftTo(1,b);
		}
		while(v._isEven()) {
			v._rShiftTo(1,v);
			if(ac) {
				if(!c._isEven() || !d._isEven()) { c._addTo(this,c); d._subTo(m,d); }
				c._rShiftTo(1,c);
			}
			else if(!d._isEven()) d._subTo(m,d);
			d._rShiftTo(1,d);
		}
		if(u.compareTo(v) >= 0) {
			u._subTo(v,u);
			if(ac) a._subTo(c,a);
			b._subTo(d,b);
		}
		else {
			v._subTo(u,v);
			if(ac) c._subTo(a,c);
			d._subTo(b,d);
		}
	}
	if(v.compareTo(BigInteger.ONE) != 0) return BigInteger.ZERO;
	if(d.compareTo(m) >= 0) return d.subtract(m);
	if(d._sigNum() < 0) d._addTo(m,d); else return d;
	if(d._sigNum() < 0) return d.add(m); else return d;
};

//BigInteger.lowPrimes = [2,3,5,7,11,13,17,19,23,29,31,37,41,43,47,53,59,61,67,71,73,79,83,89,97,101,103,107,109,113,127,131,137,139,149,151,157,163,167,173,179,181,191,193,197,199,211,223,227,229,233,239,241,251,257,263,269,271,277,281,283,293,307,311,313,317,331,337,347,349,353,359,367,373,379,383,389,397,401,409,419,421,431,433,439,443,449,457,461,463,467,479,487,491,499,503,509,521,523,541,547,557,563,569,571,577,587,593,599,601,607,613,617,619,631,641,643,647,653,659,661,673,677,683,691,701,709,719,727,733,739,743,751,757,761,769,773,787,797,809,811,821,823,827,829,839,853,857,859,863,877,881,883,887,907,911,919,929,937,941,947,953,967,971,977,983,991,997];
BigInteger.lowPrimes = [
	2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37, 41, 43, 47, 53, 59, 61, 67, 71,
	73, 79, 83, 89, 97, 101, 103, 107, 109, 113, 127, 131, 137, 139, 149, 151,
	157, 163, 167, 173, 179, 181, 191, 193, 197, 199, 211, 223, 227, 229, 233,
	239, 241, 251, 257, 263, 269, 271, 277, 281, 283, 293, 307, 311, 313, 317,
	331, 337, 347, 349, 353, 359, 367, 373, 379, 383, 389, 397, 401, 409, 419,
	421, 431, 433, 439, 443, 449, 457, 461, 463, 467, 479, 487, 491, 499, 503,
	509, 521, 523, 541, 547, 557, 563, 569, 571, 577, 587, 593, 599, 601, 607,
	613, 617, 619, 631, 641, 643, 647, 653, 659, 661, 673, 677, 683, 691, 701,
	709, 719, 727, 733, 739, 743, 751, 757, 761, 769, 773, 787, 797, 809, 811,
	821, 823, 827, 829, 839, 853, 857, 859, 863, 877, 881, 883, 887, 907, 911,
	919, 929, 937, 941, 947, 953, 967, 971, 977, 983, 991, 997,

	1009, 1013, 1019, 1021, 1031, 1033, 1039, 1049, 1051, 1061, 1063, 1069, 1087,
	1091, 1093, 1097, 1103, 1109, 1117, 1123, 1129, 1151, 1153, 1163, 1171, 1181,
	1187, 1193, 1201, 1213, 1217, 1223, 1229, 1231, 1237, 1249, 1259, 1277, 1279,
	1283, 1289, 1291, 1297, 1301, 1303, 1307, 1319, 1321, 1327, 1361, 1367, 1373,
	1381, 1399, 1409, 1423, 1427, 1429, 1433, 1439, 1447, 1451, 1453, 1459, 1471,
	1481, 1483, 1487, 1489, 1493, 1499, 1511, 1523, 1531, 1543, 1549, 1553, 1559,
	1567, 1571, 1579, 1583, 1597, 1601, 1607, 1609, 1613, 1619, 1621, 1627, 1637,
	1657, 1663, 1667, 1669, 1693, 1697, 1699, 1709, 1721, 1723, 1733, 1741, 1747,
	1753, 1759, 1777, 1783, 1787, 1789, 1801, 1811, 1823, 1831, 1847, 1861, 1867,
	1871, 1873, 1877, 1879, 1889, 1901, 1907, 1913, 1931, 1933, 1949, 1951, 1973,
	1979, 1987, 1993, 1997, 1999, 
		
	2003, 2011, 2017, 2027, 2029, 2039, 2053, 2063, 2069, 2081, 2083, 2087, 2089,
	2099, 2111, 2113, 2129, 2131, 2137, 2141, 2143, 2153, 2161, 2179, 2203, 2207,
	2213, 2221, 2237, 2239, 2243, 2251, 2267, 2269, 2273, 2281, 2287, 2293, 2297,
	2309, 2311, 2333, 2339, 2341, 2347, 2351, 2357, 2371, 2377, 2381, 2383, 2389,
	2393, 2399, 2411, 2417, 2423, 2437, 2441, 2447, 2459, 2467, 2473, 2477, 2503,
	2521, 2531, 2539, 2543, 2549, 2551, 2557, 2579, 2591, 2593, 2609, 2617, 2621,
	2633, 2647, 2657, 2659, 2663, 2671, 2677, 2683, 2687, 2689, 2693, 2699, 2707,
	2711, 2713, 2719, 2729, 2731, 2741, 2749, 2753, 2767, 2777, 2789, 2791, 2797,
	2801, 2803, 2819, 2833, 2837, 2843, 2851, 2857, 2861, 2879, 2887, 2897, 2903,
	2909, 2917, 2927, 2939, 2953, 2957, 2963, 2969, 2971, 2999,
			
	3001, 3011, 3019, 3023, 3037, 3041, 3049, 3061, 3067, 3079, 3083, 3089, 3109,
	3119, 3121, 3137, 3163, 3167, 3169, 3181, 3187, 3191, 3203, 3209, 3217, 3221,
	3229, 3251, 3253, 3257, 3259, 3271, 3299, 3301, 3307, 3313, 3319, 3323, 3329,
	3331, 3343, 3347, 3359, 3361, 3371, 3373, 3389, 3391, 3407, 3413, 3433, 3449,
	3457, 3461, 3463, 3467, 3469, 3491, 3499, 3511, 3517, 3527, 3529, 3533, 3539,
	3541, 3547, 3557, 3559, 3571, 3581, 3583, 3593, 3607, 3613, 3617, 3623, 3631,
	3637, 3643, 3659, 3671, 3673, 3677, 3691, 3697, 3701, 3709, 3719, 3727, 3733,
	3739, 3761, 3767, 3769, 3779, 3793, 3797, 3803, 3821, 3823, 3833, 3847, 3851,
	3853, 3863, 3877, 3881, 3889, 3907, 3911, 3917, 3919, 3923, 3929, 3931, 3943,
	3947, 3967, 3989,
			
	4001, 4003, 4007, 4013, 4019, 4021, 4027, 4049, 4051, 4057, 4073, 4079, 4091,
	4093, 4099, 4111, 4127, 4129, 4133, 4139, 4153, 4157, 4159, 4177, 4201, 4211,
	4217, 4219, 4229, 4231, 4241, 4243, 4253, 4259, 4261, 4271, 4273, 4283, 4289,
	4297, 4327, 4337, 4339, 4349, 4357, 4363, 4373, 4391, 4397, 4409, 4421, 4423,
	4441, 4447, 4451, 4457, 4463, 4481, 4483, 4493, 4507, 4513, 4517, 4519, 4523,
	4547, 4549, 4561, 4567, 4583, 4591, 4597, 4603, 4621, 4637, 4639, 4643, 4649,
	4651, 4657, 4663, 4673, 4679, 4691, 4703, 4721, 4723, 4729, 4733, 4751, 4759,
	4783, 4787, 4789, 4793, 4799, 4801, 4813, 4817, 4831, 4861, 4871, 4877, 4889,
	4903, 4909, 4919, 4931, 4933, 4937, 4943, 4951, 4957, 4967, 4969, 4973, 4987,
	4993, 4999,
			
	5003, 5009, 5011, 5021, 5023, 5039, 5051, 5059, 5077, 5081, 5087, 5099, 5101,
	5107, 5113, 5119, 5147, 5153, 5167, 5171, 5179, 5189, 5197, 5209, 5227, 5231, 
	5233, 5237, 5261, 5273, 5279, 5281, 5297, 5303, 5309, 5323, 5333, 5347, 5351,
	5381, 5387, 5393, 5399, 5407, 5413, 5417, 5419, 5431, 5437, 5441, 5443, 5449,
	5471, 5477, 5479, 5483, 5501, 5503, 5507, 5519, 5521, 5527, 5531, 5557, 5563,
	5569, 5573, 5581, 5591, 5623, 5639, 5641, 5647, 5651, 5653, 5657, 5659, 5669,
	5683, 5689, 5693, 5701, 5711, 5717, 5737, 5741, 5743, 5749, 5779, 5783, 5791,
	5801, 5807, 5813, 5821, 5827, 5839, 5843, 5849, 5851, 5857, 5861, 5867, 5869,
	5879, 5881, 5897, 5903, 5923, 5927, 5939, 5953, 5981, 5987
];

BigInteger.lplim = (1<<26)/BigInteger.lowPrimes[BigInteger.lowPrimes.length-1];

// (public) test primality with certainty >= 1-.5^t
BigInteger.prototype.isProbablePrime = function(t) {
	var i, x = this.abs();
	if(x.t == 1 && x.d[0] <= BigInteger.lowPrimes[BigInteger.lowPrimes.length-1]) {
		for(i = 0; i < BigInteger.lowPrimes.length; ++i)
			if(x.d[0] == BigInteger.lowPrimes[i]) return true;
		return false;
	}
	if(x._isEven()) return false;
	i = 1;
	while(i < BigInteger.lowPrimes.length) {
		var m = BigInteger.lowPrimes[i], j = i+1;
		while(j < BigInteger.lowPrimes.length && m < BigInteger.lplim) m *= BigInteger.lowPrimes[j++];
		m = x._modInt(m);
		while(i < j) if(m%BigInteger.lowPrimes[i++] == 0) return false;
	}
	return x._millerRabin(t);
};

// (protected) true if probably prime (HAC 4.24, Miller-Rabin)
BigInteger.prototype._millerRabin = function(t) {
	var n1 = this.subtract(BigInteger.ONE);
	var k = n1.getLowestSetBit();
	if(k <= 0) return false;
	var r = n1.shiftRight(k);
	t = (t+1)>>1;
	if(t > BigInteger.lowPrimes.length) t = BigInteger.lowPrimes.length;
	var a = new BigInteger(null);
	for(var i = 0; i < t; ++i) {
		//Pick bases at random, instead of starting at 2
		a._fromInt(BigInteger.lowPrimes[Math.floor(Math.random()*BigInteger.lowPrimes.length)]);
		var y = a.modPow(r,this);
		if(y.compareTo(BigInteger.ONE) != 0 && y.compareTo(n1) != 0) {
			var j = 1;
			while(j++ < k && y.compareTo(n1) != 0) {
				y = y.modPowInt(2,this);
				if(y.compareTo(BigInteger.ONE) == 0) return false;
			}
			if(y.compareTo(n1) != 0) return false;
		}
	}
	return true;
};


// BigInteger(int signum, byte[] magnitude)
// double doubleValue()
// float floatValue()
// int hashCode()
// long longValue()
// static BigInteger valueOf(long val)
