/**
 * JavaScript RSA Encryption library.
 * Depends on biginteger.class.js and rng.class.js
 *
 * Copyright (c) 2012 Jacob Lee <letsgolee@naver.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA or check at http://www.gnu.org/licenses/gpl.html
 */

/**
 * Copyright (c) 2005-2009  Tom Wu
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL TOM WU BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF
 * THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT
 * OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * In addition, the following condition applies:
 *
 * All redistributions must retain an intact copy of this copyright notice
 * and disclaimer.
 */


/*

RSAPublicKey::=SEQUENCE{
	modulus INTEGER, -- n
	publicExponent INTEGER -- e
}


RSAPrivateKey ::= SEQUENCE {
	version Version,
	modulus INTEGER, -- n
	publicExponent INTEGER, -- e
	privateExponent INTEGER, -- d
	prime1 INTEGER, -- p
	prime2 INTEGER, -- q
	exponent1 INTEGER, -- d mod (p-1)
	exponent2 INTEGER, -- d mod (q-1)
	coefficient INTEGER -- (inverse of q) mod p
}
*/

function RSA()
{
	this.n = null;
	this.e = 0; /* Public exponent */
	this.d = null;
	this.p = null;
	this.q = null;
	this.dmp1 = null; /* D mod (P-1) */
	this.dmq1 = null; /* D mod (Q-1) */
	this.coeff = null; /* 1/Q mod P */
}

// PKCS#1 (type 2, random) pad input string s to n bytes, and return a bigint
RSA.PKCS1pad2 = function(s,n) {
	s = UTF8.fromUTF16(s);

	if(n < s.length + 11) {
		alert("Message too long for RSA");
		return null;
	}
	var ba = new Array();

	var i = s.length - 1;
	while(i >= 0 && n > 0) { // encode using utf-8
		var c = s.charCodeAt(i--);
		if(c < 128) {
			ba[--n] = c;
		}
		else if((c > 127) && (c < 2048)) {
			ba[--n] = (c & 63) | 128;
			ba[--n] = (c >> 6) | 192;
		}
		else {
			ba[--n] = (c & 63) | 128;
			ba[--n] = ((c >> 6) & 63) | 128 /* 192 */;
			ba[--n] = (c >> 12) | 224;
		}
	}
	ba[--n] = 0;

	var rng = new PRNG();
/*
	var x = new Array();
	while(n > 2) { // random non-zero pad
		x[0] = 0;
		while(x[0] == 0) x = rng.nextBytes(x);
		ba[--n] = x[0];
	}
*/
	var x = 0;
	while(n > 2) { // random non-zero pad
		while(0 == (x = rng.getByte()));
		ba[--n] = x;
	}
	ba[--n] = 2;
	ba[--n] = 0;
	return new BigInteger(ba);
}

// Undo PKCS#1 (type 2, random) padding and, if valid, return the plaintext
RSA.PKCS1unpad2 = function(d,n) {
	var b = d.toByteArray();
	var i = 0;
	while(i < b.length && b[i] == 0) ++i;
	if(b.length-i != n-1 || b[i] != 2)
		return null;
	++i;
	while(b[i] != 0)
		if(++i >= b.length) return null;
	var ret = "";
	while(++i < b.length) {
		var c = b[i] & 255;
		if(c < 128) { // utf-8 decode
			ret += String.fromCharCode(c);
		}
		else if((c > 191) && (c < 224)) {
			ret += String.fromCharCode(((c & 31) << 6) | (b[i+1] & 63));
			++i;
		}
		else {
			ret += String.fromCharCode(((c & 15) << 12) | ((b[i+1] & 63) << 6) | (b[i+2] & 63));
			i += 2;
		}
	}

	ret = UTF8.toUTF16(ret);

	return ret;
}


// Set the public key fields N and e from hex strings
RSA.prototype.setPublicKey = function(N,E) {
	if(N != null && E != null && N.length > 0 && E.length > 0) {
		N = N.replace(/^00/, '');
		E = E.replace(/^00/, '');

		this.n = new BigInteger(N,16);
		this.e = parseInt(E,16);
	}
	else
		alert("Invalid RSA public key");
}

// protected
// Perform raw public operation on "x": return x^e (mod n)
RSA.prototype._rawEncrypt = function(x) {
	return x.modPowInt(this.e, this.n);
}

// Return the PKCS#1 RSA encryption of "text" as an even-length hex string
RSA.prototype.encrypt = function(text) {
	var m = RSA.PKCS1pad2(text,(this.n.bitLength()+7)>>3);
	if(m == null) return null;
	var c = this._rawEncrypt(m);
	if(c == null) return null;
	var h = c.toString(16);
	if((h.length & 1) == 0) return h; else return "0" + h;
}


/*
setPrivateKey function is not recommended
for if p and q are known then
ciphermessage ^ D mod n is the only calculation method and it consumes a lot of time!
*/
// Set the private key fields N, e, and d from hex strings
RSA.prototype.setPrivateKey = function(N,E,D) {
	if(N != null && E != null && N.length > 0 && E.length > 0) {
		N = N.replace(/^00/, '');
		E = E.replace(/^00/, '');
		D = D.replace(/^00/, '');

		this.n = new BigInteger(N,16);
		this.e = parseInt(E,16);
		this.d = new BigInteger(D,16);
	}
	else
		alert("Invalid RSA private key");
}


// Set the private key fields D, P, and Q from hex strings
RSA.prototype.setPrivateKey2 = function(D,P,Q) {
	if(D != null && P != null && Q != null) {
		D = D.replace(/^00/, '');
		P = P.replace(/^00/, '');
		Q = Q.replace(/^00/, '');

		this.d = new BigInteger(D,16);
		this.p = new BigInteger(P,16);
		this.q = new BigInteger(Q,16);

		var p1 = this.p.subtract(BigInteger.ONE);
		var q1 = this.q.subtract(BigInteger.ONE);

		this.n = this.p.multiply(this.q);
		this.dmp1 = this.d.mod(p1);
		this.dmq1 = this.d.mod(q1);
		this.coeff = this.q.modInverse(this.p);
	}
	else
		alert("Invalid RSA private key");
}



// Set the private key fields N, e, d and CRT params from hex strings
RSA.prototype.setPrivateKeyEx = function(N,E,D,P,Q,DP,DQ,C) {
	if(D != null && P != null && Q != null) {
		N = N.replace(/^00/, '');
		E = E.replace(/^00/, '');
		D = D.replace(/^00/, '');
		P = P.replace(/^00/, '');
		Q = Q.replace(/^00/, '');
		DP = DP.replace(/^00/, '');
		DQ = DQ.replace(/^00/, '');
		C = C.replace(/^00/, '');

		this.n = new BigInteger(N,16);
		this.e = parseInt(E,16);
		this.d = new BigInteger(D,16);
		this.p = new BigInteger(P,16);
		this.q = new BigInteger(Q,16);
		this.dmp1 = new BigInteger(DP,16);
		this.dmq1 = new BigInteger(DQ,16);
		this.coeff = new BigInteger(C,16);
	}
	else
		alert("Invalid RSA private key");
}


// protected
// Perform raw private operation on "x": return x^d (mod n)
RSA.prototype._rawDecrypt = function(x) {
	if(this.p == null || this.q == null)
		return x.modPow(this.d, this.n);

	// TODO: re-calculate any missing CRT params
	var xp = x.mod(this.p).modPow(this.dmp1, this.p);
	var xq = x.mod(this.q).modPow(this.dmq1, this.q);

	while(xp.compareTo(xq) < 0) xp = xp.add(this.p);
	return xp.subtract(xq).multiply(this.coeff).mod(this.p).multiply(this.q).add(xq);
}

// Return the PKCS#1 RSA decryption of "ctext".
// "ctext" is an even-length hex string and the output is a plain string.
RSA.prototype.decrypt = function(ctext) {
	var c = new BigInteger(ctext, 16);
	var m = this._rawDecrypt(c);
	if(m == null) return null;
	return RSA.PKCS1unpad2(m, (this.n.bitLength()+7)>>3);
}


// Generate a new random private key B bits long, using public expt E
RSA.prototype.generateKey = function(B,E) {
	var rng = new PRNG();
	var qs = B>>1;
	this.e = parseInt(E,16);
	var ee = new BigInteger(E,16);

	for(;;) {
		for(;;) {
			this.p = new BigInteger();
			this.p.randomPrime(B-qs ,rng, 10);
			if(this.p.subtract(BigInteger.ONE).gcd(ee).compareTo(BigInteger.ONE) == 0/* && this.p.isProbablePrime(10)*/) break;
		}
		for(;;) {
			this.q = new BigInteger();
			this.q.randomPrime(qs ,rng, 10);
			if(this.q.subtract(BigInteger.ONE).gcd(ee).compareTo(BigInteger.ONE) == 0/* && this.q.isProbablePrime(10)*/) break;
		}
		if(this.p.compareTo(this.q) <= 0) {
			var t = this.p;
			this.p = this.q;
			this.q = t;
		}
		var p1 = this.p.subtract(BigInteger.ONE);
		var q1 = this.q.subtract(BigInteger.ONE);
		var phi = p1.multiply(q1);

		if(phi.gcd(ee).compareTo(BigInteger.ONE) == 0) {
			this.n = this.p.multiply(this.q);
			this.d = ee.modInverse(phi);
			this.dmp1 = this.d.mod(p1);
			this.dmq1 = this.d.mod(q1);
			this.coeff = this.q.modInverse(this.p);
			break;
		}
	}
};



RSA.ASN1 = function(data) {
	this.error = false;
	this.data = null;

	if (data != null)
		this.data = this.parse(data);
};

RSA.ASN1.prototype.parse = function(data) {
	if (!data) {
		this.error = true;
		return null;
	}
	var result = [];
	while(data.length > 0) {
		// get the tag
		var tag = data.charCodeAt(0);
		data = data.substr(1);
		// get length
		var length = 0;
		// ignore any null tag
		if ((tag & 31) == 0x5) data = data.substr(1);
		else {
			if (data.charCodeAt(0) & 128) {
				var lengthSize = data.charCodeAt(0) & 127;
				data = data.substr(1);
				if(lengthSize > 0) length = data.charCodeAt(0);
				if(lengthSize > 1)	length = ((length << 8) | data.charCodeAt(1));
				if(lengthSize > 2) {
					this.error = true;
					return null;
				}
				data = data.substr(lengthSize);
			} else {
				length = data.charCodeAt(0);
				data = data.substr(1);
			}
		}
		// get value
		var value = "";
		if(length) {
			if (length > data.length){
				this.error = true;
				return null;
			}
			value = data.substr(0, length);
			data = data.substr(length);
		}
		if (tag & 32)
			result.push(this.parse(value)); // sequence
		else
			result.push(this._getValue((tag & 128) ? 4 : (tag & 31), value));
	}
	return result;
};

RSA.ASN1.prototype._getValue = function(tag, data) {
	if (tag == 1)
		return data ? true : false;
	else if (tag == 2) //integer
		return data;
	else if (tag == 3) //bit string
		return this.parse(data.substr(1));
	else if (tag == 5) //null
		return null;
	else if (tag == 6){ //ID
		var res = [];
		var d0 = data.charCodeAt(0);
		res.push(Math.floor(d0 / 40));
		res.push(d0 - res[0]*40);
		var stack = [];
		var powNum = 0;
		var i;
		for(i=1;i<data.length;i++){
			var token = data.charCodeAt(i);
			stack.push(token & 127);
			if ( token & 128 )
				powNum++;
			else {
				var j;
				var sum = 0;
				for(j=0;j<stack.length;j++)
					sum += stack[j] * Math.pow(128, powNum--);
				res.push(sum);
				powNum = 0;
				stack = [];
			}
		}
		return res.join(".");
	}
	return null;
};




RSA.ASN1.int2hex = function(integer) {
	integer = integer.toString(16);
	if (integer.length % 2 !== 0) {
		integer = '0' + integer;
	}
	return integer;
};

RSA.ASN1.lengthValue = function(content, extra) {
	if (typeof extra == "undefined" || extra == null) {
		extra = 0;
	}
	var len = (content.length / 2) + extra;
	if (len > 127) {
		len = RSA.ASN1.int2hex(len);
		return RSA.ASN1.int2hex(0x80 + (len.length / 2)) + len;
	} else {
		return RSA.ASN1.int2hex(len);
	}
};

RSA.ASN1.intValue = function(integer, nullPrefixed) {
	integer = RSA.ASN1.int2hex(integer);
	if (nullPrefixed) {
		integer = '00' + integer;
	}
	return '02' + RSA.ASN1.lengthValue(integer) + integer;
};



var Hex = {
	hex: "0123456789abcdef",
	encode: function(input) {
		if(!input) return false;
		var output = "";
		var k;
		var i = 0;
		do {
			k = input.charCodeAt(i++);
			output += this.hex.charAt((k >> 4) & 0xf) + this.hex.charAt(k & 0xf);
		} while (i < input.length);
		return output;
	},
	decode: function(input) {
		if(!input) return false;
		input = input.replace(/[^0-9abcdef]/g, "");
		var output = "";
		var i = 0;
		do {
			output += String.fromCharCode(((this.hex.indexOf(input.charAt(i++)) << 4) & 0xf0) | (this.hex.indexOf(input.charAt(i++)) & 0xf));
		} while (i < input.length);
		return output;
	}
};



RSA.prototype.getPrivatePEM = function() {
	var pem = '';
	pem = '020100';
	pem += RSA.ASN1.intValue(this.n, true);
	pem += RSA.ASN1.intValue(this.e, false);
	pem += RSA.ASN1.intValue(this.d, false);
	pem += RSA.ASN1.intValue(this.p, true);
	pem += RSA.ASN1.intValue(this.q, true);
	pem += RSA.ASN1.intValue(this.dmp1, true);
	pem += RSA.ASN1.intValue(this.dmq1, false);
	pem += RSA.ASN1.intValue(this.coeff, false);
	pem = '30' + RSA.ASN1.lengthValue(pem) + pem;
	return "-----BEGIN RSA PRIVATE KEY-----\n" + RSA.lineBreak(BASE64.encode(Hex.decode(pem)),64) + "\n-----END RSA PRIVATE KEY-----";
};

RSA.prototype.getPublicPEM = function() {
	var pem = '';
	pem = RSA.ASN1.intValue(this.n, true);
	pem += RSA.ASN1.intValue(this.e, false);
	pem = '30' + RSA.ASN1.lengthValue(pem) + pem;
	pem = '03' + RSA.ASN1.lengthValue(pem, 1) + '00' + pem;
	pem = '300d06092a864886f70d0101010500' + pem; // 1.2.840.113549.1.1.1
	pem = '30' + RSA.ASN1.lengthValue(pem) + pem;
	return "-----BEGIN PUBLIC KEY-----\n" + RSA.lineBreak(BASE64.encode(Hex.decode(pem)),64) + "\n-----END PUBLIC KEY-----";
};

RSA.prototype.parsePublicPEM = function(pem) {
	pem = pem.replace(/-----(BEGIN|END) (RSA )?(PUBLIC|PRIVATE) KEY-----/g, '').replace("\n", '');
	var data = new RSA.ASN1().parse(BASE64.decode(pem));

	if(data[0][0][0]=="1.2.840.113549.1.1.1")
		return this.setPublicKey(Hex.encode(data[0][1][0][0]), Hex.encode(data[0][1][0][1]));
	return false;
};

RSA.prototype.parsePrivatePEM = function(pem) {
	pem = pem.replace(/-----(BEGIN|END) (RSA )?(PUBLIC|PRIVATE) KEY-----/g, '').replace("\n", '');
	var data = new RSA.ASN1().parse(BASE64.decode(pem));

	return this.setPrivateKeyEx(Hex.encode(data[0][1]), Hex.encode(data[0][2]), Hex.encode(data[0][3]), Hex.encode(data[0][4]), Hex.encode(data[0][5]), Hex.encode(data[0][6]), Hex.encode(data[0][7]), Hex.encode(data[0][8]));
};


// Generate a new random PEM public/private keypair, which keys are B bits long, using public expt E
RSA.prototype.generatePEMKey = function(B,E)
{
	this.generateKey(B,E);

	var pemKeys = {};
	pemKeys['public'] = this.getPublicPEM();
	pemKeys['private'] = this.getPrivatePEM();

	return pemKeys;
};


RSA.lineBreak = function(s,n) {
	var ret = "";
	var i = 0;
	while(i + n < s.length) {
		ret += s.substring(i,i+n) + "\n";
		i += n;
	}
	return ret + s.substring(i,s.length);
};

/*
function byte2Hex(b) {
	if(b < 0x10)
		return "0" + b.toString(16);
	else
		return b.toString(16);
}
*/

if (typeof UTF8 == 'undefined') {

var UTF8 = {
	encode: function(input)
	{
		/* Why do we need converting functions between UTF16 and UTF8? 
		   Because the output of PHP utf8_encode() and Javascript UTF8 function output are different from each other.
		   
		   JavaScript strings are UCS-2 encoded but can represent Unicode code points
		   outside the Basic Multilingual Pane (U+0000 - U+D7FF and U+E000 - U+FFFF)
		   using two 16 bit numbers (a UTF-16 surrogate pair), 
		   the first of which must be in the range U+D800 - U+DFFF. */
		input = this.fromUTF16(input);

		input = input.replace(/\r\n/g,"\n");
		var output = "";
		for (var n = 0; n < input.length; n++) {
			var c = input.charCodeAt(n);

			if (c < 128) {
				output += String.fromCharCode(c);
			} else if ((c > 127) && (c < 2048)) {
/*
				output += String.fromCharCode((c >> 6) | 192);
				output += String.fromCharCode((c & 63) | 128);
*/
				output += String.fromCharCode((c >> 6) | 192, (c & 63) | 128);
			} else {
/*
				output += String.fromCharCode((c >> 12) | 224);
				output += String.fromCharCode(((c >> 6) & 63) | 128);
				                                                      //  see php_utf8_encode() in ext/xml/xml.c of php source.
				output += String.fromCharCode((c & 63) | 128);
*/
				output += String.fromCharCode((c >> 12) | 224, ((c >> 6) & 63) | 128 /* 192 */, (c & 63) | 128);
			}
		}
		return output;
	},

	decode: function(input)
	{
		var output = "";
		var i = 0;
		var c1 = c2 = c3 = c4 = 0;
		while ( i < input.length ) {
			c1 = input.charCodeAt(i);

			if (c1 < 128) {
				output += String.fromCharCode(c1);
				i++;
			} else if((c1 > 191) && (c1 < 224)) {
				c2 = input.charCodeAt(i+1);
				output += String.fromCharCode(((c1 & 31) << 6) | (c2 & 63));
				i += 2;
			} else {
				c2 = input.charCodeAt(i+1);
				c3 = input.charCodeAt(i+2);
				output += String.fromCharCode(((c1 & 15) << 12) | ((c2 & 63) << 6) | (c3 & 63));
				i += 3;
			}
		}

		output = this.toUTF16(output);

		return output;
	},


	fromUTF16: function(input)
	{
		/* converts UTF16 chracters to UTF8 */
		var output, i, len, c;

		output = "";
		len = input.length;
		for(i = 0; i < len; i++) {
			c = input.charCodeAt(i);
			if ((c >= 0x0001) && (c <= 0x007F)) {
				output += input.charAt(i);
			} else if (c > 0x07FF) {
				output += String.fromCharCode(0xE0 | ((c >> 12) & 0x0F));
				output += String.fromCharCode(0x80 | ((c >>  6) & 0x3F));
				output += String.fromCharCode(0x80 | ((c >>  0) & 0x3F));
			} else {
				output += String.fromCharCode(0xC0 | ((c >>  6) & 0x1F));
				output += String.fromCharCode(0x80 | ((c >>  0) & 0x3F));
			}
		}
		return output;
	},

	toUTF16: function(input)
	{
		/* converts UTF8 characters to UTF16 */
		var output, i, len, c;
		var char2, char3;

		output = "";
		len = input.length;
		i = 0;
		while(i < len) {
			c = input.charCodeAt(i++);
			switch(c >> 4) { 
				case 0:
				case 1:
				case 2:
				case 3:
				case 4:
				case 5:
				case 6:
				case 7:
					// 0xxxxxxx
					output += input.charAt(i-1);
					break;
				case 12:
				case 13:
					// 110x xxxx   10xx xxxx
					char2 = input.charCodeAt(i++);
					output += String.fromCharCode(((c & 0x1F) << 6) | (char2 & 0x3F));
					break;
				case 14:
					// 1110 xxxx  10xx xxxx  10xx xxxx
					char2 = input.charCodeAt(i++);
					char3 = input.charCodeAt(i++);
					output += String.fromCharCode(((c & 0x0F) << 12) |
								   ((char2 & 0x3F) << 6) |
								   ((char3 & 0x3F) << 0));
					break;
			}
		}

		return output;
	}
};

} // end of if


if (typeof BASE64 == 'undefined') {

var BASE64 = {
	base64: "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=",

	encode: function(input)
	{
		if (!input) {
			return false;
		}
		//input = UTF8.encode(input);
		var output = "";
		var chr1, chr2, chr3;
		var enc1, enc2, enc3, enc4;
		var i = 0;
		do {
			chr1 = input.charCodeAt(i++);
			chr2 = input.charCodeAt(i++);
			chr3 = input.charCodeAt(i++);
			enc1 = chr1 >> 2;
			enc2 = ((chr1 & 3) << 4) | (chr2 >> 4);
			enc3 = ((chr2 & 15) << 2) | (chr3 >> 6);
			enc4 = chr3 & 63;
			if (isNaN(chr2)) enc3 = enc4 = 64;
			else if (isNaN(chr3)) enc4 = 64;
			output += this.base64.charAt(enc1) + this.base64.charAt(enc2) + this.base64.charAt(enc3) + this.base64.charAt(enc4);
		} while (i < input.length);
		return output;
	},

	decode: function(input)
	{
		if(!input) return false;
		input = input.replace(/[^A-Za-z0-9\+\/\=]/g, "");
		var output = "";
		var enc1, enc2, enc3, enc4;
		var i = 0;
		do {
			enc1 = this.base64.indexOf(input.charAt(i++));
			enc2 = this.base64.indexOf(input.charAt(i++));
			enc3 = this.base64.indexOf(input.charAt(i++));
			enc4 = this.base64.indexOf(input.charAt(i++));
			output += String.fromCharCode((enc1 << 2) | (enc2 >> 4));
			if (enc3 != 64) output += String.fromCharCode(((enc2 & 15) << 4) | (enc3 >> 2));
			if (enc4 != 64) output += String.fromCharCode(((enc3 & 3) << 6) | enc4);
		} while (i < input.length);
		return output; //UTF8.decode(output);
	}
};

} // end of if