<?php
// insert_praise.php

// 데이터베이스 연결 파일 포함
include('connect_pdo.php');

// 파일 업로드 디렉토리 설정
$musicDir = 'uploads/music/';
$imageDir = 'uploads/image/';
$videoDir = 'uploads/video/';

// 업로드 디렉토리가 존재하지 않으면 생성
foreach ([$musicDir, $imageDir, $videoDir] as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
}

// 폼이 제출되었는지 확인
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // 입력 데이터 수집 및 유효성 검사
    $chapter = trim($_POST['chapter']);
    $title = trim($_POST['title']);
    $first = trim($_POST['first']);
    $lyrics = trim($_POST['lyrics']);

    $errors = [];

    // 필수 입력 필드 확인
    if (empty($chapter)) $errors[] = "장 정보를 입력하세요.";
    if (empty($title)) $errors[] = "제목을 입력하세요.";
    if (empty($first)) $errors[] = "가사의 첫 줄을 입력하세요.";
    if (empty($lyrics)) $errors[] = "가사를 입력하세요.";

    // 파일 업로드 처리 함수
    function handleFileUpload($file, $dir, $allowedTypes, $maxSize = 10485760) { // maxSize 기본 10MB
        if ($file['error'] === UPLOAD_ERR_NO_FILE) {
            return ['', '']; // 파일 업로드 안 함
        }

        if ($file['error'] !== UPLOAD_ERR_OK) {
            return ['', "파일 업로드 중 오류가 발생했습니다."];
        }

        // 파일 크기 검사
        if ($file['size'] > $maxSize) {
            return ['', "파일 크기가 너무 큽니다. 최대 " . ($maxSize / 1048576) . "MB까지 업로드 가능합니다."];
        }

        // MIME 타입 검사
        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mime = $finfo->file($file['tmp_name']);
        if (!in_array($mime, $allowedTypes)) {
            return ['', "허용되지 않은 파일 형식입니다."];
        }

        // 고유한 파일 이름 생성
        $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
        $basename = bin2hex(random_bytes(8)) . '.' . $ext;
        $targetPath = $dir . $basename;

        // 파일 이동
        if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
            return ['', "파일을 서버에 저장하는 데 실패했습니다."];
        }

        return [$targetPath, ''];
    }

    // 음악 파일 업로드
    list($mp3Path, $mp3Error) = handleFileUpload(
        $_FILES['mp3'],
        $musicDir,
        ['audio/mpeg', 'audio/mp3', 'audio/wav']
    );
    if ($mp3Error) $errors[] = "음악 파일: " . $mp3Error;

    // 사진 파일 업로드
    list($photoPath, $photoError) = handleFileUpload(
        $_FILES['photo'],
        $imageDir,
        ['image/jpeg', 'image/png', 'image/gif']
    );
    if ($photoError) $errors[] = "사진 파일: " . $photoError;

    // 비디오 파일 업로드
    list($videoPath, $videoError) = handleFileUpload(
        $_FILES['video'],
        $videoDir,
        ['video/mp4', 'video/webm', 'video/ogg']
    );
    if ($videoError) $errors[] = "비디오 파일: " . $videoError;

    // 오류가 없으면 데이터베이스에 삽입
    if (empty($errors)) {
        try {
            $sql = "INSERT INTO aa_praise (chapter, title, mp3, photo, video, first, lyrics) 
                    VALUES (:chapter, :title, :mp3, :photo, :video, :first, :lyrics)";
            $stmt = $conn->prepare($sql);
            $stmt->execute([
                ':chapter' => $chapter,
                ':title' => $title,
                ':mp3' => $mp3Path,
                ':photo' => $photoPath,
                ':video' => $videoPath,
                ':first' => $first,
                ':lyrics' => $lyrics
            ]);
            $successMessage = "데이터가 성공적으로 삽입되었습니다.";
        } catch (PDOException $e) {
            $errors[] = "데이터 삽입 실패: " . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="ko">
<head>
    <meta charset="UTF-8">
    <title>찬양 데이터 입력 - PC 전용</title>
    <style>
        /* PC 전용 스타일 */
        body {
            font-family: Arial, sans-serif;
            background-color: #f8f9fa;
            padding: 20px;
        }
        h1 {
            text-align: center;
            color: #343a40;
            margin-bottom: 30px;
        }
        form {
            max-width: 800px;
            margin: 0 auto;
            background-color: #ffffff;
            padding: 25px;
            border-radius: 8px;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
        }
        table {
            width: 100%;
            border-collapse: collapse;
        }
        td {
            padding: 10px 15px;
            vertical-align: top;
        }
        td.label {
            width: 20%;
            font-weight: bold;
            background-color: #f1f1f1;
        }
        input[type="text"],
        textarea,
        input[type="file"] {
            width: 100%;
            padding: 8px;
            border: 1px solid #ced4da;
            border-radius: 4px;
            box-sizing: border-box;
        }
        textarea {
            resize: vertical;
            height: 100px;
        }
        .submit-btn {
            text-align: center;
            margin-top: 20px;
        }
        input[type="submit"] {
            background-color: #007bff;
            color: #ffffff;
            padding: 10px 25px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 16px;
        }
        input[type="submit"]:hover {
            background-color: #0056b3;
        }
        .error-message {
            color: red;
            margin-bottom: 15px;
        }
        .success-message {
            color: green;
            margin-bottom: 15px;
        }
    </style>
</head>
<body>
    <h1>찬양 데이터 입력</h1>
    <form action="insert_praise.php" method="POST" enctype="multipart/form-data">
        <?php if (!empty($errors)): ?>
            <div class="error-message">
                <ul>
                    <?php foreach ($errors as $error): ?>
                        <li><?= htmlspecialchars($error) ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <?php if (!empty($successMessage)): ?>
            <div class="success-message">
                <?= htmlspecialchars($successMessage) ?>
            </div>
        <?php endif; ?>

        <table>
            <tr>
                <td class="label"><label for="chapter">장</label></td>
                <td>
                    <input type="text" id="chapter" name="chapter" required 
                           value="<?= isset($_POST['chapter']) ? htmlspecialchars($_POST['chapter']) : '' ?>">
                </td>
            </tr>
            <tr>
                <td class="label"><label for="title">제목</label></td>
                <td>
                    <input type="text" id="title" name="title" required 
                           value="<?= isset($_POST['title']) ? htmlspecialchars($_POST['title']) : '' ?>">
                </td>
            </tr>
            <tr>
                <td class="label"><label for="mp3">음악 파일 (MP3)</label></td>
                <td>
                    <input type="file" id="mp3" name="mp3" accept=".mp3,audio/*">
                </td>
            </tr>
            <tr>
                <td class="label"><label for="photo">사진 파일 (이미지)</label></td>
                <td>
                    <input type="file" id="photo" name="photo" accept=".jpg,.jpeg,.png,.gif,image/*">
                </td>
            </tr>
            <tr>
                <td class="label"><label for="video">비디오 파일 (MP4)</label></td>
                <td>
                    <input type="file" id="video" name="video" accept=".mp4,video/*">
                </td>
            </tr>
            <tr>
                <td class="label"><label for="first">가사의 첫 줄</label></td>
                <td>
                    <input type="text" id="first" name="first" required 
                           value="<?= isset($_POST['first']) ? htmlspecialchars($_POST['first']) : '' ?>">
                </td>
            </tr>
            <tr>
                <td class="label"><label for="lyrics">가사</label></td>
                <td>
                    <textarea id="lyrics" name="lyrics" required><?= isset($_POST['lyrics']) ? htmlspecialchars($_POST['lyrics']) : '' ?></textarea>
                </td>
            </tr>
        </table>
        <div class="submit-btn">
            <input type="submit" value="데이터 삽입">
        </div>
    </form>
</body>
</html>


